//
//  VideoFrameProcessorInterface.h
//  ml_transformers_iphoneos
//
//  Created by Guy Mininberg on 08/11/2022.
//

//
//  VideoFrameProcessor.h
//  ml_transformers_iphoneos
//
//  Created by Guy Mininberg on 08/11/2022.
//

#pragma once

#include <memory>
#include <iostream>
#include <sstream>

#include "ml_transformers/ml_transformers.h"

#if defined(_WIN32)
#include <Windows.h>
#endif

#if defined(WEBRTC_MAC) || defined(WEBRTC_IOS)
#ifdef __OBJC__
#define RTC_FWD_DECL_OBJC_CLASS(classname) @class classname
#else
#define RTC_FWD_DECL_OBJC_CLASS(classname) typedef struct objc_object classname
#endif
#endif

#define FORCE_LOGS 1

#if defined(DEBUG) || defined(FORCE_LOGS)
#ifdef ML_ANDROID
#include <android/log.h>
#endif
static void PrintLog(const std::string& str){
#ifdef ML_ANDROID
    __android_log_print(ANDROID_LOG_DEBUG, "VonageMLTransformers", "%s", str.c_str());
#elif defined(_WIN32)
    std::wstring widestr = std::wstring(str.begin(), str.end());
    OutputDebugString(widestr.c_str());
    fprintf(stdout, "%s", str.c_str());
#else
    fprintf(stdout, "%s", str.c_str());
#endif
}
#define V_LOG(message)                                                      \
    do {                                                                    \
        std::stringstream s;                                                \
        s << __FILE__ << ":" << __LINE__ << " - " << message << std::endl;  \
        PrintLog(s.str());                                                  \
    } while(0)
#else
#define V_LOG(message)
#endif

namespace cv {
class Mat;
}

namespace vonage {

class TransformerInfoInterface;

class VideoFrameProcessorInterface{
public:
    static std::unique_ptr<VideoFrameProcessorInterface> CreateVideoFrameProcessor();
    static std::unique_ptr<VideoFrameProcessorInterface> CreateDefaultVideoFrameProcessor();
    static std::unique_ptr<uint_fast8_t[]> make_unique_uint_fast8_t_ptr(size_t size);
    static std::unique_ptr<float[]> make_unique_float_ptr(size_t size);

    virtual ~VideoFrameProcessorInterface() = default;
    virtual uint8_t Init(const BaseBackgroundEffectConfig* config) = 0;
    virtual uint8_t ProcessFrame(int width, int height,
                                 uint8_t* data_y, int stride_y,
                                 uint8_t* data_u, int stride_u,
                                 uint8_t* data_v, int stride_v,
                                 int rotation,
                                 uint64_t timestamp_ms) = 0;
#if defined(WEBRTC_MAC) || defined(WEBRTC_IOS)
    virtual uint8_t AppleProcessFrame(CVPixelBufferRef buffer,
                                      int rotation,
                                      uint64_t timestamp_us) = 0;
#endif
    virtual uint8_t SetTransformerConfig(const BaseBackgroundEffectConfig* config) = 0;

    VideoMLTransformerInteface::StatusReturnValue GetLastWarning() const;
    VideoMLTransformerInteface::StatusReturnValue GetLastError() const;

protected:
    VideoMLTransformerInteface::StatusValue last_warning_ = {static_cast<uint8_t>(VideoMLTransformerInteface::WarningCode::kOkWraning), {""}};
    VideoMLTransformerInteface::StatusValue last_error_ = {static_cast<uint8_t>(VideoMLTransformerInteface::ErrorCode::kOkError), {""}};

    uint8_t UpdateMediaPipePostProcessOptions(const BaseBackgroundEffectConfig* config);
};

}
