#ifndef ML_TRANSFORMERS_ML_TRANSFORMERS_H_
#define ML_TRANSFORMERS_ML_TRANSFORMERS_H_

#include <string>
#include <memory>
#include <vector>

#if defined(WEBRTC_MAC) || defined(WEBRTC_IOS)
#import <CoreVideo/CoreVideo.h>
#endif

#if defined(_WIN32)
#ifdef VONAGE_LIBRARY_IMPL
#define VONAGE_EXPORT __declspec(dllexport)
#else
#define VONAGE_EXPORT __declspec(dllimport)
#endif
#else
#define VONAGE_EXPORT
#endif

namespace vonage {

class VideoFrameProcessorInterface;

/** Background effect type.

    Enumerates all available background effects.
 */
enum class BackgroundEffectType {
    kBlur, /**<  Blur background. */
    kReplace, /**< Virtual background. */
};

/** Blur level enumeration listing all possible levels.

    Enumerates all blur levels possible for blur-related transformer instances.
 */
enum class BlurRadius {
    kNone, /**<   No blur level specified. */
    kHigh, /**<   High blur level. */
    kLow,/**<   Low blur level. */
    kCustom, /**< Custom blur level depends on user settings*/
};

/** Base struct for ML background effect transformer configuration.

 */
class VONAGE_EXPORT BaseBackgroundEffectConfig {
public:
    BaseBackgroundEffectConfig(BackgroundEffectType type, const std::vector<std::string>& tflite_model_paths);
    virtual ~BaseBackgroundEffectConfig() = default;
    BackgroundEffectType GetType() const;
    const std::vector<std::string>& GetTFLiteModelPaths() const;

private:
    BackgroundEffectType type_; /**<  Background effect type enumeration value. */
    std::vector<std::string> tflite_model_paths_; /**< List (vector) of TFLite model file paths. */
};

/** ML background blur transformer configuration struct.
When using kCustom radius custom_radius must be provided.
When running on GPU Bilateral Filter is used (See https://en.wikipedia.org/wiki/Bilateral_filter).
When running on CPU Box Blur is used (See https://en.wikipedia.org/wiki/Box_blur).
Please consider this when using (custom) user-provided blur level.
 */
class VONAGE_EXPORT BlurBackgroundEffectConfig : public BaseBackgroundEffectConfig {
public:
    BlurBackgroundEffectConfig(const std::vector<std::string>& tflite_model_paths, BlurRadius radius, uint8_t custom_radius);
    virtual ~BlurBackgroundEffectConfig() = default;
    BlurRadius GetRadius() const;
    uint8_t GetCustomRadius() const;
private:
    BlurRadius radius_; /**<  Blur level enumeration value. */
    uint8_t    custom_radius_; /**< Blur level to be used as radius when (custom) user-provided value is selected. */
};

/** ML background replacement (virtual background) transformer configuration struct.

 */
class VONAGE_EXPORT ReplaceBackgroundEffectConfig : public BaseBackgroundEffectConfig {
public:
    ReplaceBackgroundEffectConfig(const std::vector<std::string>& tflite_model_paths, const std::string& image_file_path);
    ReplaceBackgroundEffectConfig(const std::vector<std::string>& tflite_model_paths, const std::wstring& image_file_path);

    virtual ~ReplaceBackgroundEffectConfig() = default;
    const std::string& GetFilePath() const;
private:
    std::string image_file_path_; /**< Path to the image resource. Supported formats are: PNG, TIFF and JPEG. */
};

class VONAGE_EXPORT VideoMLTransformerInteface {
public:
    enum class ReturnCode {
        kSuccess = 0,
        kWarning = 1,
        kError = 2
    };

    enum class WarningCode {
        kOkWraning = 0,
        kGenericWarning = 1,
        kFpsWarning = 3,
        kPerformanceWarning = 4,
        kProcessWarning = 5,
    };

    enum class ErrorCode {
        kOkError = 0,
        kGenericError = 1,
        kFileNotFoundError = 2,
        kInitializedError = 3,
        kConfigurationError = 4,
        kNotSupportedError = 5,
        kImageLoadingError = 6,
        kProcessError = 7,
    };

    virtual ~VideoMLTransformerInteface() = default;

    virtual uint8_t Transform(int width, int height,
                              uint8_t* data_y, int stride_y,
                              uint8_t* data_u, int stride_u,
                              uint8_t* data_v, int stride_v,
                              int rotation,
                              uint64_t timestamp_ms) = 0;
#if defined(WEBRTC_MAC) || defined(WEBRTC_IOS)
    virtual uint8_t AppleTransform(CVPixelBufferRef buffer,
                                   int rotation,
                                   uint64_t timestamp_us) = 0;
#endif
    using StatusValue = std::pair<uint8_t, std::string>;
    using StatusReturnValue = const StatusValue&;
    virtual StatusReturnValue GetLastWarning() const = 0;
    virtual StatusReturnValue GetLastError() const = 0;
};

class VONAGE_EXPORT BackgroundEffectVideoMLTransformer final : public VideoMLTransformerInteface {
public:
    static uint8_t ForceCpuProcess(bool force);
    static uint8_t DisableGpuDelegate(bool disable);
    static uint8_t SetNumThreads(uint32_t num_threads);

    BackgroundEffectVideoMLTransformer(const BaseBackgroundEffectConfig* config);
    virtual ~BackgroundEffectVideoMLTransformer();

    uint8_t Init();
    uint8_t UpdateConfig(const BaseBackgroundEffectConfig* config);
    uint8_t Transform(int width, int height,
                      uint8_t* data_y, int stride_y,
                      uint8_t* data_u, int stride_u,
                      uint8_t* data_v, int stride_v,
                      int rotation,
                      uint64_t timestamp_ms) override;
#if defined(WEBRTC_MAC) || defined(WEBRTC_IOS)
    virtual uint8_t AppleTransform(CVPixelBufferRef buffer,
                                   int rotation,
                                   uint64_t timestamp_us) override;
#endif
    StatusReturnValue GetLastWarning() const override;
    StatusReturnValue GetLastError() const override;

    float GetAverageLatencyNs();

private:
    static bool force_cpu_process_;
    static bool disable_gpu_delegate_;
    static uint32_t num_threads_;

    StatusValue last_warning_ = {static_cast<uint8_t>(WarningCode::kOkWraning), {""}};
    StatusValue last_error_ {static_cast<uint8_t>(ErrorCode::kOkError), {""}};
    const BaseBackgroundEffectConfig* config_ = nullptr;
    std::unique_ptr<VideoFrameProcessorInterface> frame_processor_;
    std::vector<float> frame_transform_latency_ns_;
};

} // namespace vonage

#endif // ML_TRANSFORMERS_ML_TRANSFORMERS_H_
