/** @file vonage_sframe_decryptor.h
    @brief Vonage SFrame Decryptor for E2EE.

    This file includes the Vonage implementation for decryptors according to SFrame dratf RFC at https://www.ietf.org/archive/id/draft-ietf-sframe-enc-01.html.
    It implements the webrtc::FrameDecryptorInterface WebRTC interface according to the RFC linked above.
*/
#ifndef MODULES_VONAGE_API_CRYPTO_SFRAME_VONAGE_SFRAME_DECRYPTOR_H
#define MODULES_VONAGE_API_CRYPTO_SFRAME_VONAGE_SFRAME_DECRYPTOR_H

#include "api/crypto/frame_decryptor_interface.h"

#include "modules/vonage/api/crypto/sframe/vonage_sframe_engine.h"
#include "modules/vonage/api/crypto/sframe/vonage_sframe_observer.h"

namespace vonage {

class SFrameEngineInterface;

class SFrameFrameDecryptor : public webrtc::FrameDecryptorInterface {
public:
    /** SFrame decryptor create function.

         Helper factory function to easily create decryptor instances.

         @param kid Key ID (KID) value associated to the encryptor instance using a given key.
         @param base_key_data Data chunck containing SFrame base key to the given KID.
         @param base_key_length Length for the data of the chunck containing SFrame base key.
         @param suite SFrame ciphersuite used by the encryptor instance.
         @param observer Observer instance the encryptor will use for reporting events and messages.
         @return The decryptor instance.
     */
    static rtc::scoped_refptr<SFrameFrameDecryptor> Create(SFrameEngineCipherSuite suite, SFrameFrameEncryptionObserver* observer);

    virtual ~SFrameFrameDecryptor();

    /** Update base key method.

         Updates the base key used by the decryptor instance associated to given KID.

         @param kid Key ID (KID) value associated to a given key.
         @param base_key_data Data chunck containing SFrame base key to the given KID.
         @param base_key_length Length for the data of the chunck containing SFrame base key.
         @return Flag indicating whether the base key was updated correctly or not.
     */
    bool AddOrUpdateBaseKey(uint64_t kid, const uint8_t* base_key_data, size_t base_key_length);

    /** Remove KID method.

         Removes the given KID and the base key associated to it.

         @param kid Key ID (KID) value associated to a given key.
         @return Flag indicating whether the KID was removed correctly or not.
     */
    bool RemoveBaseKey(uint64_t kid);

    /** Enable or disable decryptor function method.

         Enables or disables decryption. This is a helper method intended to be used only for demo purposes.

         @param enabled Flag indicating whether to enabled or disabled the decryption function.
         @return Flag indicating whether the decryption function was enabled or disabled correctly.
     */
    bool SetEnabled(bool enabled);

    // FrameDecryptorInterface implementation.
    webrtc::FrameDecryptorInterface::Result Decrypt(cricket::MediaType media_type,
                                                    webrtc::FrameDecryptorInterface::CodecType codec_type,
                                                    const std::vector<uint32_t>& csrcs,
                                                    rtc::ArrayView<const uint8_t> additional_data,
                                                    rtc::ArrayView<const uint8_t> encrypted_frame,
                                                    rtc::ArrayView<uint8_t> frame) override;
    size_t GetMaxPlaintextByteSize(cricket::MediaType media_type,
                                   size_t encrypted_frame_size) override;
    bool isOk();

protected:
    SFrameFrameDecryptor(SFrameEngineCipherSuite suite, SFrameFrameEncryptionObserver* observer);

    SFrameFrameEncryptionObserver::CodecType FrameDecryptorToObserverCodecType(webrtc::FrameDecryptorInterface::CodecType codec_type) const;

private:
    bool enabled_ = true;
    uint64_t kid_;
    std::unique_ptr<SFrameEngineInterface> engine_;
    SFrameFrameEncryptionObserver* observer_ = nullptr;
};

} // namespace vonage

#endif // MODULES_VONAGE_API_CRYPTO_SFRAME_VONAGE_SFRAME_DECRYPTOR_H
