/** @file vonage_sframe_encryptor.h
    @brief Vonage SFrame Encryptor for E2EE.

    This file includes the Vonage implementation for encryptors according to SFrame dratf RFC at https://www.ietf.org/archive/id/draft-ietf-sframe-enc-01.html.
    It implements the webrtc::FrameEncryptorInterface WebRTC interface according to the RFC linked above.
*/
#ifndef MODULES_VONAGE_API_CRYPTO_SFRAME_VONAGE_SFRAME_ENCRYPTOR_H
#define MODULES_VONAGE_API_CRYPTO_SFRAME_VONAGE_SFRAME_ENCRYPTOR_H

#include "api/crypto/frame_encryptor_interface.h"

#include "modules/vonage/api/crypto/sframe/vonage_sframe_engine.h"
#include "modules/vonage/api/crypto/sframe/vonage_sframe_observer.h"

namespace vonage {

class SFrameEngineInterface;

class SFrameFrameEncryptor : public webrtc::FrameEncryptorInterface {
public:
    /** SFrame encryptor create function.

         Helper factory function to easily create encryptor instances.

         @param kid Key ID (KID) value associated to the encryptor instance using a given key.
         @param base_key_data Data chunck containing SFrame base key to the given KID.
         @param base_key_length Length for the data of the chunck containing SFrame base key.
         @param suite SFrame ciphersuite used by the encryptor instance.
         @param observer Observer instance the encryptor will use for reporting events and messages.
         @return The encryptor instance.
     */
    static rtc::scoped_refptr<SFrameFrameEncryptor> Create(uint64_t kid, const uint8_t* base_key_data, size_t base_key_length, SFrameEngineCipherSuite suite, SFrameFrameEncryptionObserver* observer);

    virtual ~SFrameFrameEncryptor();
    
    /** Update base key method.

         Updates the base key (associated to the encryptor Key ID) used by the encryptor instance.

         @param kid Key ID (KID) value associated to the encryptor instance using a given key.
         @param base_key_data Data chunck containing SFrame base key to the given KID.
         @return Flag indicating whether the base key was updated correctly or not.
     */
    bool UpdateBaseKey(const uint8_t* base_key_data, size_t base_key_length);

    // FrameEncryptorInterface implementation.
    int Encrypt(cricket::MediaType media_type,
                webrtc::FrameEncryptorInterface::CodecType codec_type,
                uint32_t ssrc,
                rtc::ArrayView<const uint8_t> additional_data,
                rtc::ArrayView<const uint8_t> frame,
                rtc::ArrayView<uint8_t> encrypted_frame,
                size_t* bytes_written) override;
    size_t GetMaxCiphertextByteSize(cricket::MediaType media_type,
                                    size_t frame_size) override;
    bool isOk();

protected:
    SFrameFrameEncryptor(uint64_t kid, const uint8_t* base_key_data, size_t base_key_length, SFrameEngineCipherSuite suite, SFrameFrameEncryptionObserver* observer);

    SFrameFrameEncryptionObserver::CodecType FrameEncryptorToObserverCodecType(webrtc::FrameEncryptorInterface::CodecType codec_type) const;

private:
    uint64_t kid_;
    std::unique_ptr<SFrameEngineInterface> engine_;
    SFrameFrameEncryptionObserver* observer_ = nullptr;
};

} // namespace vonage

#endif // MODULES_VONAGE_API_CRYPTO_SFRAME_VONAGE_SFRAME_ENCRYPTOR_H
