#pragma once
#include "api/scoped_refptr.h"
#include "api/function_view.h"
#include "modules/vonage/api/version.h"

#include <memory>


#if VONAGE_WEBRTC_VERSION == 99
#include "rtc_base/task_utils/to_queued_task.h"
#elif VONAGE_WEBRTC_VERSION == 121
#include "api/task_queue/pending_task_safety_flag.h"
#endif

namespace rtc {
class Thread;
}

namespace vonage {

class ThreadWrapperInternal{
public:
    virtual ~ThreadWrapperInternal() = default;
    virtual void SyncInvokeInternal(rtc::FunctionView<void()> func) = 0;
#if VONAGE_WEBRTC_VERSION == 99
    virtual void AsyncInvokeInternal(std::unique_ptr<webrtc::QueuedTask> task, uint32_t timeout_ms) = 0;
#elif VONAGE_WEBRTC_VERSION == 121
    virtual void AsyncInvokeInternal(absl::AnyInvocable<void() &&> task, uint32_t timeout_ms) = 0;
#endif
    virtual rtc::Thread* GetThread() const = 0;
    virtual rtc::scoped_refptr<webrtc::PendingTaskSafetyFlag> GetTaskSafetyFlag() const = 0;
    virtual bool IsCurrent() const = 0;
};

class RTC_LOCKABLE RTC_EXPORT ThreadWrapper {
public:
    ThreadWrapper(bool with_socket_server, const std::string& thread_name);
    ThreadWrapper(rtc::Thread* thread);
    virtual ~ThreadWrapper();
    rtc::Thread* GetThread() const;
    bool IsCurrent() const;
    
    template<class Functor,
    typename ReturnT = std::invoke_result_t<Functor>,
    typename = typename std::enable_if<std::is_void<ReturnT>::value>::type>
    void SyncInvoke(Functor&& functor){
        thread_wrapper_impl_->SyncInvokeInternal(std::forward<Functor>(functor));
    }
    
    template<class Functor,
    typename ReturnT = std::invoke_result_t<Functor>,
    typename = typename std::enable_if<!std::is_void<ReturnT>::value>::type>
    ReturnT SyncInvoke(Functor&& functor){
        ReturnT result;
        SyncInvoke([&]{
            result = functor();
        });
        return result;
    }
    
    template<class Functor>
    void AsyncInvoke(Functor&& functor){
#if VONAGE_WEBRTC_VERSION == 99
        auto task = std::make_unique<webrtc::webrtc_new_closure_impl::SafetyClosureTask<Functor>>(thread_wrapper_impl_->GetTaskSafetyFlag(),
              std::forward<Functor>(functor));
        thread_wrapper_impl_->AsyncInvokeInternal(std::move(task), 0);
#elif VONAGE_WEBRTC_VERSION == 121
        thread_wrapper_impl_->AsyncInvokeInternal(std::forward<Functor>(functor), 0);
#endif        
    }
    
    template<class Functor>
    void AsyncInvoke(Functor&& functor, uint32_t timeout_ms){
#if VONAGE_WEBRTC_VERSION == 99
        auto task = std::make_unique<webrtc::webrtc_new_closure_impl::SafetyClosureTask<Functor>>(thread_wrapper_impl_->GetTaskSafetyFlag(),
              std::forward<Functor>(functor));
        thread_wrapper_impl_->AsyncInvokeInternal(std::move(task), timeout_ms);
#elif VONAGE_WEBRTC_VERSION == 121
        thread_wrapper_impl_->AsyncInvokeInternal(std::forward<Functor>(functor), timeout_ms);
#endif        
    }
    
private:
    std::unique_ptr<ThreadWrapperInternal> thread_wrapper_impl_;
};
}
