#pragma once

#include "absl/base/attributes.h"
#include "api/audio_options.h"
#include "api/audio/audio_frame.h"
#include "api/video/video_frame.h"
#include "modules/vonage/api/helpers/thread_wrapper.h"
#include "media_processor_status_code.h"
#include "modules/vonage/api/version.h"

namespace rtc{
class Thread;
}

namespace webrtc {

// Helper interface useful to share info with transformer base class below.
class BaseFrameTransformerNotifier {
public:
    BaseFrameTransformerNotifier() = default;
    virtual ~BaseFrameTransformerNotifier() = default;

    virtual void OnAudioOptions(const cricket::AudioOptions& options);
    // Note: This list of methods can grow as needed.
};

// BaseFrameTransformer observer class as a mechanism to observe transformer events.
class BaseFrameTransformerObserver {
public:
    friend class MediaProcessorStatus;
    
    BaseFrameTransformerObserver(vonage::ThreadWrapper* observer_thread);
    BaseFrameTransformerObserver(rtc::Thread* observer_thread);
    virtual ~BaseFrameTransformerObserver();

#if VONAGE_WEBRTC_VERSION != 121
    ABSL_DEPRECATED("Use OnWarning(uint8_t code, const std::string& message) instead")
    virtual void OnWarning(MediaProcessorWarningCode code, const std::string& message) = 0;
    ABSL_DEPRECATED("Use OnError(uint8_t code, const std::string& message) instead")
    virtual void OnError(MediaProcessorErrorCode code, const std::string& message) = 0;
#endif
    virtual void OnWarning(uint8_t code, const std::string& message) = 0;
    virtual void OnError(uint8_t code, const std::string& message) = 0;

private:
#if VONAGE_WEBRTC_VERSION != 121
    void ProxyOnWarning(MediaProcessorWarningCode code, const std::string& message);
    void ProxyOnError(MediaProcessorErrorCode code, const std::string& message);
#endif
    void ProxyOnWarning(uint8_t code, const std::string& message);
    void ProxyOnError(uint8_t code, const std::string& message);

private:
    vonage::ThreadWrapper* observer_thread_;
    bool owns_thread_instance_;
};

// We can see BaseFrameTransformer class as a combination of:
// - https://streams.spec.whatwg.org/#transformer-api.
// - https://vonage.github.io/media-processor-docs/docs/api/interfaces/MediaProcessorInterface
template<typename T>
class BaseFrameTransformer : public BaseFrameTransformerNotifier {
public:
    BaseFrameTransformer(BaseFrameTransformerObserver* observer) : observer_(observer){;}
    virtual ~BaseFrameTransformer(){ observer_ = nullptr; }

    virtual void Transform(T* target_frame) = 0;

protected:
    BaseFrameTransformerObserver* observer_;
};
}
