/** @file media_processor.h
    @brief Vonage Media Processor.

    This file includes the type definition for an Vonage Media processor class along with
    several method declarations useful when dealing with a instance.
*/
#ifndef MODULES_VONAGE_API_MEDIA_PROCESSOR_MEDIA_PROCESSOR_H
#define MODULES_VONAGE_API_MEDIA_PROCESSOR_MEDIA_PROCESSOR_H

#include <memory>

#include "modules/vonage/api/media_processor/base_frame_transformer.h"

namespace cricket {
class MediaEngineInterface;
}

namespace rtc {
class Thread;
}

namespace webrtc {
class AudioFrame;
class MediaStreamTrackInterface;
class PeerConnectionInterface;
class VideoFrame;
}

namespace vonage {

/** Source type enumeration for consumers using MediaProcessor instances.

    This enumeration represents several source types expected to be used by consumers of the library.
 */
enum class VonageSourceType {
    kNone, /**< Not specified type .*/
    kAutomation, /**< Source related to some kind of automated usage of the library such as continous integration environment .*/
    kTest, /**< Source related to some kind of test likely to be manual .*/
    kVBC, /**< Source related to some use from Vonage Business  .*/
    kVideo, /**< Source related to some use from Vonage video client SDKs  .*/
    kVoice /**< Source related to some use from Vonage voice client SDKs  .*/
};

/** VonageMetadata stuct definition.
 */
struct VonageMetadata {
    std::string appId = ""; /**< Some string identifying the application using the library .*/
    VonageSourceType sourceType = VonageSourceType::kNone; /**< Source type enumeration value for the consumer using the library .*/

    // TODO: Add IP Proxy support.
};

/** MediaProcessor template class definition.

 We can see MediaProcessor class as a combination of:
 - https://vonage.github.io/media-processor-docs/classes/MediaProcessor.html
 - https://vonage.github.io/media-processor-docs/classes/MediaProcessorConnector.html
 */
template<class T>
class MediaProcessor {
public:
    virtual ~MediaProcessor() = default;

    /** Sets some metadata for telemetry.
     @param metadata Specifies the addional information being sent with the telemetry collected by the library.
     @return bool Flag indicating either success or error in the method.
     */
    virtual bool SetVonageMedatata(const VonageMetadata& metadata) = 0;
    /** Getter helper method to get access to the metadata set
     @return The metadata.
     */
    virtual VonageMetadata GetVonageMedatata() const = 0;

    /** Sets the array of transfromer instances that will be hold and ran by the media processor instance.
     @param transformers The array of transfromer instances.
     @return bool Flag indicating either success or error in the method.
     */
    virtual bool SetTransformers(std::vector<std::shared_ptr<webrtc::BaseFrameTransformer<T>>> transformers) = 0;

    /** Sets the mutable track to process.
     @param track The media stream track instance to be processed.
     @return bool Flag indicating either success or error in the method.
     */
    virtual bool SetTrack(rtc::scoped_refptr<webrtc::MediaStreamTrackInterface> track) = 0;

    // TODO: Add observer setter method for error and warning notifications.
};

/** VideoProcessor class definition.

 Represents the media processor for video.

 See an example bellow.

 @code

 // ...

 class VonageInverseColorVideoTransformer : public webrtc::BaseFrameTransformer<webrtc::VideoFrame> {
 public:
     VonageInverseColorVideoTransformer() {}
     virtual ~VonageInverseColorVideoTransformer() {}

     // webrtc::BaseFrameTransformer
     void Transform(webrtc::VideoFrame* target_frame) override {
         RTC_LOG_T_F(LS_VERBOSE) << "VonageInverseColorVideoTransformer";

         if (target_frame == nullptr) {
             RTC_LOG_T_F(LS_WARNING) << "Video frame is null";
             return;
         }

         rtc::scoped_refptr<webrtc::VideoFrameBuffer> buffer(target_frame->video_frame_buffer());
         // Do red and blue inverse colors by replacing the U V component in the YUV data with V U.
         target_frame->set_video_frame_buffer(webrtc::I420Buffer::Copy(target_frame->width(),
                                                                     target_frame->height(),
                                                                     buffer->ToI420()->DataY(),
                                                                     buffer->ToI420()->StrideY(),
                                                                     buffer->ToI420()->DataV(),
                                                                     buffer->ToI420()->StrideV(),
                                                                     buffer->ToI420()->DataU(),
                                                                     buffer->ToI420()->StrideU()));
     }
 };

 //..

 std::unique_ptr<vonage::VideoProcessor> video_processor_;
 std::vector<std::shared_ptr<webrtc::BaseFrameTransformer<webrtc::VideoFrame>>> video_transformers_;

 //..

 video_transformers_.push_back(std::make_shared<VonageInverseColorVideoTransformer>());

 //..

 rtc::scoped_refptr<webrtc::VideoTrackInterface> video_track_interface = peer_connection_factory_->CreateVideoTrack(kVideoLabel, video_source_);
 video_processor_.reset(new vonage::VideoProcessor());
 video_processor_->SetTransformers(video_transformers_);
 if (video_processor_->SetTrack(video_track_interface) == false) {
     RTC_LOG_T_F(LS_WARNING) << "Video processor set track method failed";
 }
 @endcode
 */
class VideoProcessor: public MediaProcessor<webrtc::VideoFrame> {
public:
    VideoProcessor();
    virtual ~VideoProcessor();

    // MediaProcessor
    bool SetVonageMedatata(const VonageMetadata& metadata) override;
    VonageMetadata GetVonageMedatata() const override;
    bool SetTransformers(std::vector<std::shared_ptr<webrtc::BaseFrameTransformer<webrtc::VideoFrame>>> transformers) override;
    bool SetTrack(rtc::scoped_refptr<webrtc::MediaStreamTrackInterface> track) override;

private:
    bool SetTransformersInternal();
    bool RemoveTransformersInternal();

    rtc::scoped_refptr<webrtc::MediaStreamTrackInterface> track_;
    std::vector<std::shared_ptr<webrtc::BaseFrameTransformer<webrtc::VideoFrame>>> transformers_;
};

/** AudioProcessor class definition.

 Represents the media processor for audio.
 
*/
class AudioProcessor: public MediaProcessor<webrtc::AudioFrame> {
public:
    /** AudioProcessor constructor

     Use this constructor when the peer connection instance was created from a peer connection factory instance where the WebRTC worker thread was not provided (the application did not create the WebRTC worker thread).
     The developer must ensure the worker thread outlives the peer connection instance.
     */
    AudioProcessor(rtc::scoped_refptr<webrtc::PeerConnectionInterface> peer_connection);
    /** AudioProcessor constructor

     Use this constructor when the peer connection instance was created from a peer connection factory instance where the WebRTC worker thread was provided (the application created the WebRTC worker thread).
     The developer must ensure the worker thread outlives the peer connection instance.
     */
    AudioProcessor(rtc::scoped_refptr<webrtc::PeerConnectionInterface> peer_connection, rtc::Thread* worker_thread);
    virtual ~AudioProcessor();

    // MediaProcessor
    bool SetVonageMedatata(const VonageMetadata& metadata) override;
    VonageMetadata GetVonageMedatata() const override;
    bool SetTransformers(std::vector<std::shared_ptr<webrtc::BaseFrameTransformer<webrtc::AudioFrame>>> transformers) override;
    bool SetTrack(rtc::scoped_refptr<webrtc::MediaStreamTrackInterface> track) override;

private:
    bool SetTransformersInternal();
    bool RemoveTransformersInternal();

    bool SetTransformersInternal_w(cricket::MediaEngineInterface* media_engine);
    void RemoveTransformersInternal_w(cricket::MediaEngineInterface* media_engine);

    rtc::scoped_refptr<webrtc::MediaStreamTrackInterface> track_;
    std::vector<std::shared_ptr<webrtc::BaseFrameTransformer<webrtc::AudioFrame>>> transformers_;
    rtc::scoped_refptr<webrtc::PeerConnectionInterface> peer_connection_;
    rtc::Thread* worker_thread_ = nullptr;
};

}

#endif // MODULES_VONAGE_API_MEDIA_PROCESSOR_MEDIA_PROCESSOR_H
