package com.vungle.warren;

import com.google.gson.annotations.SerializedName;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;

import androidx.annotation.IntDef;
import androidx.annotation.Keep;
import androidx.annotation.NonNull;

/**
 * Settings and Configuration Class for playAd Requests.
 * To prevent any issues, generate a new AdConfig object per playAd() request.
 */
public class AdConfig extends BaseAdConfig {

    @SerializedName("ordinal")
    private int ordinal;

    @IntDef(flag = true, value = {MUTED, IMMEDIATE_BACK, IMMERSIVE, TRANSITION_ANIMATE})
    public @interface Settings {
    }

    /**
     * Allow the user to close the ad immediately after starting it. Usually there is a delay before
     * the X or skip is shown. The back button is the same as the X.
     */
    public static final int IMMEDIATE_BACK = 1 << 1;

    /**
     * Whether or not to use immersive mode. Defaults to true.
     */
    public static final int IMMERSIVE = 1 << 2;

    /**
     * Apply a fade-in/fade-out animation to the advertisement activity
     */
    public static final int TRANSITION_ANIMATE = 1 << 3;

    /**
     * Enum for Ad Sizes.
     */
    /* Kept in here to avoid forcing unnecessary integration breakage. */
    @Keep
    public enum AdSize {

        /** should be used only along with {@link Banners#getBanner(String, BannerAdConfig, PlayAdCallback)} for IAB standard MREC size **/
        VUNGLE_MREC("mrec",300, 250),
        /** should be used only along with {@link Vungle#playAd(String, AdConfig, PlayAdCallback)} for generic Ad **/
        VUNGLE_DEFAULT("default",-1,-1),
        BANNER("banner",320,50),
        BANNER_SHORT("banner_short",300,50),
        BANNER_LEADERBOARD("banner_leaderboard",728, 90);

        private final int width, height;
        private final String name;

        AdSize(String name, int width, int height) {
            this.width = width;
            this.height = height;
            this.name = name;
        }

        public String getName() {
            return this.name;
        }

        public int getWidth() {
            return width;
        }

        public int getHeight() {
            return height;
        }

        /**
         * Finds suitable {@link AdSize} for given {@link String} name
         *
         * @param name {@link String} to compare available String representations against
         * @return {@link AdSize} that corresponds to given name or {@link #VUNGLE_DEFAULT}
         */
        public static AdSize fromName(String name) {
            for (AdSize e : AdSize.values()) {
                if (e.name.equals(name)) {
                    return e;
                }
            }
            return VUNGLE_DEFAULT;
        }

        /**
         * @param adSize {@link AdSize} to be checked
         * @return {@code true} if provided AdSize is suitable for non-banner(default, mrec)
         */
        public static boolean isDefaultAdSize(@NonNull AdSize adSize) {
            return adSize == VUNGLE_DEFAULT || adSize == VUNGLE_MREC;
        }

        /**
         * @param adSize {@link AdSize} to be checked
         * @return {@code true} if provided AdSize is suitable for Banner
         */
        public static boolean isBannerAdSize(@NonNull AdSize adSize) {
            return adSize == BANNER || adSize == BANNER_LEADERBOARD || adSize == BANNER_SHORT
                    || adSize == VUNGLE_MREC;
        }

        /**
         * @param adSize {@link AdSize} to be checked
         * @return {@code true} if provided AdSize is suitable for Banner but not MREC
         */
        public static boolean isNonMrecBannerAdSize(@NonNull AdSize adSize) {
            return adSize != VUNGLE_MREC && isBannerAdSize(adSize);
        }
    }

    /**
     * When set the SDK will check if it has an Ad that can be rendered in the specific rotation
     */
    @SerializedName("adOrientation")
    private @Orientation int adOrientation = AUTO_ROTATE;

    @IntDef({PORTRAIT, LANDSCAPE, AUTO_ROTATE, MATCH_VIDEO})
    public @interface Orientation {}

    public static final int PORTRAIT = 0;
    public static final int LANDSCAPE = 1;
    public static final int AUTO_ROTATE = 2;
    public static final int MATCH_VIDEO = 3;

    protected boolean orientationChangedByApi;

    public AdConfig() {
        super();
    }

    public AdConfig(AdSize adSize) {
        super(adSize);
    }

    public AdConfig(BaseAdConfig baseAdConfig) {
        super(baseAdConfig);
    }

    /**
     * <p>Sets whether the Android back button will be immediately enabled during the
     * video ad, or it will be inactive until the on screen close button appears
     * <i>(the default)</i>.</p>
     * <p>
     * <p>Once enabled, the Android back button allows the user to skip the video
     * ad and proceed to the post-roll if one exists; if the ad does not have a
     * post-roll the ad simply ends.</p>
     *
     * @param isBackButtonImmediatelyEnabled <code>true</code> if back button should
     *                                       be enabled before ad close button appears; otherwise, <code>false</code>
     */
    public void setBackButtonImmediatelyEnabled(boolean isBackButtonImmediatelyEnabled) {
        if (isBackButtonImmediatelyEnabled) {
            settings = settings | IMMEDIATE_BACK;
        } else {
            settings = settings & ~IMMEDIATE_BACK;
        }
    }

    /**
     * Sets whether
     * <a href="https://developer.android.com/training/system-ui/immersive.html" target="_blank">immersive mode</a>
     * will be enabled for KitKat (API level 19) and later devices.
     *
     * @param isImmersiveMode <code>true</code> if immersive mode will be
     *                        enabled for KitKat+ devices; otherwise, <code>false</code>.
     */
    public void setImmersiveMode(boolean isImmersiveMode) {
        if (isImmersiveMode) {
            settings = settings | IMMERSIVE;
        } else {
            settings = settings & ~IMMERSIVE;
        }
    }

    /**
     * <p>Sets whether the video transition animation should be enabled or disabled.
     * The default is enabled (a fade-in/fade-out animation).</p>
     *
     * @param isTransitionAnimationEnabled <code>true</code> if the
     *                                     video transition animation should be enabled;
     *                                     <code>false</code> if it should be disabled.
     */
    public void setTransitionAnimationEnabled(boolean isTransitionAnimationEnabled) {
        if (isTransitionAnimationEnabled) {
            settings = settings | TRANSITION_ANIMATE;
        } else {
            settings = settings & ~TRANSITION_ANIMATE;
        }
    }

    /**
     * @return The number of advertisements set by the developer that have played before this ad
     */
    public int getOrdinal() {
        return ordinal;
    }

    /**
     * Inlcuding this advertisement, the number of advertisements that have been played
     * this session. This is not tracked internally because apps can have multiple ad
     * providers.
     *
     * @param ordinal Count in integers
     */
    public void setOrdinal(int ordinal) {
        this.ordinal = ordinal;
    }

    /**
     * Returns AdOrientation
     *
     * @return AdOrientation
     */
    public @Orientation int getAdOrientation() {
        return adOrientation;
    }

    /**
     * Sets the AdOrientation
     *
     * @param adOrientation for the Ad
     */
    public void setAdOrientation(@Orientation int adOrientation) {
        this.adOrientation = adOrientation;

        orientationChangedByApi = true;
    }

    public static final int TOP_LEFT = 0;
    public static final int TOP_RIGHT = 1;
    public static final int BOTTOM_LEFT = 2;
    public static final int BOTTOM_RIGHT = 3;

    @IntDef({TOP_LEFT, TOP_RIGHT, BOTTOM_LEFT, BOTTOM_RIGHT})
    public @interface AdOptionsPosition {
    }

    private @AdOptionsPosition int adOptionsPosition = TOP_RIGHT;

    /**
     * The privacy icon overlay is set to the top right corner by default.
     *
     * @param position Apps can change which corner this overlay is rendered in by setting this property.
     */
    public void setAdOptionsPosition(@AdOptionsPosition int position) {
        this.adOptionsPosition = position;
    }

    public @AdOptionsPosition int getAdOptionsPosition() {
        return this.adOptionsPosition;
    }
}
