package com.vungle.warren;

import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.ALREADY_PLAYING_ANOTHER_AD;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.error.VungleException.MISSING_HBP_EVENT_ID;
import static com.vungle.warren.error.VungleException.OPERATION_CANCELED;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.RENDER_ERROR;
import static com.vungle.warren.model.Advertisement.VIEWING;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.Report;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.tasks.SendReportsJob;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.utility.Constants;

import java.util.Map;

public class AdEventListener implements AdContract.AdvertisementPresenter.EventListener {

    private static final String TAG = AdEventListener.class.getCanonicalName();
    private final Repository repository;
    private final AdLoader adLoader;
    private final JobRunner jobRunner;
    private final VisionController visionController;

    private final Map<String, Boolean> playOperations;
    private final PlayAdCallback playAdCallback;
    private final AdRequest request;

    private boolean successfulView;

    /**
     * The percentage of the ad that was viewed. We initialize to -1 in order to differentiate
     * between the ad never starting.
     */
    private int percentViewed = -1;

    private boolean adRewarded;
    private Placement placement;
    private Advertisement advertisement;

    AdEventListener(@NonNull AdRequest request,
                    @NonNull Map<String, Boolean> playOperations,
                    @Nullable PlayAdCallback playAdCallback,
                    @NonNull Repository repository,
                    @NonNull AdLoader adLoader,
                    @NonNull JobRunner jobRunner,
                    @NonNull VisionController visionController,
                    @Nullable Placement placement,
                    @Nullable Advertisement advertisement) {
        this.request = request;
        this.playOperations = playOperations;
        this.playAdCallback = playAdCallback;
        this.repository = repository;
        this.adLoader = adLoader;
        this.jobRunner = jobRunner;
        this.visionController = visionController;
        this.placement = placement;
        this.advertisement = advertisement;
        playOperations.put(request.getPlacementId(), true);
    }

    public void onNext(@NonNull String s, String value, String id) {
        loadAdvertisement();
        if (advertisement == null) {
            Log.e(TAG, "No Advertisement for ID");
            onFinished();
            if (playAdCallback != null) {
                playAdCallback.onError(request.getPlacementId(), new VungleException(AD_UNABLE_TO_PLAY));
                VungleLogger.error("AdEventListener#PlayAdCallback", id + ": AD_UNABLE_TO_PLAY");
            }
            return;
        }

        loadPlacement();
        if (placement == null) {
            Log.e(TAG, "No Placement for ID");
            onFinished();
            if (playAdCallback != null) {
                playAdCallback.onError(request.getPlacementId(), new VungleException(PLACEMENT_NOT_FOUND));
                VungleLogger.error("AdEventListener#PlayAdCallback", "PLACEMENT_NOT_FOUND: " + id);
            }
            return;
        }

        try {
            if (s.equals("start")) {
                repository.saveAndApplyState(advertisement, id, VIEWING);

                if (playAdCallback != null) {
                    playAdCallback.onAdStart(id);
                    VungleLogger.debug("AdEventListener#PlayAdCallback", "onAdStart: " + id);
                }

                percentViewed = 0;
                placement = repository.load(request.getPlacementId(), Placement.class).get();
                if (placement != null) {
                    /// If the placement is auto-cached or HBP is enabled, download a new advertisement(s) for it soon.
                    adLoader.loadEndlessIfNeeded(placement, placement.getAdSize(), 0, request.getIsExplicit());
                }
                if (visionController.isEnabled()) {
                    visionController.reportData(
                            advertisement.getCreativeId(),
                            advertisement.getCampaignId(),
                            advertisement.getAdvertiserAppId());
                }
            } else if (s.equals("end")) {
                Log.d("Vungle", "Cleaning up metadata and assets for placement " + id + " and advertisement " + advertisement.getId());
                /// Cleanup the asset folder for the advertisement
                repository.saveAndApplyState(advertisement, id, Advertisement.DONE);
                repository.updateAndSaveReportState(id, advertisement.getAppID(), Report.NEW, Report.READY);

                /// Schedule a job to send the report soon
                jobRunner.execute(SendReportsJob.makeJobInfo(false));

                onFinished();

                /// Inform the listener about the advertisement ending
                if (playAdCallback != null) {
                    playAdCallback.onAdEnd(id,
                            (successfulView || percentViewed >= 80),
                            (value != null && value.equals("isCTAClicked")));
                    playAdCallback.onAdEnd(id);
                    SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                            .setEvent(SessionEvent.DID_CLOSE)
                            .addData(SessionAttribute.EVENT_ID, advertisement.getId())
                            .build());
                    VungleLogger.debug("AdEventListener#PlayAdCallback", "onAdEnd: " + id);
                }

            } else if (s.equals("successfulView") && placement.isIncentivized()) {
                successfulView = true;
                if (!adRewarded) {
                    adRewarded = true;
                    if (playAdCallback != null) {
                        playAdCallback.onAdRewarded(id);
                        SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                                .setEvent(SessionEvent.REWARDED)
                                .addData(SessionAttribute.EVENT_ID, advertisement.getId())
                                .build());
                        VungleLogger.debug("AdEventListener#PlayAdCallback", "onAdRewarded: " + id);
                    }
                }
            } else if ("open".equals(s) && playAdCallback != null) {
                if ("adClick".equals(value)) {
                    playAdCallback.onAdClick(id);
                    VungleLogger.debug("AdEventListener#PlayAdCallback", "onAdClick: " + id);
                } else if ("adLeftApplication".equals(value)) {
                    playAdCallback.onAdLeftApplication(id);
                    VungleLogger.debug("AdEventListener#PlayAdCallback", "onAdLeftApplication: " + id);
                }
            } else if ("adViewed".equals(s) && playAdCallback != null) {
                playAdCallback.onAdViewed(id);
            } else if (Constants.ATTACH.equals(s) && playAdCallback != null) {
                playAdCallback.creativeId(value);
            }
        } catch (DatabaseHelper.DBException ignored) {
            onError(new VungleException(DB_ERROR), id);
        }
    }

    @Override
    public void onError(VungleException exception, String placementId) {
        loadAdvertisement();

        if (advertisement != null && exception.getExceptionCode() == RENDER_ERROR) {
            adLoader.dropCache(advertisement.getId());
            return;
        }

        if (advertisement != null
                && exception.getExceptionCode() != ALREADY_PLAYING_ANOTHER_AD//todo missing test for the code
                && exception.getExceptionCode() != OPERATION_CANCELED
                && exception.getExceptionCode() != MISSING_HBP_EVENT_ID) {
            try {
                repository.saveAndApplyState(advertisement, placementId, Advertisement.ERROR);
                loadPlacement();
                if (placement != null) {
                    adLoader.loadEndlessIfNeeded(placement, placement.getAdSize(), 0, false);
                }
            } catch (DatabaseHelper.DBException e) {
                exception = new VungleException(DB_ERROR);
            }
        }

        onFinished();

        if (playAdCallback != null) {
            playAdCallback.onError(placementId, exception);
            VungleLogger.error("AdEventListener#PlayAdCallback",
                    exception.getLocalizedMessage() + " :" + placementId);
        }
    }

    protected void onFinished() {
        playOperations.remove(request.getPlacementId());
    }

    private void loadAdvertisement() {
        if (advertisement == null) {
            advertisement = repository.findValidAdvertisementForPlacement(request.getPlacementId(), request.getEventId()).get();
        }
    }

    private void loadPlacement() {
        if (placement == null) {
            placement = repository.load(request.getPlacementId(), Placement.class).get();
        }
    }

}

