package com.vungle.warren;

import static com.vungle.warren.downloader.DownloadRequest.Priority.HIGHEST;
import static com.vungle.warren.error.VungleException.AD_FAILED_TO_DOWNLOAD;
import static com.vungle.warren.error.VungleException.ASSET_DOWNLOAD_ERROR;
import static com.vungle.warren.error.VungleException.ASSET_DOWNLOAD_RECOVERABLE;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.error.VungleException.INVALID_SIZE;
import static com.vungle.warren.error.VungleException.MISSING_HBP_EVENT_ID;
import static com.vungle.warren.error.VungleException.NETWORK_ERROR;
import static com.vungle.warren.error.VungleException.NO_SERVE;
import static com.vungle.warren.error.VungleException.NO_SPACE_TO_DOWNLOAD_ASSETS;
import static com.vungle.warren.error.VungleException.NO_SPACE_TO_LOAD_AD;
import static com.vungle.warren.error.VungleException.NO_SPACE_TO_LOAD_AD_AUTO_CACHED;
import static com.vungle.warren.error.VungleException.OPERATION_CANCELED;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.SERVER_ERROR;
import static com.vungle.warren.error.VungleException.SERVER_RETRY_ERROR;
import static com.vungle.warren.error.VungleException.SERVER_TEMPORARY_UNAVAILABLE;
import static com.vungle.warren.error.VungleException.UNKNOWN_ERROR;
import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;
import static com.vungle.warren.model.AdAsset.Status.DOWNLOAD_SUCCESS;
import static com.vungle.warren.model.AdAsset.Status.PROCESSED;
import static com.vungle.warren.model.Advertisement.ERROR;
import static com.vungle.warren.model.Advertisement.KEY_POSTROLL;
import static com.vungle.warren.model.Advertisement.KEY_TEMPLATE;
import static com.vungle.warren.model.Advertisement.NEW;
import static com.vungle.warren.model.Advertisement.READY;
import static com.vungle.warren.model.Advertisement.TYPE_VUNGLE_MRAID;
import static com.vungle.warren.model.Placement.TYPE_VUNGLE_BANNER;

import android.text.TextUtils;
import android.util.Log;
import android.webkit.URLUtil;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.vungle.warren.AdConfig.AdSize;
import com.vungle.warren.analytics.JobDelegateAnalytics;
import com.vungle.warren.downloader.AssetDownloadListener;
import com.vungle.warren.downloader.AssetDownloadListener.DownloadError.ErrorReason;
import com.vungle.warren.downloader.AssetPriority;
import com.vungle.warren.downloader.DownloadRequest;
import com.vungle.warren.downloader.Downloader;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.AdAsset;
import com.vungle.warren.model.AdAsset.FileType;
import com.vungle.warren.model.AdAsset.Status;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.JsonUtil;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.model.admarkup.AdMarkupV2;
import com.vungle.warren.network.Call;
import com.vungle.warren.network.Callback;
import com.vungle.warren.network.Response;
import com.vungle.warren.omsdk.OMInjector;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionConstants;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.tasks.DownloadJob;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.ui.HackMraid;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.FileUtility;
import com.vungle.warren.utility.UnzipUtility;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

/**
 * The AdLoader is responsible for loading Ad and it's assets.
 */
public class AdLoader {

    private static final String TAG = AdLoader.class.getCanonicalName();
    public static final long EXPONENTIAL_RATE = 2;
    public static final long RETRY_DELAY = 2L * 1000L;
    public static final int RETRY_COUNT = 5;
    public static final boolean DEFAULT_LOAD_OPTIMIZATION_ENABLED = false;
    /* Logging contexts */
    public static final String TT_DOWNLOAD_CONTEXT = "ttDownloadContext";
    private static final String LOAD_AD_EXECUTE_CONTEXT = "AdLoader#loadAd#execute; loadAd sequence";
    private static final String FETCH_AD_METADATA_CONTEXT = "AdLoader#fetchAdMetadata; loadAd sequence";
    private static final String DOWNLOAD_AD_ASSETS_CONTEXT = "AdLoader#downloadAdAssets; loadAd sequence";
    private static final String GET_ASSET_DOWNLOAD_LISTENER_CONTEXT = "AdLoader#getAssetDownloadListener; loadAd sequence";
    private static final String ON_ASSET_DOWNLOAD_FINISHED_CONTEXT = "AdLoader#onAssetDownloadFinished; loadAd sequence";
    private static final String DOWNLOAD_AD_CALLBACK_ON_DOWNLOAD_COMPLETED_CONTEXT = "AdLoader#DownloadAdCallback#onDownloadCompleted; loadAd sequence";

    private static final String NOT_A_DIR = "not a dir";
    /* Multiple usage Logging formats */
    private static final String STRING_AND_OP_ID_FORMAT = "%1$s; request = %2$s";
    private static final String OP_ID_AND_ADVERTISEMENT_FORMAT = "request = %1$s; advertisement = %2$s";
    private static final String PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_3_4 = "request = %3$s; advertisement = %4$s";
    private static final String PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_2_3 = "request = %2$s; advertisement = %3$s";

    @IntDef(value = {ReschedulePolicy.EXPONENTIAL, ReschedulePolicy.EXPONENTIAL_ENDLESS_AD})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ReschedulePolicy {
        int EXPONENTIAL = 0;
        /**
         * Try to download assets {@link AdLoader#RETRY_COUNT} times.
         */
        int EXPONENTIAL_ENDLESS_AD = 1;
    }

    @IntDef(value = {Priority.HIGHEST, Priority.HIGH, Priority.LOWEST})
    public @interface Priority {
        int HIGHEST = 0;//for external API call
        int HIGH = 1;//for Placements with priority
        int LOWEST = Integer.MAX_VALUE;//for Placements without priority
    }

    public static class Operation {
        @NonNull
        final AdRequest request;
        @NonNull
        final AdSize size;
        long delay;
        long retryDelay;
        int retry;
        int retryLimit;
        @ReschedulePolicy
        int policy;
        @NonNull
        final Set<LoadAdCallback> loadAdCallbacks = new CopyOnWriteArraySet<>();
        @NonNull
        final AtomicBoolean loading;
        boolean logError;
        @Priority
        int priority;
        List<DownloadRequest> requests = new CopyOnWriteArrayList<>();

        public Operation(@NonNull AdRequest request,
                         @NonNull AdSize size,
                         long delay,
                         long retryDelay,
                         int retryLimit,
                         @ReschedulePolicy int policy,
                         int retry,
                         boolean logError,
                         @Priority int priority,
                         @Nullable LoadAdCallback... loadAdCallbacks) {
            this.request = request;
            this.delay = delay;
            this.retryDelay = retryDelay;
            this.retryLimit = retryLimit;
            this.policy = policy;
            this.retry = retry;
            this.loading = new AtomicBoolean();
            this.size = size;
            this.logError = logError;
            this.priority = priority;
            if (loadAdCallbacks != null) {
                this.loadAdCallbacks.addAll(Arrays.asList(loadAdCallbacks));
            }
        }

        Operation delay(long delay) {
            return new Operation(request, size, delay, retryDelay, retryLimit, policy, retry, logError, priority, loadAdCallbacks.toArray(new LoadAdCallback[0]));
        }

        Operation retryDelay(long retryDelay) {
            return new Operation(request, size, delay, retryDelay, retryLimit, policy, retry, logError, priority, loadAdCallbacks.toArray(new LoadAdCallback[0]));
        }

        Operation retry(int retry) {
            return new Operation(request, size, delay, retryDelay, retryLimit, policy, retry, logError, priority, loadAdCallbacks.toArray(new LoadAdCallback[0]));
        }

        /**
         * Merges this operation with other.
         */
        void merge(Operation other) {
            //keep current: adSize, id, loading
            delay = Math.min(delay, other.delay);
            retryDelay = Math.min(retryDelay, other.retryDelay);
            retryLimit = Math.min(retryLimit, other.retryLimit);
            //EXPONENTIAL_ENDLESS_AD will not deliver callback
            policy = (other.policy == ReschedulePolicy.EXPONENTIAL ? other.policy : policy);
            retry = Math.min(retry, other.retry);
            logError |= other.logError;
            //keep highest priority
            priority = Math.min(priority, other.priority);
            loadAdCallbacks.addAll(other.loadAdCallbacks);
        }

        @NonNull
        @Override
        public String toString() {
            return "request=" + request.toString() +
                    " size=" + size.toString() +
                    " priority=" + priority +
                    " policy=" + policy +
                    " retry=" + retry + "/" + retryLimit +
                    " delay=" + delay + "->" + retryDelay +
                    " log=" + logError;
        }

        @VisibleForTesting
        @NonNull
        public AdRequest getRequest() {
            return request;
        }

        @NonNull
        public AdSize getSize() {
            return size;
        }

        public boolean getLogError() {
            return logError;
        }

        public @Priority int getPriority() {
            return priority;
        }
    }

    /**
     * A state-tracking field which contains information about ongoing load operations. There can
     * only ever be a single load operation per active AdRequest.
     */
    private final Map<AdRequest, Operation> loadOperations = new ConcurrentHashMap<>();
    private final Map<AdRequest, Operation> pendingOperations = new ConcurrentHashMap<>();
    private final List<Operation> startingOperations = new CopyOnWriteArrayList<>();

    private final OperationSequence sequence;
    @Nullable
    private AdRequest sequenceLoadingRequest = null;

    //fixed values
    @NonNull
    private final Repository repository;
    @NonNull
    private final Executors sdkExecutors;
    @NonNull
    private final VungleApiClient vungleApiClient;
    @NonNull
    private final CacheManager cacheManager;
    @NonNull
    private final Downloader downloader;
    @NonNull
    private final RuntimeValues runtimeValues;
    @NonNull
    private final AtomicReference<JobRunner> jobRunnerRef = new AtomicReference<>();
    @NonNull
    private final VungleStaticApi vungleApi;
    @NonNull
    private final VisionController visionController;
    @NonNull
    private final OMInjector omInjector;

    private boolean adLoadOptimizationEnabled = DEFAULT_LOAD_OPTIMIZATION_ENABLED;

    public AdLoader(
            @NonNull Executors sdkExecutors,
            @NonNull Repository repository,
            @NonNull VungleApiClient vungleApiClient,
            @NonNull CacheManager cacheManager,
            @NonNull Downloader downloader,
            @NonNull RuntimeValues runtimeValues,
            @NonNull VungleStaticApi vungleApi,
            @NonNull VisionController visionController,
            @NonNull OperationSequence sequence,
            @NonNull OMInjector omInjector
    ) {
        this.sdkExecutors = sdkExecutors;
        this.repository = repository;
        this.vungleApiClient = vungleApiClient;
        this.cacheManager = cacheManager;
        this.downloader = downloader;
        this.runtimeValues = runtimeValues;
        this.vungleApi = vungleApi;
        this.visionController = visionController;
        this.sequence = sequence;
        this.omInjector = omInjector;
    }

    public void init(@NonNull JobRunner jobRunner) {
        this.jobRunnerRef.set(jobRunner);
        downloader.init();
    }

    private boolean canReDownload(Advertisement advertisement) {
        if (advertisement == null || (advertisement.getState() != NEW && advertisement.getState() != READY)) {
            return false;
        }

        List<AdAsset> adAssets = repository.loadAllAdAssets(advertisement.getId()).get();
        if (adAssets == null || adAssets.size() == 0) {
            return false;
        }

        for (AdAsset asset : adAssets) {
            //Discard ad if unzipped asset is missing
            if (asset.fileType == FileType.ZIP_ASSET) {
                File file = new File(asset.localPath);
                if (!fileIsValid(file, asset))
                    return false;

            } else if (TextUtils.isEmpty(asset.serverPath)) {
                return false;
            }
        }

        return true;
    }

    @WorkerThread
    public boolean canPlayAd(final Advertisement advertisement) {
        if (advertisement == null || advertisement.getState() != READY) {
            return false;
        }

        return hasAssetsFor(advertisement);
    }

    @WorkerThread
    public boolean canRenderAd(Advertisement advertisement) {
        if (advertisement == null)
            return false;


        if (advertisement.getState() != Advertisement.READY && advertisement.getState() != Advertisement.VIEWING)
            return false;

        return hasAssetsFor(advertisement);
    }

    public void clear() {
        Set<AdRequest> requests = new HashSet<>();
        requests.addAll(loadOperations.keySet());
        requests.addAll(pendingOperations.keySet());
        for (AdRequest request : requests) {
            Operation loading = loadOperations.remove(request);
            startingOperations.remove(loading);
            onError(loading, OPERATION_CANCELED);
            onError(pendingOperations.remove(request), OPERATION_CANCELED);
        }
        for (Operation op : startingOperations) {
            startingOperations.remove(op);
            onError(op, OPERATION_CANCELED);
        }
        sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                sequenceLoadingRequest = null;
                for (OperationSequence.Entry op : sequence.removeAll()) {
                    onError(op.operation, OPERATION_CANCELED);
                }
            }
        });
    }

    public boolean isLoading(AdRequest request) {
        //not include starting/waiting operations
        Operation op = loadOperations.get(request);
        return op != null && op.loading.get();
    }

    private void setLoading(AdRequest request, boolean loading) {
        Operation op = loadOperations.get(request);
        if (op != null) {
            op.loading.set(loading);
        }
    }

    /**
     * should be called from {@link DownloadJob} only
     */
    public void loadPendingInternal(final AdRequest request) {
        Operation op = pendingOperations.remove(request);
        if (op == null)
            return;

        load(op.delay(0));
    }

    public void load(@NonNull final Operation op) {
        JobRunner jobRunner = this.jobRunnerRef.get();

        if (jobRunner == null) {
            VungleLogger.error("AdLoader#load; loadAd sequence",
                    String.format("Cannot load operation %s; job runner is null", op));
            onError(op, VUNGLE_NOT_INTIALIZED);
            return;
        }
        if (op.request.getIsExplicit()) {
            SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.LOAD_AD)
                    .addData(SessionAttribute.PLACEMENT_ID, op.request.getPlacementId()).build());
        }

        checkAndUpdateHBPPlacementBannerSize(op.request.getPlacementId(), op.size);

        @Nullable Operation pending = pendingOperations.remove(op.request);
        if (pending != null) {
            op.merge(pending);
        }

        if (op.delay <= 0) {
            //todo: for temporary logs
            op.request.timeStamp.set(System.currentTimeMillis());

            startingOperations.add(op);
            sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                @Override
                public void run() {
                    if (!startingOperations.contains(op)) {
                        return;
                    }

                    Operation starting = op;
                    AdLoader.Operation current = loadOperations.get(starting.request);
                    if (current != null) {
                        int oldPriority = current.priority;
                        current.merge(starting);
                        if (current.priority < oldPriority) {
                            onChangePriority(current);
                        }
                    } else {
                        OperationSequence.Entry next = sequence.remove(starting.request);
                        if (next != null) {
                            next.operation.merge(starting);
                            starting = next.operation;
                        }

                        if (starting.priority <= AdLoader.Priority.HIGHEST) {
                            startLoading(starting);
                        } else {
                            sequence.offer(next == null ? new OperationSequence.Entry(starting) : next);
                            tryLoadNextInQueue(null);
                        }
                    }
                    startingOperations.remove(starting);
                }
            }, new Runnable() {
                @Override
                public void run() {
                    onError(op, VungleException.OUT_OF_MEMORY);
                }
            });

        } else {
            pendingOperations.put(op.request, op);
            jobRunner.execute(DownloadJob.makeJobInfo(op.request).setDelay(op.delay).setUpdateCurrent(true));
        }
    }


    // Mutates the AdSize in the case that we have a banner and multiple ads per placement /
    // header bidding available. Objective is to update the default AdSize to the parameter
    // passed in the loadAd call to be used on future loadAd calls if no AdSize is yet available.
    private void checkAndUpdateHBPPlacementBannerSize(
            final String placementId, final AdConfig.AdSize newAdSize) {
        repository.load(placementId, Placement.class, new Repository.LoadCallback<Placement>() {
            @Override
            public void onLoaded(Placement placement) {
                if (
                        placement != null &&
                                placement.isMultipleHBPEnabled() &&
                                placement.getPlacementAdType() == TYPE_VUNGLE_BANNER &&
                                placement.getAdSize() != newAdSize
                ) {
                    placement.setAdSize(newAdSize);
                    repository.save(placement, null, false);
                }
            }
        });
    }

    @WorkerThread
    private void startLoading(Operation op) {
        loadOperations.put(op.request, op);
        loadAd(op);
    }

    @WorkerThread
    private void tryLoadNextInQueue(@Nullable AdRequest finished) {
        if (sequenceLoadingRequest == null || sequenceLoadingRequest.equals(finished)) {
            sequenceLoadingRequest = null;
            OperationSequence.Entry next = sequence.poll();
            if (next != null) {
                sequenceLoadingRequest = next.operation.request;
                startLoading(next.operation);
            }
        }
    }

    private void onChangePriority(Operation op) {
        for (DownloadRequest request : op.requests) {
            AssetPriority priority = getAssetPriority(op.priority, request.path);
            request.setPriority(priority);
            downloader.updatePriority(request);
        }
    }

    private void onError(@Nullable Operation op, @VungleException.ExceptionCode int code) {
        VungleLogger.error("AdLoader#onError; loadAd sequence",
                String.format("Error %1$s occured; operation is %2$s",
                        new VungleException(code), op != null ? op : "null"));

        if (op != null) {
            for (LoadAdCallback loadAdCallback : op.loadAdCallbacks) {
                loadAdCallback.onError(op.request.getPlacementId(), new VungleException(code));
            }
        }
    }

    private VungleException reposeCodeToVungleException(int code) {
        if (recoverableServerCode(code)) {
            return new VungleException(SERVER_TEMPORARY_UNAVAILABLE);
        } else {
            return new VungleException(SERVER_ERROR);
        }
    }

    private boolean recoverableServerCode(int code) {
        return code == 408 || (500 <= code && code < 600);
    }

    private VungleException retrofitToVungleException(Throwable throwable) {
        if (throwable instanceof UnknownHostException) {
            return new VungleException(AD_FAILED_TO_DOWNLOAD);
        } else if (throwable instanceof IOException) {
            return new VungleException(NETWORK_ERROR);
        } else {
            return new VungleException(AD_FAILED_TO_DOWNLOAD);
        }
    }

    /**
     * Request the Vungle SDK to load the assets for an advertisement by the placement identifier.
     * This will cause us to request an ad from the Ad Server, and if the bid is filled, the assets
     * will be downloaded. It is possible for the bid to get no responses and therefore no assets
     * will be loaded. In this case, the SDK will automatically retryCount at a later time, specified by
     * the Vungle Server. The callback will be notified that the assets are pending download.
     *
     * @param op operation
     */
    @WorkerThread
    private void loadAd(@NonNull final Operation op) {
        final long adRequestStartTimeStamp = System.currentTimeMillis();

        if (!vungleApi.isInitialized()) {
            VungleLogger.error(LOAD_AD_EXECUTE_CONTEXT, "Vungle is not initialized");
            onDownloadFailed(new VungleException(VUNGLE_NOT_INTIALIZED), op.request, null);
            return;
        }

        Placement placement = repository.load(op.request.getPlacementId(), Placement.class).get();
        if (placement == null) {
            VungleLogger.error(LOAD_AD_EXECUTE_CONTEXT, "placement not found for " + op.request);
            onDownloadFailed(new VungleException(PLACEMENT_NOT_FOUND), op.request, null);
            return;
        }

        if (!placement.isValid()) {
            onDownloadFailed(new VungleException(VungleException.UNSUPPORTED_CONFIGURATION), op.request, null);
            return;
        }

        if (isSizeInvalid(placement, op.size)) {
            VungleLogger.error(LOAD_AD_EXECUTE_CONTEXT,
                    "size is invalid, size = " + op.size);
            onDownloadFailed(new VungleException(INVALID_SIZE), op.request, null);
            return;
        }
                if (
                        placement.getPlacementAdType() == Placement.TYPE_VUNGLE_BANNER &&
                        !placement.isMultipleHBPEnabled()
                ) {
            List<Advertisement> advs = repository.findValidAdvertisementsForPlacement(placement.getId(), op.request.getEventId()).get();
            if (advs != null) {
                boolean deleted = false;
                for (Advertisement adv : advs) {
                    if (adv.getAdConfig().getAdSize() != op.size) {
                        try {
                            repository.deleteAdvertisement(adv.getId());
                        } catch (DatabaseHelper.DBException e) {
                            VungleLogger.error(LOAD_AD_EXECUTE_CONTEXT,
                                    "cannot delete advertisement, request = " + op.request);
                            onDownloadFailed(new VungleException(DB_ERROR), op.request, null);
                            return;
                        }
                        deleted = true;
                    }
                }
                if (deleted) {
                            loadEndlessIfNeeded(placement, op.size, 0L, op.request.getIsExplicit());
                }
            }
        }

                @Nullable Advertisement advertisement = null;
                int type = op.request.getType();
                if (type == AdRequest.Type.NORMAL || type == AdRequest.Type.SINGLE_HBP) {
                    advertisement = repository.findValidAdvertisementForPlacement(placement.getId(), op.request.getEventId()).get();
                    if (op.request.getAdMarkup() != null &&
                            advertisement == null &&
                            op.request.getAdMarkup().getVersion() == 2) {
                        AdMarkupV2 adMarkupV2 = (AdMarkupV2) op.request.getAdMarkup();
                        advertisement = adMarkupV2.getAdvertisement();
                        try {
                            repository.save(advertisement);
                        } catch (DatabaseHelper.DBException e) {
                            Log.e(TAG, "Failed to persist ad from Real Time Ad");
                        }
                    }
            //eventId must be passed
                    if (placement.isMultipleHBPEnabled() && op.request.getType() == AdRequest.Type.NORMAL) {
                        String eventId = op.request.getEventId();
                        if (eventId == null) {
                    onDownloadFailed(new VungleException(MISSING_HBP_EVENT_ID), op.request, null);
                    return;
                } else if (advertisement == null) {
                    onDownloadFailed(new VungleException(VungleException.AD_UNABLE_TO_PLAY), op.request, null);
                    return;
                }
            }

            /// If the assets are already loaded and have not expired, do not download again.
            if (advertisement != null && canPlayAd(advertisement)) {
                tryLoadNextInQueue(op.request);
                onReady(op.request, placement, advertisement);
                return;
            } else if (canReDownload(advertisement)) {
                Log.d(TAG, "Found valid adv but not ready - downloading content");

                final VungleSettings settings = runtimeValues.settings.get();
                if (settings == null || cacheManager.getBytesAvailable() < settings.getMinimumSpaceForAd()) {
                    if (advertisement.getState() != ERROR) {
                        try {
                            repository.saveAndApplyState(advertisement, op.request.getPlacementId(), ERROR);
                        } catch (DatabaseHelper.DBException e) {
                            VungleLogger.error(
                                    "AdLoader#loadAd#execute; loadAd sequence; canReDownload branch",
                                    "cannot save/apply ERROR state, request = " + op.request);
                            onDownloadFailed(new VungleException(DB_ERROR), op.request, null);
                            return;
                        }
                    }
                    VungleLogger.error(LOAD_AD_EXECUTE_CONTEXT,
                            "failed to download assets, no space; request = " + op.request);
                    onDownloadFailed(new VungleException(NO_SPACE_TO_DOWNLOAD_ASSETS), op.request, null);
                    return;
                }

                /// Update the instance state
                setLoading(op.request, true);

                if (advertisement.getState() != NEW) {
                    try {
                        repository.saveAndApplyState(advertisement, op.request.getPlacementId(), NEW);
                    } catch (DatabaseHelper.DBException e) {
                        VungleLogger.error(
                                "AdLoader#loadAd#execute; loadAd sequence; canReDownload branch",
                                "cannot save/apply NEW state, request = " + op.request);
                        onDownloadFailed(new VungleException(DB_ERROR), op.request, null);
                        return;
                    }
                }
                advertisement.setAdRequestStartTime(adRequestStartTimeStamp);
                advertisement.setAssetDownloadStartTime(System.currentTimeMillis());
                tryLoadNextInQueue(op.request);
                downloadAdAssets(op, advertisement);
                return;
            }
        } else if (op.request.getType() == AdRequest.Type.NO_ASSETS && isReadyForHBP(op, repository)) {
            /// If all ads are in NEW or READY state, do not download again.
            tryLoadNextInQueue(op.request);
            onReady(op.request, placement, null);
            return;
        }

        if (placement.getWakeupTime() > System.currentTimeMillis()) {
            onDownloadFailed(new VungleException(NO_SERVE), op.request, null);
            VungleLogger.warn("AdLoader#loadAd#execute; loadAd sequence; snoozed branch",
                    String.format("Placement with id %s is snoozed ", placement.getId()));
            Log.w(TAG, "Placement " + placement.getId() + " is  snoozed");

            //rescheduling download to time when placement is active
            Log.d(TAG, "Placement " + placement.getId() + " is sleeping rescheduling it ");
                    loadEndlessIfNeeded(placement, op.size, placement.getWakeupTime() - System.currentTimeMillis(), false);
        } else {
            String advMsg = op.request.getType() == AdRequest.Type.NO_ASSETS ? "advs" : "adv";
            Log.i(TAG, "didn't find cached " + advMsg + " for " + op.request + " downloading");

            if (advertisement != null) {
                try {
                    repository.saveAndApplyState(advertisement, op.request.getPlacementId(), ERROR);
                } catch (DatabaseHelper.DBException e) {
                    VungleLogger.error(
                            "AdLoader#loadAd#execute; loadAd sequence; last else branch",
                            "cannot save/apply ERROR state, request = " + op.request);
                    onDownloadFailed(new VungleException(DB_ERROR), op.request, null);
                    return;
                }
            }

            /// Download the ad
            final VungleSettings settings = runtimeValues.settings.get();
            if (settings != null && cacheManager.getBytesAvailable() < settings.getMinimumSpaceForAd()) {
                VungleLogger.error(
                        "AdLoader#loadAd#execute; loadAd sequence; last else branch",

                        String.format("no space to load, isAutoCached = %1$s, request = %2$s",
                                placement.isAutoCached(), op.request));

                onDownloadFailed(new VungleException(placement.isAutoCached()
                        ? NO_SPACE_TO_LOAD_AD_AUTO_CACHED : NO_SPACE_TO_LOAD_AD), op.request, null);
                return;
            }

            //Clear placement data as no valid adv found?
            Log.d(TAG, "No " + advMsg + " for placement " + placement.getId() + " getting new data ");

            /// Update the instance state
            setLoading(op.request, true);
            fetchAdMetadata(op, placement);
        }
    }

    private boolean isReadyForHBP(@NonNull Operation op, @NonNull Repository repository) {
        List<Advertisement> advs = repository.findValidAdvertisementsForPlacement(op.request.getPlacementId(), null).get();
        //todo drop extra Ads?
        return advs != null && advs.size() >= op.request.getAdCount();
    }

    private boolean isSizeInvalid(Placement placement, AdSize size) {
        // if...else... checking two condition
        // 1 : If placement is BANNER TYPE but requested ad size is non banner than return INVALID_SIZE error
        // 2 : Else If placement is NON_BANNER TYPE but requested ad size is banner than return INVALID_SIZE error
        return (placement.getPlacementAdType() == Placement.TYPE_VUNGLE_BANNER && !AdSize.isNonMrecBannerAdSize(size))
                || (placement.getPlacementAdType() == Placement.TYPE_DEFAULT && !AdSize.isDefaultAdSize(size));
    }

    /**
     * Multi-step method that downloads the required assets for the given placement identifier. It begins by
     * fetching an advertisement for the given placement, which the ad server provides along with all the
     * metadata and pointers to the downloadable assets. Once this information is ready, we download the
     * individual assets. Once the assets have finished downloading, we inform the callback of the success
     * of the operation.
     *
     * @param op        Operation.
     * @param placement The placement should be loaded
     */
    private void fetchAdMetadata(@NonNull final Operation op, @NonNull Placement placement) {
        final long requestStartTime = System.currentTimeMillis();
        if (op.request.getAdMarkup() instanceof AdMarkupV2) {
            AdMarkupV2 adMarkupV2 = (AdMarkupV2) op.request.getAdMarkup();
            handleAdMetaData(op, requestStartTime, adMarkupV2.getAdvertisement(), placement, new JsonObject());
            return;
        }
        VungleLogger.verbose(true, TAG, TT_DOWNLOAD_CONTEXT,
                String.format("Start to request ad, request = %1$s, at: %2$d", op.request, requestStartTime));
        vungleApiClient.requestAd(
                op.request.getPlacementId(), AdSize.isNonMrecBannerAdSize(op.size) ? op.size.getName() : "",
                placement.isHeaderBidding(),
                visionController.isEnabled() ? visionController.getPayload() : null
        ).enqueue(new Callback<JsonObject>() {
            @Override
            public void onFailure(Call<JsonObject> call, final Throwable e) {
                VungleLogger.verbose(true, TAG, TT_DOWNLOAD_CONTEXT,
                        String.format("Request ad failed, request = %1$s, elapsed time = %2$dms", op.request, System.currentTimeMillis() - requestStartTime));
                VungleLogger.error(
                        FETCH_AD_METADATA_CONTEXT,
                        String.format("failed to request ad, request = %1$s, throwable = %2$s", op.request, e));
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        onDownloadFailed(retrofitToVungleException(e), op.request, null);
                    }
                }, new Runnable() {
                    @Override
                    public void run() {
                        onCriticalFail(VungleException.OUT_OF_MEMORY, op.request);
                    }
                });
            }

            @Override
            public void onResponse(Call<JsonObject> call, final Response<JsonObject> response) {
                VungleLogger.verbose(true, TAG, TT_DOWNLOAD_CONTEXT,
                        String.format("Request ad got response, request = %1$s, elapsed time = %2$dms", op.request, System.currentTimeMillis() - requestStartTime));
                //Lets do it in separate thread, since there are few IO calls involved.
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        /// The Ad Server has given us metadata for the advertisement, we now download the assets

                        /// Load the placement information from disk. If it cannot be found, an error has
                        /// occurred. Most likely this would be due to a cache clear or db dump.
                        Placement placement = repository.load(op.request.getPlacementId(), Placement.class).get();
                        if (placement == null) {
                            Log.e(TAG, "Placement metadata not found for requested advertisement.");
                            VungleLogger.error(
                                    FETCH_AD_METADATA_CONTEXT,
                                    "Placement metadata not found for requested advertisement." +
                                            " request = " + op.request);
                            onDownloadFailed(new VungleException(UNKNOWN_ERROR), op.request, null);
                            return;
                        }

                        if (!response.isSuccessful()) {
                            long retryAfterHeaderValue = vungleApiClient.getRetryAfterHeaderValue(response);
                            if (retryAfterHeaderValue > 0 && (placement.isAutoCached() || placement.isMultipleHBPEnabled())) {
                                loadEndlessIfNeeded(placement, op.size, retryAfterHeaderValue, false);
                                VungleLogger.error(
                                        FETCH_AD_METADATA_CONTEXT,
                                        "Response was not successful, retrying; request = " + op.request);
                                onDownloadFailed(new VungleException(SERVER_RETRY_ERROR), op.request, null);
                                return;
                            }

                            /// We can't do anything if the request failed.
                            Log.e(TAG, "Failed to retrieve advertisement information");
                            VungleLogger.error(
                                    FETCH_AD_METADATA_CONTEXT,
                                    String.format("Response was not successful, not retrying;" +
                                            "request = %1$s; responseCode = %2$s", op.request, response.code()));
                            onDownloadFailed(reposeCodeToVungleException(response.code()), op.request, null);
                            return;
                        }

                        JsonObject jsonObject = response.body();
                        Log.d(TAG, "Ads Response: " + jsonObject);
                        if (jsonObject != null && jsonObject.has("ads") && !jsonObject.get("ads").isJsonNull()) {
                            JsonArray ads = jsonObject.getAsJsonArray("ads");

                            if (ads == null || ads.size() == 0) {
                                // If there is no ad available, we need to inform the caller about the no serve response.
                                VungleLogger.error(
                                        FETCH_AD_METADATA_CONTEXT,
                                        "Response was successful, but no ads; request = " + op.request);
                                onDownloadFailed(new VungleException(NO_SERVE), op.request, null);
                                return;
                            }

                            JsonObject ad = ads.get(0).getAsJsonObject();
                            JsonObject adMarkup = ad.get("ad_markup").getAsJsonObject();
                            handleAdMetaData(op, requestStartTime, ad, placement, adMarkup);
                        } else {
                            VungleLogger.error(FETCH_AD_METADATA_CONTEXT,
                                    String.format("Response has no ads; placement = %1$s;" +
                                                    "op.request = %2$s; response = %3$s",
                                            placement, op.request, jsonObject));
                            onDownloadFailed(new VungleException(NO_SERVE), op.request, null);
                        }
                    }
                }, new Runnable() {
                    @Override
                    public void run() {
                        onCriticalFail(VungleException.OUT_OF_MEMORY, op.request);
                    }
                });
            }
        });
    }

    private void handleAdMetaData(Operation op,
                                  long requestStartTime,
                                  JsonObject advertisement,
                                  Placement placement,
                                  JsonObject adMarkup) {
        try {
            handleAdMetaData(op,
                    requestStartTime,
                    new Advertisement(advertisement),
                    placement,
                    adMarkup);
        } catch (IllegalArgumentException badAd) {
            /// Failed to parse the advertisement, which means there was no ad served.
            /// Check for a sleep code and schedule a download job.
            if (adMarkup.has("sleep")) {
                int sleep = adMarkup.get("sleep").getAsInt();

                /// Set this sleep time on the placement. We should not attempt to load
                /// more ads for this placement ID until the time has elapsed.
                placement.snooze(sleep);
                try {
                    VungleLogger.warn(FETCH_AD_METADATA_CONTEXT,
                            String.format("badAd - snoozed placement " +
                                    STRING_AND_OP_ID_FORMAT, placement, op.request));
                    repository.save(placement);
                } catch (DatabaseHelper.DBException ignored) {
                    VungleLogger.error(FETCH_AD_METADATA_CONTEXT,
                            String.format("badAd - can't save snoozed placement " +
                                    STRING_AND_OP_ID_FORMAT, placement, op.request));
                    onDownloadFailed(new VungleException(DB_ERROR), op.request, null);
                    return;
                }
                /// If the placement is auto-cached or HBP is enabled, schedule a download as soon as
                /// it wakes up. Note that since sleep is in seconds, we need to
                /// convert to milliseconds.
                loadEndlessIfNeeded(placement, op.size, sleep * 1000L, false);
            }
            VungleLogger.error(FETCH_AD_METADATA_CONTEXT,
                    String.format("badAd; can't proceed " +
                            STRING_AND_OP_ID_FORMAT, placement, op.request));

            onDownloadFailed(new VungleException(NO_SERVE), op.request, null);
        }
    }

    // Works with advertisement model to setup different responsibilities and handles any errors
    // that may occur.
    private void handleAdMetaData(Operation op,
                                  long requestStartTime,
                                  Advertisement advertisement,
                                  Placement placement,
                                  JsonObject adMarkup) throws IllegalArgumentException {
        final HeaderBiddingCallback bidTokenCallBack = runtimeValues.headerBiddingCallback.get();
        try {
            if (visionController.isEnabled()) {
                if (JsonUtil.hasNonNull(adMarkup, VisionController.DATA_SCIENCE_CACHE)) {
                    visionController.setDataScienceCache(adMarkup.get(VisionController.DATA_SCIENCE_CACHE).getAsString());
                } else {
                    visionController.setDataScienceCache(null);
                }
            }

            /*
             *  Validation for Ad having same Ad Id. (Not the real scenario, only happens when using mock response)
             *  This code failing some test cases if there is auto-cached Ad in the config response,
             *  because generally for test case we are using mock response which has same ID.
             *  So loadAd() is called two times one for autocache ad and one for loadAd called manually,
             * in this case for second loadAd() response is giving operation_canceled error callback.
             */
            Advertisement advertisementInDB = repository.load(advertisement.getId(), Advertisement.class).get();
            if (advertisementInDB != null) {
                int state = advertisementInDB.getState();
                if (state == Advertisement.NEW || state == Advertisement.READY || state == Advertisement.VIEWING) {
                    Log.d(TAG, "Operation Cancelled");
                    onDownloadFailed(new VungleException(OPERATION_CANCELED), op.request, null);
                    return;
                }
            }

            if (placement.isHeaderBidding() && bidTokenCallBack != null) {
                bidTokenCallBack.onBidTokenAvailable(op.request.getPlacementId(), advertisement.getBidToken());
            }

            //clear data if exists
            repository.deleteAdvertisement(advertisement.getId());

            Set<Map.Entry<String, String>> entries = advertisement.getDownloadableUrls().entrySet();

            final File destinationDir = getDestinationDir(advertisement);
            if (destinationDir == null || !destinationDir.isDirectory()) {
                VungleLogger.error(FETCH_AD_METADATA_CONTEXT,
                        String.format("Response was successful, but adv " +
                                        "directory is %1$s; op.request = %2$s, ad.getId() = %3$s",
                                destinationDir == null ? "null" : NOT_A_DIR,
                                op.request, advertisement.getId()));
                onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
                return;
            }

            for (Map.Entry<String, String> entry : entries) {
                if (isUrlValid(entry.getValue())) {
                    saveAsset(advertisement, destinationDir,
                            entry.getKey(), entry.getValue());
                } else {
                    VungleLogger.error(FETCH_AD_METADATA_CONTEXT,
                            String.format("Response was successful, but one of " +
                                            "downloadable urls is neither http nor https : " +
                                            "url = %1$s; op.request = %2$s, ad.getId() = %3$s",
                                    entry.getValue(), op.request, advertisement.getId()));
                    onDownloadFailed(new VungleException(AD_FAILED_TO_DOWNLOAD), op.request, advertisement.getId());
                    return;
                }
            }

            if (placement.getPlacementAdType() == Placement.TYPE_VUNGLE_BANNER &&
                    (advertisement.getAdType() != TYPE_VUNGLE_MRAID || !"banner".equals(advertisement.getTemplateType()))) {
                VungleLogger.error(FETCH_AD_METADATA_CONTEXT,
                        String.format("Response was successful, but placement is banner" +

                                        " while %1$s; op.request = %2$s, ad.getId() = %3$s",

                                advertisement.getAdType() != TYPE_VUNGLE_MRAID
                                        ? "ad type is not MRAID"
                                        : "advertisement template type is not banner",
                                op.request, advertisement.getId()));

                onDownloadFailed(new VungleException(NO_SERVE), op.request, advertisement.getId());
                return;
            }

            advertisement.getAdConfig().setAdSize(op.size);
            advertisement.setAdRequestStartTime(requestStartTime);
            advertisement.setAssetDownloadStartTime(System.currentTimeMillis());
            advertisement.setHeaderBidding(placement.isHeaderBidding());
            repository.saveAndApplyState(advertisement, op.request.getPlacementId(), NEW);

            int type = op.request.getType();
            if (type == AdRequest.Type.NORMAL || type == AdRequest.Type.SINGLE_HBP) {
                tryLoadNextInQueue(op.request);
                downloadAdAssets(op, advertisement);
            } else if (op.request.getType() == AdRequest.Type.NO_ASSETS) {
                if (isReadyForHBP(op, repository)) {
                    tryLoadNextInQueue(op.request);
                    onReady(op.request, placement, null);
                } else {
                    fetchAdMetadata(op, placement);
                }
            }
        } catch (DatabaseHelper.DBException e) {
            VungleLogger.error(FETCH_AD_METADATA_CONTEXT,
                    String.format("BadAd - DBException; can't proceed; placement = " +

                                    "%1$s; op.request = %2$s; exception = %3$s",
                            placement, op.request, e));
            onDownloadFailed(new VungleException(DB_ERROR), op.request, null);
        }
    }

    @Nullable
    File getDestinationDir(Advertisement advertisement) {
        return repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
    }

    void saveAsset(Advertisement advertisement, File destinationDir, String key, String url) throws DatabaseHelper.DBException {
        String path = destinationDir.getPath() + File.separator + key;

        @FileType int type = path.endsWith(KEY_POSTROLL) || path.endsWith(KEY_TEMPLATE)
                ? FileType.ZIP
                : FileType.ASSET;

        AdAsset adAsset = new AdAsset(advertisement.getId(), url, path);
        adAsset.status = Status.NEW;
        adAsset.fileType = type;
        try {
            repository.save(adAsset);
        } catch (DatabaseHelper.DBException e) {
            VungleLogger.error("AdLoader#saveAsset; loadAd sequence",
                    String.format("Can't save adAsset %1$s; exception = %2$s", adAsset, e));
            throw e;
        }
    }

    /**
     * @param op            Operation
     * @param advertisement Ad
     */
    private void downloadAdAssets(final Operation op, final Advertisement advertisement) {
        op.requests.clear();

        //validating URL , if one or more URLs is empty or not valid, SDK cannot determine how important that asset
        // is or what will be its side-effect. Its best to fail and drop the entire Ad-delivery.
        for (Map.Entry<String, String> entry : advertisement.getDownloadableUrls().entrySet()) {
            if (TextUtils.isEmpty(entry.getKey()) || TextUtils.isEmpty(entry.getValue()) ||
                    !URLUtil.isValidUrl(entry.getValue())) {
                VungleLogger.error(DOWNLOAD_AD_ASSETS_CONTEXT,
                        String.format("One or more ad asset URLs is empty or not valid;" +
                                OP_ID_AND_ADVERTISEMENT_FORMAT, op.request, advertisement));
                onDownloadFailed(new VungleException(AD_FAILED_TO_DOWNLOAD), op.request, null);
                Log.e(TAG, "Aborting, Failed to download Ad assets for: " + advertisement.getId());
                return;
            }
        }

        try {
            repository.save(advertisement);
        } catch (DatabaseHelper.DBException e) {
            VungleLogger.error(DOWNLOAD_AD_ASSETS_CONTEXT,
                    String.format("Cannot save advertisement op.request = %1$s; advertisement = %2$s",
                            op.request, advertisement));
            onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
            return;
        }

        /// Kick off the downloads in sequence. The downloader implementation will
        /// decide how to handle concurrent downloads, but this operation will
        /// notify the subscriber when all assets have been downloaded.

        List<AdAsset> assets = repository.loadAllAdAssets(advertisement.getId()).get();
        if (assets == null) {
            VungleLogger.error(DOWNLOAD_AD_ASSETS_CONTEXT,
                    String.format("Cannot load all ad assets; op.request = %1$s; advertisement = %2$s",
                            op.request, advertisement));
            onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
            return;
        }

        boolean foundVideoAsset = false;

        for (AdAsset asset : assets) {
            if (asset.status == Status.DOWNLOAD_SUCCESS) {
                if (fileIsValid(new File(asset.localPath), asset)) {
                    if (FileUtility.isVideoFile(asset.serverPath)) {
                        foundVideoAsset = true;
                        SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.ADS_CACHED)
                                .addData(SessionAttribute.EVENT_ID, advertisement.getId())
                                .build());
                    }
                    continue;
                }

                if (asset.fileType == FileType.ZIP_ASSET) {
                    VungleLogger.error(DOWNLOAD_AD_ASSETS_CONTEXT,
                            String.format("Cannot download ad assets - asset filetype is zip_asset;" +
                                    OP_ID_AND_ADVERTISEMENT_FORMAT, op.request, advertisement));
                    onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisement.getId());
                    return;
                }
            }

            if (asset.status == Status.PROCESSED && asset.fileType == FileType.ZIP) {
                continue;
            }

            if (TextUtils.isEmpty(asset.serverPath)) {
                VungleLogger.error(DOWNLOAD_AD_ASSETS_CONTEXT,
                        String.format("Cannot download ad assets - empty ;" +
                                OP_ID_AND_ADVERTISEMENT_FORMAT, op.request, advertisement));
                onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisement.getId());
                return;
            }

            DownloadRequest downloadRequest = getDownloadRequest(op.priority, asset, advertisement.getId());

            if (asset.status == Status.DOWNLOAD_RUNNING) {
                downloader.cancelAndAwait(downloadRequest, 1000);
                downloadRequest = getDownloadRequest(op.priority, asset, advertisement.getId());
            }

            Log.d(TAG, "Starting download for " + asset);
            asset.status = Status.DOWNLOAD_RUNNING;
            try {
                repository.save(asset);
            } catch (DatabaseHelper.DBException e) {
                VungleLogger.error(DOWNLOAD_AD_ASSETS_CONTEXT,
                        String.format("Can't save asset %1$s; exception = %2$s", asset, e));
                onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
                return;
            }
            op.requests.add(downloadRequest);

            if (FileUtility.isVideoFile(asset.serverPath)) {
                SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.ADS_CACHED)
                        .addData(SessionAttribute.EVENT_ID, advertisement.getId())
                        .addData(SessionAttribute.URL, asset.serverPath)
                        .build());
                foundVideoAsset = true;
            }
        }

        if (!foundVideoAsset) {
            SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.ADS_CACHED)
                    .addData(SessionAttribute.EVENT_ID, advertisement.getId())
                    .addData(SessionAttribute.VIDEO_CACHED, SessionConstants.NONE)
                    .build());
        }

        //All assets were downloaded already
        if (op.requests.size() == 0) {
            onAssetDownloadFinished(op, advertisement.getId(), Collections.<AssetDownloadListener.DownloadError>emptyList(), true);
            return;
        }

        VungleLogger.verbose(true, TAG, TT_DOWNLOAD_CONTEXT,
                String.format("Start to download assets,  request = %1$s at: %2$d", op.request, System.currentTimeMillis()));
        AssetDownloadListener downloadListener = getAssetDownloadListener(advertisement, op);
        for (DownloadRequest downloadRequest : op.requests) {
            downloader.download(downloadRequest, downloadListener);
        }
    }

    private DownloadRequest getDownloadRequest(@Priority int priority, AdAsset asset, String advertisementId) {
        AssetPriority assetPriority = getAssetPriority(priority, asset.localPath);
        return new DownloadRequest(
                Downloader.NetworkType.ANY,
                assetPriority,
                asset.serverPath, asset.localPath,
                false,
                asset.identifier, advertisementId);
    }

    private AssetPriority getAssetPriority(@Priority int priority, @NonNull String assetPath) {
        int firstPriority = Math.max(DownloadRequest.Priority.CRITICAL + 1, priority);
        int secondPriority = getAssetPriority(assetPath, adLoadOptimizationEnabled);
        return new AssetPriority(firstPriority, secondPriority);
    }

    // TODO: Move to Advertisement
    public static @DownloadRequest.Priority
    int getAssetPriority(@NonNull String path, boolean enableAdLoadOpt) {

        if (!enableAdLoadOpt) {
            return DownloadRequest.Priority.HIGHEST;
        }
        int secondPriority = DownloadRequest.Priority.HIGH;
        if (path.endsWith(KEY_TEMPLATE)) {
            secondPriority = HIGHEST;
        }
        return secondPriority;
    }

    @NonNull
    private AssetDownloadListener getAssetDownloadListener(
            final Advertisement advertisement,
            final Operation op
    ) {

        return new AssetDownloadListener() {

            AtomicLong downloadCount = new AtomicLong(op.requests.size());
            List<DownloadError> errors = Collections.synchronizedList(new ArrayList<DownloadError>());

            @Override
            public void onError(@NonNull final DownloadError downloadError,
                                final @Nullable DownloadRequest downloadRequest) {

                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        Log.e(TAG, "Download Failed");
                        if (downloadRequest != null) {
                            String id = downloadRequest.cookieString;
                            AdAsset asset = TextUtils.isEmpty(id) ? null
                                    : repository.load(id, AdAsset.class).get();

                            if (asset != null) {
                                errors.add(downloadError);
                                asset.status = Status.DOWNLOAD_FAILED;
                                try {
                                    repository.save(asset);
                                } catch (DatabaseHelper.DBException e) {
                                    errors.add(new DownloadError(
                                            -1,
                                            new VungleException(DB_ERROR),
                                            ErrorReason.INTERNAL_ERROR
                                    ));
                                }
                            } else {
                                errors.add(new DownloadError(
                                        -1,
                                        new IOException("Downloaded file not found!"),
                                        ErrorReason.REQUEST_ERROR
                                ));
                            }
                        } else {
                            errors.add(new DownloadError(-1,
                                    new RuntimeException("error in request"), ErrorReason.INTERNAL_ERROR));
                        }

                        if (downloadCount.decrementAndGet() <= 0) {
                            onAssetDownloadFinished(op, advertisement.getId(), errors, true);
                        }
                    }
                }, new Runnable() {
                    @Override
                    public void run() {
                        onCriticalFail(VungleException.OUT_OF_MEMORY, op.request);
                    }
                });
            }

            @Override
            public void onProgress(@NonNull Progress progress,
                                   @NonNull DownloadRequest downloadRequest) {

            }

            @Override
            public void onSuccess(@NonNull final File downloadedFile,
                                  @NonNull final DownloadRequest downloadRequest) {
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        if (!downloadedFile.exists()) {
                            VungleLogger.error(GET_ASSET_DOWNLOAD_LISTENER_CONTEXT,
                                    String.format("Downloaded file %1$s doesn't exist", downloadedFile.getPath()));
                            onError(new DownloadError(
                                            -1,
                                            new IOException("Downloaded file not found!"),
                                            ErrorReason.FILE_NOT_FOUND_ERROR
                                    ),
                                    downloadRequest);   //AdAsset table will be updated in onError callback
                            return;
                        }

                        String id = downloadRequest.cookieString;
                        AdAsset adAsset = id == null ? null : repository.load(id, AdAsset.class).get();
                        if (adAsset == null) {
                            VungleLogger.error(GET_ASSET_DOWNLOAD_LISTENER_CONTEXT,
                                    String.format("adAsset is null because %1$s, downloadRequest = %2$s",
                                            id == null ? "id is null" : "repository returned null",
                                            downloadRequest));
                            onError(new DownloadError(
                                            -1,
                                            new IOException("Downloaded file not found!"),
                                            ErrorReason.REQUEST_ERROR
                                    ),
                                    downloadRequest);   //AdAsset table will be updated in onError callback
                            return;
                        }

                        adAsset.fileType = isZip(downloadedFile) ? FileType.ZIP : FileType.ASSET;
                        adAsset.fileSize = downloadedFile.length();
                        adAsset.status = Status.DOWNLOAD_SUCCESS;
                        try {
                            repository.save(adAsset);
                        } catch (DatabaseHelper.DBException e) {
                            VungleLogger.error(GET_ASSET_DOWNLOAD_LISTENER_CONTEXT,
                                    String.format("Can't save adAsset %1$s; exception = %2$s", adAsset, e));
                            onError(new DownloadError(-1, new VungleException(DB_ERROR), ErrorReason.INTERNAL_ERROR), downloadRequest);
                            return;
                        }

                        if (isZip(downloadedFile)) {
                            // Inject OMSDK
                            injectOMIfNeeded(op, advertisement);

                            // onAdLoaded callback will be triggered when template is downloaded.
                            processTemplate(op, adAsset, advertisement);
                        }

                        if (downloadCount.decrementAndGet() <= 0) {
                            /// In the case of an MRAID ad, we also have to update the
                            /// cacheable_replacements map to point to the local files.N
                            /// Otherwise, the html will load them from the CDN and could
                            /// cause the user to experience lag.

                            boolean isNeedTriggerCallback = advertisement.isNativeTemplateType() || !(isAdLoadOptimizationEnabled(advertisement));
                            onAssetDownloadFinished(op, advertisement.getId(), errors, isNeedTriggerCallback);
                        }
                    }
                }, new Runnable() {
                    @Override
                    public void run() {
                        onCriticalFail(VungleException.OUT_OF_MEMORY, op.request);
                    }
                });
            }
        };
    }

    private boolean isZip(File downloadedFile) {
        return downloadedFile.getName().equals(KEY_POSTROLL) || downloadedFile.getName().equals(KEY_TEMPLATE);
    }

    private boolean isUrlValid(String url) {
        return !TextUtils.isEmpty(url) && (URLUtil.isHttpsUrl(url) || URLUtil.isHttpUrl(url));
    }

    private void processTemplate(@NonNull final Operation op,
                                 @NonNull final AdAsset asset,
                                 @NonNull final Advertisement advertisement) {

        if (asset.status != DOWNLOAD_SUCCESS) {
            onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisement.getId());
            return;
        }

        File f = new File(asset.localPath);
        if (!fileIsValid(f, asset)) {
            VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                    String.format("Assets file not valid %1$s; asset = %2$s," +
                                    PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_3_4,
                            f.getPath(), asset.toString(), op.request, advertisement));
            onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisement.getId());
            return;
        }

        if (asset.fileType == FileType.ZIP) {
            long unzipStartTime = System.currentTimeMillis();
            VungleLogger.verbose(true, TAG, TT_DOWNLOAD_CONTEXT,
                    String.format("Start to unzip assets, request  = %1$s, at: %2$d", op.request, unzipStartTime));
            try {
                List<AdAsset> assets = repository.loadAllAdAssets(advertisement.getId()).get();
                unzipFile(advertisement, asset, f, assets);
            } catch (IOException e) {
                // Remove from cache as zip is malformed
                VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                        String.format("Unzip failed %1$s; asset = %2$s," +
                                        PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_3_4,
                                f.getPath(), asset.toString(), op.request, advertisement));
                downloader.dropCache(asset.serverPath);
                onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisement.getId());
                return;
            } catch (DatabaseHelper.DBException e) {
                VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                        String.format("Issue(s) with database: exception = %1$s; asset = %2$s," +
                                        PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_3_4,
                                e, asset.toString(), op.request, advertisement));
                onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
                return;
            }
            VungleLogger.verbose(true, TAG, TT_DOWNLOAD_CONTEXT,
                    String.format("Unzip assets completed, request  = %1$s, elapsed time = %2$dms", op.request, System.currentTimeMillis() - unzipStartTime));
        }

        if (isAdLoadOptimizationEnabled(advertisement)) {
            VungleLogger.verbose(true, TAG, TT_DOWNLOAD_CONTEXT,
                    String.format("Ad ready to play, request  = %1$s, elapsed time = %2$dms", op.request, System.currentTimeMillis() - advertisement.adRequestStartTime));
            onDownloadCompleted(op.request, advertisement.getId());
        }
    }

    private boolean injectOMIfNeeded(@NonNull final Operation op,
                                  @NonNull final Advertisement advertisement) {

        if (advertisement.getOmEnabled()) {
            try {
                File destinationDir = getDestinationDir(advertisement);
                if (destinationDir == null || !destinationDir.isDirectory()) {
                    VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                            String.format("Mraid ad; bad destinationDir - %1$s" +
                                            PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_2_3,
                                    destinationDir == null ? "null" : NOT_A_DIR, op.request, advertisement));
                    onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
                    return false;
                }
                List<File> injected = omInjector.injectJsFiles(destinationDir);
                for (File file : injected) {
                    AdAsset asset = new AdAsset(advertisement.getId(), null, file.getPath());
                    asset.fileSize = file.length();
                    asset.fileType = FileType.ASSET;
                    asset.status = Status.DOWNLOAD_SUCCESS;
                    repository.save(asset);
                }
            } catch (IOException e) {
                onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisement.getId());
                return false;
            } catch (DatabaseHelper.DBException e) {
                onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
                return false;
            }
        }
        return true;
    }

    private void onAssetDownloadFinished(@NonNull final Operation op,
                                         @NonNull final String advertisementId,
                                         @NonNull List<AssetDownloadListener.DownloadError> errors,
                                         boolean triggerCallback) {

        VungleLogger.verbose(true, TAG, TT_DOWNLOAD_CONTEXT,
                String.format("Assets download completed, request  = %1$s, at: %2$d", op.request, System.currentTimeMillis()));
        if (errors.isEmpty()) {
            // We need to reload advertisement from db because advertisement may be outdated.
            // The advertisement state could be changed to "READY" in onDownloadCompleted().
            final Advertisement advertisement = repository.load(advertisementId, Advertisement.class).get();
            if (advertisement == null) {
                VungleLogger.error(DOWNLOAD_AD_CALLBACK_ON_DOWNLOAD_COMPLETED_CONTEXT,
                        String.format("advertisement is null: request = %1$s; advertisementId = %2$s",
                                op.request, advertisementId));
                onDownloadFailed(new VungleException(AD_FAILED_TO_DOWNLOAD), op.request, advertisementId);
                return;
            }
            List<AdAsset> assets = repository.loadAllAdAssets(advertisementId).get();

            if (assets == null || assets.size() == 0) {
                VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                        String.format("Assets are %1$s; request = %2$s; advertisement = %3$s",
                                assets == null ? "null" : "empty", op.request, advertisementId));
                if (triggerCallback) {
                    onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisementId);
                }
                return;
            }

            for (AdAsset asset : assets) {
                if (asset.status == Status.DOWNLOAD_SUCCESS) {
                    File f = new File(asset.localPath);

                    if (!fileIsValid(f, asset)) {
                        VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                                String.format("Assets file not valid %1$s; asset = %2$s," +
                                                PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_3_4,
                                        f.getPath(), asset.toString(), op.request, advertisement));
                        if (triggerCallback) {
                            onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisement.getId());
                        }
                        return;
                    }
                } else if (asset.fileType == FileType.ZIP && asset.status != PROCESSED) {
                    VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                            String.format("Zip asset left unprocessed asset = %1$s," +
                                            PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_2_3,
                                    asset.toString(), op.request, advertisement));
                    onDownloadFailed(new VungleException(ASSET_DOWNLOAD_ERROR), op.request, advertisement.getId());
                    return;
                }
            }

            if (advertisement.getAdType() == TYPE_VUNGLE_MRAID) {
                File destinationDir = getDestinationDir(advertisement);
                if (destinationDir == null || !destinationDir.isDirectory()) {
                    VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                            String.format("Mraid ad; bad destinationDir - %1$s" +
                                            PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_2_3,
                                    destinationDir == null ? "null" : NOT_A_DIR, op.request, advertisement));
                    if (triggerCallback) {
                        onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
                    }
                    return;
                }

                Log.d(TAG, "saving MRAID for " + advertisement.getId());
                advertisement.setMraidAssetDir(destinationDir);
                try {
                    repository.save(advertisement);
                } catch (DatabaseHelper.DBException e) {
                    VungleLogger.error(ON_ASSET_DOWNLOAD_FINISHED_CONTEXT,
                            String.format("Issue(s) with database: can't save advertisement;" +
                                            "exception = %1$s; request = %2$s; advertisement = %3$s",
                                    e, op.request, advertisement));
                    if (triggerCallback) {
                        onDownloadFailed(new VungleException(DB_ERROR), op.request, advertisement.getId());
                    }
                    return;
                }
            }
            if (triggerCallback) {
                onDownloadCompleted(op.request, advertisement.getId());
            }

        } else {
            VungleException endError = null;

            for (AssetDownloadListener.DownloadError downloadError : errors) {
                VungleException error;

                if (VungleException.getExceptionCode(downloadError.cause) == DB_ERROR) {
                    endError = new VungleException(DB_ERROR);
                    break;
                }

                if (recoverableServerCode(downloadError.serverCode) && downloadError.reason == ErrorReason.REQUEST_ERROR) {
                    error = new VungleException(ASSET_DOWNLOAD_RECOVERABLE);
                } else if (downloadError.reason == ErrorReason.CONNECTION_ERROR) {
                    error = new VungleException(ASSET_DOWNLOAD_RECOVERABLE);
                } else {
                    error = new VungleException(ASSET_DOWNLOAD_ERROR);
                }

                endError = error;

                if (endError.getExceptionCode() == ASSET_DOWNLOAD_ERROR)
                    break;
            }

            final VungleException exception = endError;

            if (triggerCallback) {
                onDownloadFailed(exception, op.request, advertisementId);
            }

        }
    }

    @WorkerThread
    public void onDownloadCompleted(@NonNull AdRequest request, @NonNull final String advertisementId) {
        Log.d(TAG, "download completed " + request);

        final Placement placement = repository.load(request.getPlacementId(), Placement.class).get();
        if (placement == null) {
            VungleLogger.error(DOWNLOAD_AD_CALLBACK_ON_DOWNLOAD_COMPLETED_CONTEXT,

                    String.format("loaded placement is null: request = %1$s; advertisementId = %2$s",
                            request, advertisementId));
            onDownloadFailed(new VungleException(PLACEMENT_NOT_FOUND), request, advertisementId);
            return;
        }

        final Advertisement advertisement = TextUtils.isEmpty(advertisementId)
                ? null
                : repository.load(advertisementId, Advertisement.class).get();
        if (advertisement == null) {
            VungleLogger.error(DOWNLOAD_AD_CALLBACK_ON_DOWNLOAD_COMPLETED_CONTEXT,
                    String.format("advertisement is null: request = %1$s; advertisementId = %2$s",
                            request, advertisementId));
            onDownloadFailed(new VungleException(AD_FAILED_TO_DOWNLOAD), request, advertisementId);
            return;
        }

        advertisement.setFinishedDownloadingTime(System.currentTimeMillis());

        try {
            repository.saveAndApplyState(advertisement, request.getPlacementId(), READY);
        } catch (DatabaseHelper.DBException e) {
            VungleLogger.error(DOWNLOAD_AD_CALLBACK_ON_DOWNLOAD_COMPLETED_CONTEXT,
                    String.format("Can't save/apply state READY: exception = %1$s;" +
                                    PLACEMENT_ID_AND_ADVERTISEMENT_FORMAT_2_3,
                            e, request, advertisement));
            onDownloadFailed(new VungleException(DB_ERROR), request, advertisementId);
            return;
        }
        onReady(request, placement, advertisement);
    }

    @WorkerThread
    public void onReady(@NonNull AdRequest request, @NonNull final Placement placement, @Nullable final Advertisement advertisement) {
        setLoading(request, false);

        //for ads that are already cached and SDK receives loadAd it should also fire callback with bid token
        HeaderBiddingCallback headerBiddingCallback = runtimeValues.headerBiddingCallback.get();
        if (advertisement != null && placement.isHeaderBidding() && headerBiddingCallback != null) {
            headerBiddingCallback.adAvailableForBidToken(request.getPlacementId(), advertisement.getBidToken());
        }

        Log.i(TAG, "found already cached valid adv, calling onAdLoad callback for request " + request);
        /// We have assets for this placement. Verify that the advertisement link is still valid.
        final InitCallback initCallback = runtimeValues.initCallback.get();
        int type = request.getType();
        if (placement.isAutoCached() && initCallback != null && (type == AdRequest.Type.SINGLE_HBP || type == AdRequest.Type.NORMAL)) {
            initCallback.onAutoCacheAdAvailable(request.getPlacementId());
        }

        Operation operation = loadOperations.remove(request);
        String advertisementId = advertisement != null ? advertisement.getId() : null;

        if (operation != null) {
            placement.setAdSize(operation.size);

            try {
                repository.save(placement);
            } catch (DatabaseHelper.DBException e) {
                VungleLogger.error("AdLoader#DownloadAdCallback#onReady; loadAd sequence",
                        String.format("Can't save placement: exception = %1$s;" +
                                        "placement = %2$s; advertisement = %3$s",
                                e, placement, advertisement));
                onDownloadFailed(new VungleException(DB_ERROR), request, advertisementId);
                return;
            }

            //todo: temp logs
            Log.i(TAG, "loading took " + (System.currentTimeMillis() - request.timeStamp.get()) + "ms for:" + request);

            if (request.getIsExplicit()) {
                SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.LOAD_AD_END)
                        .addData(SessionAttribute.SUCCESS, true)
                        .addData(SessionAttribute.PLACEMENT_ID, placement.getId())
                        .build());
            }

            for (LoadAdCallback loadAdCallback : operation.loadAdCallbacks) {
                if(loadAdCallback instanceof LoadNativeAdCallbackWrapper) {
                    LoadNativeAdCallbackWrapper nativeAdCallback = (LoadNativeAdCallbackWrapper)loadAdCallback;
                    nativeAdCallback.onAdLoad(advertisement);
                } else {
                    loadAdCallback.onAdLoad(request.getPlacementId());
                }
            }

            SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.AD_AVAILABLE)
                    .addData(SessionAttribute.EVENT_ID, advertisement != null ? advertisement.getId() : null)
                    .addData(SessionAttribute.PLACEMENT_ID, request.getPlacementId())
                    .build());

            if (request.getIsExplicit()) {
                List<String> notifications =
                        advertisement != null ? advertisement.getWinNotifications() : new ArrayList<String>();
                sendWinNotification(operation, notifications);
            }
        }
    }

    @WorkerThread
    public void onDownloadFailed(@NonNull VungleException exception, @NonNull AdRequest request, @Nullable final String advertisementId) {
        @Nullable final Operation op = loadOperations.remove(request);
        tryLoadNextInQueue(request);

        Placement placement = repository.load(request.getPlacementId(), Placement.class).get();
        Advertisement advertisement = advertisementId == null
                ? null
                : repository.load(advertisementId, Advertisement.class).get();

        if (placement == null) {
            if (advertisement != null) {
                try {
                    repository.saveAndApplyState(advertisement, request.getPlacementId(), ERROR);
                } catch (DatabaseHelper.DBException ignored) {
                    exception = new VungleException(DB_ERROR);
                }
            }

            if (op != null) {
                for (LoadAdCallback loadAdCallback : op.loadAdCallbacks) {
                    loadAdCallback.onError(request.getPlacementId(), exception);
                }
            }
            setLoading(request, false);
            return;
        }

        if (placement != null && request.getIsExplicit()) {
            SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.LOAD_AD_END)
                    .addData(SessionAttribute.SUCCESS, false)
                    .addData(SessionAttribute.PLACEMENT_ID, placement.getId())
                    .build());
        }

        boolean canRetry = false;
        boolean stopInfinite = false;
        @Advertisement.State int state = ERROR;

        switch (exception.getExceptionCode()) {
            case SERVER_TEMPORARY_UNAVAILABLE:
            case NETWORK_ERROR:
                canRetry = true;
                break;
            case ASSET_DOWNLOAD_RECOVERABLE:
                if (advertisement != null) {
                    canRetry = true;
                    state = NEW;
                }
                break;
            case OPERATION_CANCELED:
            case NO_SERVE:
            case SERVER_RETRY_ERROR:
            case MISSING_HBP_EVENT_ID:
                stopInfinite = true;
                break;
            default:
                break;
        }

        if (op == null || op.logError) {
            Log.e(TAG, "Failed to load Ad/Assets for " + request + ". Cause : ", exception);
        }

        setLoading(request, false);

        if (op != null) {
            try {
                if (op.policy == ReschedulePolicy.EXPONENTIAL) {
                    if (op.retry < op.retryLimit && canRetry) {
                        if (advertisement != null) {
                            repository.saveAndApplyState(advertisement, request.getPlacementId(), state);
                        }
                        load(op.delay(op.retryDelay).retryDelay(op.retryDelay * EXPONENTIAL_RATE).retry(op.retry + 1));
                        return;
                    }
                } else if (op.policy == ReschedulePolicy.EXPONENTIAL_ENDLESS_AD && !stopInfinite) {
                    int retry = op.retry;
                    if (retry < op.retryLimit && canRetry) {
                        retry++;
                    } else {
                        retry = 0;//forever request /ads until stop
                        state = ERROR;
                    }
                    if (advertisement != null) {
                        repository.saveAndApplyState(advertisement, request.getPlacementId(), state);
                    }
                    load(op.delay(op.retryDelay).retryDelay(op.retryDelay * EXPONENTIAL_RATE).retry(retry));
                    return;
                }

                if (advertisement != null) {
                    repository.saveAndApplyState(advertisement, request.getPlacementId(), ERROR);
                }

            } catch (DatabaseHelper.DBException e) {
                exception = new VungleException(DB_ERROR);
            }

            for (LoadAdCallback loadAdCallback : op.loadAdCallbacks) {
                loadAdCallback.onError(request.getPlacementId(), exception);
            }
        }
    }

    //call from any thread, cleanup what we can
    public void onCriticalFail(@VungleException.ExceptionCode int code, @NonNull AdRequest request) {
        @Nullable final Operation op = loadOperations.remove(request);
        onError(op, code);
    }

    //helper functions
    public void load(AdRequest request, AdConfig adConfig, LoadAdCallback listener) {
        load(new Operation(
                request,
                adConfig.getAdSize(),
                0,
                RETRY_DELAY,
                RETRY_COUNT,
                ReschedulePolicy.EXPONENTIAL,
                0,
                true,
                Priority.HIGHEST,
                listener
        ));
    }

    // isExplicit should be set to false in the case that load operation had failed. In these cases,
    // this method is attempting to queue an auto-cached ad, which is not representative of winning
    // a bid.
    public void loadEndlessIfNeeded(@NonNull final Placement placement,
                                    @NonNull AdSize size,
                                    final long delay,
                                    boolean isExplicit) {
        if (placement.isMultipleHBPEnabled() &&
                placement.getPlacementAdType() == TYPE_VUNGLE_BANNER &&
                !AdSize.isBannerAdSize(size)) {
            size = placement.getRecommendedAdSize();
        }
        if (isSizeInvalid(placement, size))
            return;

        int priority = placement.getAutoCachePriority();
        final VungleSettings settings = runtimeValues.settings.get();
        if (settings != null && placement.getId().equals(settings.getPriorityPlacement())) {
            priority = Priority.HIGHEST;
        }

        AdRequest request = null;
        if (placement.isMultipleHBPEnabled() && !placement.isSingleHBPEnabled()) {
            request = new AdRequest(placement.getId(), AdRequest.Type.NO_ASSETS, placement.getMaxHbCache(), isExplicit);
        }
        // Expect to treat a max_cache of 1 on a multiple header bidding placement with the same
        // treatment as an auto-cached ad.
        else if (placement.isSingleHBPEnabled()) {
            request = new AdRequest(placement.getId(), AdRequest.Type.SINGLE_HBP, 1L, isExplicit);
        } else if (placement.isAutoCached()) {
            request = new AdRequest(placement.getId(), AdRequest.Type.NORMAL, 1L, isExplicit);
        }
        if (request != null) {
            load(new Operation(
                    request,
                    size,
                    delay,
                    RETRY_DELAY,
                    RETRY_COUNT,
                    ReschedulePolicy.EXPONENTIAL_ENDLESS_AD,
                    0,
                    false,
                    priority
            ));
        }
    }

    private void unzipFile(Advertisement advertisement,
                           AdAsset zipAsset,
                           @NonNull final File downloadedFile,
                           List<AdAsset> allAssets) throws IOException, DatabaseHelper.DBException {
        final List<String> existingPaths = new ArrayList<>();
        for (AdAsset asset : allAssets) {
            if (asset.fileType == FileType.ASSET) {
                existingPaths.add(asset.localPath);
            }
        }

        File destinationDir = getDestinationDir(advertisement);
        if (destinationDir == null || !destinationDir.isDirectory()) {
            VungleLogger.error("AdLoader#unzipFile; loadAd sequence",
                    String.format("Can't unzip file: destination dir is %1$s; advertisement = %2$s",
                            destinationDir == null ? "null" : NOT_A_DIR, advertisement));
            throw new IOException("Unable to access Destination Directory");
        }

        List<File> extractedFiles = UnzipUtility.unzip(downloadedFile.getPath(), destinationDir.getPath(), new UnzipUtility.Filter() {
            @Override
            public boolean matches(String extractPath) {

                File toExtract = new File(extractPath);

                for (String existing : existingPaths) {
                    File existingFile = new File(existing);

                    if (existingFile.equals(toExtract))
                        return false;

                    if (toExtract.getPath().startsWith(existingFile.getPath() + File.separator))
                        return false;
                }

                return true;
            }
        });

        if (downloadedFile.getName().equals(KEY_TEMPLATE)) {
            //  Updating mraid.js
            // Find the mraid.js file and append the MRAID code. Why this is not done by the server is
            // confusing to me but I assume there must be some historical context to it.
            File mraidJS = new File(destinationDir.getPath() + File.separator + "mraid.js");
            if (mraidJS.exists()) {        //mraid.js exits
                PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(mraidJS, true)));
                HackMraid.apply(out);
                out.close();
            }
        }

        for (File file : extractedFiles) {
            AdAsset extractedAsset = new AdAsset(advertisement.getId(), null, file.getPath());
            extractedAsset.fileSize = file.length();
            extractedAsset.fileType = FileType.ZIP_ASSET;
            extractedAsset.parentId = zipAsset.identifier;
            extractedAsset.status = Status.DOWNLOAD_SUCCESS;
            repository.save(extractedAsset);
        }

        Log.d(TAG, "Uzipped " + destinationDir);
        FileUtility.printDirectoryTree(destinationDir);

        zipAsset.status = Status.PROCESSED;
        repository.save(zipAsset, new Repository.SaveCallback() {
            @Override
            public void onSaved() {
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            FileUtility.delete(downloadedFile);
                        } catch (IOException e) {
                            //ignored
                            Log.e(TAG, "Error on deleting zip assets archive", e);
                        }
                    }
                });
            }

            @Override
            public void onError(Exception ignored) {
            }
        });
    }

    boolean hasAssetsFor(Advertisement advertisement) throws IllegalStateException {
        if (advertisement == null) {
            return false;
        }
        List<AdAsset> adAssets = repository.loadAllAdAssets(advertisement.getId()).get();

        if (adAssets == null || adAssets.size() == 0) {
            return false;
        }


        boolean isAllAssetAvailable = true;

        for (AdAsset adAsset : adAssets) {

            if (adAsset.fileType == FileType.ZIP) {
                if (adAsset.status == PROCESSED) {
                    continue;
                }

                isAllAssetAvailable = false;
                break;
            } else {
                // if ad load optimization is enabled, no need to check for those assets which have remote url.
                if (isUrlValid(adAsset.serverPath) && isAdLoadOptimizationEnabled(advertisement)) {
                    continue;
                }
                // unzipped files and omsdk injected files
                if (adAsset.status != DOWNLOAD_SUCCESS) {
                    isAllAssetAvailable = false;
                    break;
                }

                File file = new File(adAsset.localPath);
                if (!fileIsValid(file, adAsset)) {
                    isAllAssetAvailable = false;
                    break;
                }
            }
        }

        return isAllAssetAvailable;
    }

    public boolean isAdLoadOptimizationEnabled(Advertisement advertisement) {
        return (adLoadOptimizationEnabled && advertisement != null &&
                advertisement.getAdType() == TYPE_VUNGLE_MRAID);
    }

    void sendWinNotification(Operation op, List<String> notifications) {
        if (notifications.isEmpty()) {
            return;
        }

        JobRunner jobRunner = this.jobRunnerRef.get();
        if (jobRunner == null) {
            VungleLogger.error("AdLoader#load; loadAd sequence",
                    String.format("Cannot load operation %s; job runner is null", op));
            onError(op, VUNGLE_NOT_INTIALIZED);
            return;
        }
        new JobDelegateAnalytics(jobRunner).ping(notifications.toArray(new String[0]));
    }

    private boolean fileIsValid(File file, AdAsset adAsset) {
        return file.exists() && file.length() == adAsset.fileSize;
    }

    @VisibleForTesting
    Collection<Operation> getPendingOperations() {
        return pendingOperations.values();
    }

    @VisibleForTesting
    Collection<Operation> getRunningOperations() {
        return loadOperations.values();
    }

    void setAdLoadOptimizationEnabled(boolean enabled) {
        adLoadOptimizationEnabled = enabled;
    }

    public void dropCache(String advertisementId) {
        List<AdAsset> adAssets = repository.loadAllAdAssets(advertisementId).get();
        if (adAssets == null) {
            Log.w(TAG, "No assets found in ad cache to cleanup");
            return;
        }
        HashSet<String> urls = new HashSet<>();
        for (AdAsset asset : adAssets) {
            urls.add(asset.serverPath);
        }
        Advertisement advertisement = repository.load(advertisementId, Advertisement.class).get();
        if (advertisement != null) {
            urls.addAll(advertisement.getDownloadableUrls().values());
        }

        for (String url : urls) {
            downloader.dropCache(url);
        }
    }
}
