package com.vungle.warren;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.model.admarkup.AdMarkup;

import java.io.Serializable;
import java.util.concurrent.atomic.AtomicLong;

public final class AdRequest implements Serializable {

    // Represents if the AdRequest is created when Vungle's API functionality is directly accessed
    // through the SDK. If false, it signifies that AdRequest is being used for mediation purposes.
    private final boolean isExplicit;

    @IntDef(value = {Type.NORMAL, Type.NO_ASSETS, Type.SINGLE_HBP})
    public @interface Type {
        int NORMAL = 0;
        // Used for delayed caching where ad assets are downloaded on demand.
        int NO_ASSETS = 1;
        // In the case that there's a max_hb_cache of 1, the behavior resembles ad auto-cached
        // behavior.
        int SINGLE_HBP = 2;
    }

    @NonNull
    private final String placementId;

    @Nullable
    private final AdMarkup adMarkup;

    private final @Type
    int type;

    private final long adCount;

    //todo: for temporary logs
    public AtomicLong timeStamp = new AtomicLong(0L);

    public AdRequest(@NonNull String placementId, @Type int type, long adCount, boolean isExplicit) {
        this.placementId = placementId;
        this.adMarkup = null;
        this.type = type;
        this.adCount = adCount;
        this.isExplicit = isExplicit;
    }

    public AdRequest(@NonNull String placementId, @Nullable AdMarkup adMarkup, boolean isExplicit) {
        this.placementId = placementId;
        this.adMarkup = adMarkup;
        this.type = Type.NORMAL;
        this.adCount = 1L;
        this.isExplicit = isExplicit;
    }

    public AdRequest(@NonNull String placementId, boolean isExplicit) {
        this(placementId, null, isExplicit);
    }

    @NonNull
    public String getPlacementId() {
        return placementId;
    }

    @Nullable
    public String getEventId() {
        return adMarkup == null ? null : adMarkup.getEventId();
    }

    @Nullable
    public String[] getImpression() {
        AdMarkup adMarkup = getAdMarkup();
        return adMarkup != null ? getAdMarkup().getImpressions() : null;
    }

    @Type
    public int getType() {
        return type;
    }

    public long getAdCount() {
        return adCount;
    }


    @Nullable
    public AdMarkup getAdMarkup() {
        return adMarkup;
    }

    @Nullable
    public boolean getIsExplicit() {
        return isExplicit;
    }

    @Override
    public String toString() {
        return "AdRequest{" +
                "placementId='" + placementId + '\'' +
                ", adMarkup=" + adMarkup +
                ", type=" + type +
                ", adCount=" + adCount +
                ", isExplicit=" + isExplicit +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        AdRequest request = (AdRequest) o;

        if (type != request.type) return false;
        if (!placementId.equals(request.placementId)) return false;
        return adMarkup != null ? adMarkup.equals(request.adMarkup) : request.adMarkup == null;
    }

    @Override
    public int hashCode() {
        int result = placementId.hashCode();
        result = 31 * result + (adMarkup != null ? adMarkup.hashCode() : 0);
        result = 31 * result + type;
        return result;
    }
}
