package com.vungle.warren;

import android.annotation.SuppressLint;
import android.app.Dialog;
import android.content.Context;
import android.os.AsyncTask;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.analytics.AdAnalytics;
import com.vungle.warren.analytics.JobDelegateAnalytics;
import com.vungle.warren.downloader.DownloadRequest;
import com.vungle.warren.downloader.Downloader;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.AdAsset;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.omsdk.OMTracker;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.ui.CloseDelegate;
import com.vungle.warren.ui.JavascriptBridge;
import com.vungle.warren.ui.OrientationDelegate;
import com.vungle.warren.ui.contract.NativeAdContract;
import com.vungle.warren.ui.contract.WebAdContract;
import com.vungle.warren.ui.presenter.LocalAdPresenter;
import com.vungle.warren.ui.presenter.MRAIDAdPresenter;
import com.vungle.warren.ui.presenter.NativeAdPresenter;
import com.vungle.warren.ui.state.OptionsState;
import com.vungle.warren.ui.view.FullAdWidget;
import com.vungle.warren.ui.view.LocalAdView;
import com.vungle.warren.ui.view.MRAIDAdView;
import com.vungle.warren.ui.view.NativeAdView;
import com.vungle.warren.ui.view.VungleWebClient;
import com.vungle.warren.utility.Constants;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.HandlerScheduler;

import java.io.File;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicReference;

import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.error.VungleException.INVALID_SIZE;
import static com.vungle.warren.error.VungleException.MISSING_HBP_EVENT_ID;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;
import static com.vungle.warren.ui.contract.AdContract.AdView;
import static com.vungle.warren.ui.contract.AdContract.AdvertisementPresenter;

/**
 * Created by zioye on 1/16/2018.
 */
public class AdvertisementPresentationFactory implements PresentationFactory {

    private static final String TAG = AdvertisementPresentationFactory.class.getSimpleName();

    private static final String EXTRA_ADVERTISEMENT = "ADV_FACTORY_ADVERTISEMENT";

    private final JobRunner jobRunner;
    private VungleApiClient apiClient;
    private BaseTask task;
    private Repository repository;
    private VungleStaticApi vungleStaticApi;
    private Advertisement currentAdvertisement;
    private final AdLoader adLoader;
    private final OMTracker.Factory omTrackerFactory;
    private final ExecutorService taskExecutor;

    AdvertisementPresentationFactory(@NonNull AdLoader adLoader,
                                     @NonNull VungleStaticApi vungleStaticApi,
                                     @NonNull Repository repository,
                                     @NonNull VungleApiClient vungleApiClient,
                                     @NonNull JobRunner jobRunner,
                                     @NonNull OMTracker.Factory omTrackerFactory,
                                     @NonNull ExecutorService taskExecutor) {
        this.vungleStaticApi = vungleStaticApi;
        this.repository = repository;
        this.apiClient = vungleApiClient;
        this.jobRunner = jobRunner;
        this.adLoader = adLoader;
        this.omTrackerFactory = omTrackerFactory;
        this.taskExecutor = taskExecutor;
    }

    /**
     * Factory method for creating an {@link AdvertisementPresenter}. Using the placement ID and the
     * persistor and designer objects, it will create the {@link Advertisement} from the data on disk
     * and determine which renderer is most appropriate for this placement.
     *
     * @param context      {@link Context} in which {@link View} or {@link Dialog} can be created
     * @param request
     * @param fullAdWidget fullscreen ad widget
     */
    @Override
    public void getFullScreenPresentation(@NonNull final Context context,
                                          @NonNull final AdRequest request,
                                          @NonNull final FullAdWidget fullAdWidget,
                                          @Nullable final OptionsState optionsState,
                                          @NonNull final CloseDelegate closeDelegate,
                                          @NonNull final OrientationDelegate orientationDelegate,
                                          @Nullable Bundle savedState,
                                          @NonNull final FullScreenCallback fullscreenCallback) {
        /// Extract placement and currentAdvertisement information, validate current SDK state.

        cancelTask();
        task = new FullScreenPresentationTask(
                context,
                adLoader,
                request,
                repository,
                vungleStaticApi,
                jobRunner,
                apiClient,
                fullAdWidget,
                optionsState,
                orientationDelegate,
                closeDelegate,
                fullscreenCallback,
                onModelLoadListener,
                savedState,
                omTrackerFactory
        );

        task.executeOnExecutor(taskExecutor);
    }

    private void cancelTask() {
        if (task != null) {
            task.cancel(true);
            task.clear();
        }
    }

    /**
     * Factory method for creating an {@link AdvertisementPresenter}. Using the placement ID and the
     * persistor and designer objects, it will create the {@link Advertisement} from the data on disk
     * and determine which renderer is most appropriate for this placement.
     */
    @Override
    public void getBannerViewPresentation(Context context,
                                          @NonNull final AdRequest request,
                                          @Nullable final AdConfig adConfig,
                                          @NonNull final CloseDelegate closeDelegate,
                                          @NonNull final ViewCallback viewCallback) {

        cancelTask();
        task = new BannerViewPresentationTask(
                context,
                request,
                adConfig,
                adLoader,
                repository,
                vungleStaticApi,
                jobRunner,
                viewCallback,
                null,
                onModelLoadListener,
                apiClient,
                omTrackerFactory
        );

        task.executeOnExecutor(taskExecutor);
    }

    /**
     * Factory method for creating an {@link AdvertisementPresenter}. Using the placement ID and the
     * persistor and designer objects, it will create the {@link Advertisement} from the data on disk
     * and determine which renderer is most appropriate for this placement.
     */
    @Override
    public void getNativeViewPresentation(@NonNull final Context context,
                                          @NonNull final NativeAdLayout nativeAdLayout,
                                          @NonNull final AdRequest request,
                                          @Nullable final AdConfig adConfig,
                                          @NonNull final NativeViewCallback viewCallback) {

        cancelTask();
        task = new NativeViewPresentationTask(
                context,
                nativeAdLayout,
                request,
                adConfig,
                adLoader,
                repository,
                vungleStaticApi,
                jobRunner,
                viewCallback,
                null,
                onModelLoadListener
        );

        task.executeOnExecutor(taskExecutor);
    }

    @Override
    public void saveState(Bundle bundle) {
        bundle.putString(EXTRA_ADVERTISEMENT, currentAdvertisement == null ? null : currentAdvertisement.getId());
    }

    @Override
    public void destroy() {
        cancelTask();
    }

    private static class PresentationResultHolder {
        private AdView adView;
        private AdvertisementPresenter advertisementPresenter;
        private VungleException exception;
        private VungleWebClient webClient;

        PresentationResultHolder(VungleException exception) {
            this.exception = exception;
        }

        PresentationResultHolder(AdView adView,
                                 AdvertisementPresenter advertisementPresenter,
                                 VungleWebClient webClient) {
            this.adView = adView;
            this.advertisementPresenter = advertisementPresenter;
            this.webClient = webClient;
        }
    }

    private static class FullScreenPresentationTask extends BaseTask {

        private final AdLoader adLoader;

        @SuppressLint("StaticFieldLeak")
        private FullAdWidget fullAdWidget;
        @SuppressLint("StaticFieldLeak")
        private Context context;

        private final AdRequest request;
        private final OptionsState optionsState;
        private final FullScreenCallback fullscreenCallback;
        private final Bundle savedState;
        private final JobRunner jobRunner;
        private final VungleApiClient apiClient;
        private final CloseDelegate closeDelegate;
        private final OrientationDelegate orientationDelegate;
        private Advertisement advertisement;
        private final OMTracker.Factory omTrackerFactory;

        FullScreenPresentationTask(Context context,
                                   AdLoader adLoader,
                                   AdRequest request,
                                   Repository repository,
                                   VungleStaticApi vungleStaticApi,
                                   JobRunner jobRunner,
                                   VungleApiClient apiClient,
                                   FullAdWidget fullAdWidget,
                                   OptionsState optionsState,
                                   OrientationDelegate orientationDelegate,
                                   CloseDelegate closeDelegate,
                                   FullScreenCallback fullscreenCallback,
                                   OnModelLoadListener onModelLoadListener,
                                   Bundle savedState,
                                   OMTracker.Factory omTrackerFactory) {
            super(repository, vungleStaticApi, onModelLoadListener);
            this.request = request;
            this.fullAdWidget = fullAdWidget;
            this.optionsState = optionsState;
            this.context = context;
            this.fullscreenCallback = fullscreenCallback;
            this.savedState = savedState;
            this.jobRunner = jobRunner;
            this.apiClient = apiClient;
            this.orientationDelegate = orientationDelegate;
            this.closeDelegate = closeDelegate;
            this.adLoader = adLoader;
            this.omTrackerFactory = omTrackerFactory;
        }

        @Override
        void clear() {
            super.clear();
            context = null;
            fullAdWidget = null;
        }

        @Override
        protected PresentationResultHolder doInBackground(Void... voids) {
            Pair<Advertisement, Placement> data;
            try {
                data = loadPresentationData(request, savedState);
            } catch (VungleException e) {
                return new PresentationResultHolder(e);
            }

            advertisement = data.first;
            Placement placement = data.second;

            if (!adLoader.canRenderAd(advertisement)) {
                Log.e(TAG, "Advertisement is null or assets are missing");
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            if (placement.getPlacementAdType() == Placement.TYPE_VUNGLE_NATIVE) {
                return new PresentationResultHolder(
                        new VungleException(VungleException.INCORRECT_DEFAULT_API_USAGE_NATIVE));
            } else if (placement.getPlacementAdType() != Placement.TYPE_DEFAULT) {
                return new PresentationResultHolder(
                        new VungleException(VungleException.INCORRECT_DEFAULT_API_USAGE));
            }

            AdAnalytics adAnalytics = new JobDelegateAnalytics(jobRunner);

            String appId = null;
            Cookie appIdCookie = repository.load(Cookie.APP_ID, Cookie.class).get();
            if (appIdCookie != null && !TextUtils.isEmpty(appIdCookie.getString(Constants.APP_ID))) {
                appId = appIdCookie.getString(Constants.APP_ID);
            }
            Cookie configCookie = repository.load(Cookie.CONFIG_COOKIE, Cookie.class).get();
            boolean isAdDownloadOptEnabled = configCookie != null && configCookie.getBoolean("isAdDownloadOptEnabled");
            //if feature is enabled and not all assets downloaded then we need to query AdAsset
            //to ensure if theres any more Assets that are downloaded
            if(isAdDownloadOptEnabled && !advertisement.assetsFullyDownloaded) {
                List<AdAsset> adAssets = repository.loadAllAdAssetByStatus(advertisement.getId(),
                        AdAsset.Status.DOWNLOAD_SUCCESS);
                if(!adAssets.isEmpty()) {
                    advertisement.updateMRAIDTokensFromAssetDB(adAssets);
                    try {
                        repository.save(advertisement);
                    } catch (DatabaseHelper.DBException e) {
                        Log.e(TAG, "Unable to update tokens");
                    }
                }
            }

            Executors executors = ServiceLocator.getInstance(context).getService(Executors.class);
            ExecutorService offloadExecutor = executors.getOffloadExecutor();
            VungleWebClient webClient = new VungleWebClient(advertisement, placement, offloadExecutor);
            File assetDir = repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
            if (assetDir == null || !assetDir.isDirectory()) {
                Log.e(TAG, "Advertisement assets dir is missing");
                return new PresentationResultHolder(new VungleException(DB_ERROR));
            }

            switch (advertisement.getAdType()) {
                case Advertisement.TYPE_VUNGLE_LOCAL:
                    LocalAdPresenter localPresenter = new LocalAdPresenter(
                            advertisement,
                            placement,
                            repository,
                            new HandlerScheduler(),
                            adAnalytics,
                            webClient,
                            optionsState,
                            assetDir,
                            request.getImpression()
                    );
                    LocalAdView localView = new LocalAdView(context, fullAdWidget, orientationDelegate, closeDelegate);

                    return new PresentationResultHolder(localView, localPresenter, webClient);

                case Advertisement.TYPE_VUNGLE_MRAID:
                    OMTracker omTracker = omTrackerFactory.make(apiClient.getOmEnabled() && advertisement.getOmEnabled());
                    webClient.setWebViewObserver(omTracker);

                    MRAIDAdPresenter mraidPresenter = new MRAIDAdPresenter(
                            advertisement,
                            placement,
                            repository,
                            new HandlerScheduler(),
                            adAnalytics,
                            webClient,
                            optionsState,
                            assetDir,
                            omTracker,
                            request.getImpression()
                    );
                    MRAIDAdView mraidView = new MRAIDAdView(context, fullAdWidget, orientationDelegate, closeDelegate);

                    return new PresentationResultHolder(mraidView, mraidPresenter, webClient);

                default:
                    return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }
        }

        @Override
        protected void onPostExecute(PresentationResultHolder result) {
            super.onPostExecute(result);

            if (!isCancelled() && fullscreenCallback != null) {
                if (result.exception != null) {
                    Log.e(TAG, "Exception on creating presenter", result.exception);
                    fullscreenCallback.onResult(new Pair<AdView, AdvertisementPresenter>(null, null), result.exception);
                    return;
                }

                fullAdWidget.linkWebView(result.webClient, new JavascriptBridge(result.advertisementPresenter));
                fullscreenCallback.onResult(new Pair(result.adView, result.advertisementPresenter), result.exception);
            }
        }
    }

    private static class BannerViewPresentationTask extends BaseTask {

        @SuppressLint("StaticFieldLeak")
        private Context context;
        private final AdRequest request;
        private final AdConfig adConfig;
        private final ViewCallback viewCallback;
        private final Bundle savedState;
        private final JobRunner jobRunner;
        private final AdLoader adLoader;
        private final VungleApiClient apiClient;
        private final OMTracker.Factory omTrackerFactory;

        BannerViewPresentationTask(Context context,
                                    AdRequest request,
                                   AdConfig adConfig,
                                   AdLoader adLoader,
                                   Repository repository,
                                   VungleStaticApi vungleStaticApi,
                                   JobRunner jobRunner,
                                   ViewCallback viewCallback,
                                   Bundle savedState,
                                   OnModelLoadListener onModelLoadListener,
                                   VungleApiClient apiClient,
                                   OMTracker.Factory omTrackerFactory) {
            super(repository, vungleStaticApi, onModelLoadListener);
            this.context = context;
            this.request = request;
            this.adConfig = adConfig;
            this.viewCallback = viewCallback;
            this.savedState = savedState;
            this.jobRunner = jobRunner;
            this.adLoader = adLoader;
            this.apiClient = apiClient;
            this.omTrackerFactory = omTrackerFactory;
        }

        @Override
        void clear() {
            super.clear();
            context = null;
        }

        @Override
        protected PresentationResultHolder doInBackground(Void... voids) {
            Pair<Advertisement, Placement> data;
            try {
                data = loadPresentationData(request, savedState);
            } catch (VungleException e) {
                return new PresentationResultHolder(e);
            }

            Advertisement advertisement = data.first;

            if (advertisement.getAdType() != Advertisement.TYPE_VUNGLE_MRAID) {
                Log.e(TAG, "Invalid Ad Type for Native Ad.");
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            Placement placement = data.second;

            if (!adLoader.canPlayAd(advertisement)) {
                Log.e(TAG, "Advertisement is null or assets are missing");
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            Cookie configCookie = repository.load(Cookie.CONFIG_COOKIE, Cookie.class).get();
            boolean isAdDownloadOptEnabled = configCookie != null && configCookie.getBoolean("isAdDownloadOptEnabled");
            //if feature is enabled and not all assets downloaded then we need to query AdAsset
            //to ensure if theres any more Assets that are downloaded
            if(isAdDownloadOptEnabled && !advertisement.assetsFullyDownloaded) {
                List<AdAsset> adAssets = repository.loadAllAdAssetByStatus(advertisement.getId(),
                        AdAsset.Status.DOWNLOAD_SUCCESS);
                if(!adAssets.isEmpty()) {
                    advertisement.updateMRAIDTokensFromAssetDB(adAssets);
                    try {
                        repository.save(advertisement);
                    } catch (DatabaseHelper.DBException e) {
                        Log.e(TAG, "Unable to update tokens");
                    }
                }
            }

            AdAnalytics adAnalytics = new JobDelegateAnalytics(jobRunner);

            Executors executors = ServiceLocator.getInstance(context).getService(Executors.class);
            ExecutorService offloadExecutor = executors.getOffloadExecutor();
            VungleWebClient webClient = new VungleWebClient(advertisement, placement, offloadExecutor);
            File assetDir = repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
            if (assetDir == null || !assetDir.isDirectory()) {
                Log.e(TAG, "Advertisement assets dir is missing");
                return new PresentationResultHolder(new VungleException(DB_ERROR));
            }

            if ("mrec".equals(advertisement.getTemplateType()) &&
                    adConfig.getAdSize() != AdConfig.AdSize.VUNGLE_MREC) {
                Log.e(TAG, "Corresponding AdConfig#setAdSize must be passed for the type/size of banner ad");
                return new PresentationResultHolder(new VungleException(INVALID_SIZE));
            }

            if (placement.getPlacementAdType() == Placement.TYPE_DEFAULT) {
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            advertisement.configure(adConfig);
            try {
                repository.save(advertisement);
            } catch (DatabaseHelper.DBException e) {
                return new PresentationResultHolder(new VungleException(DB_ERROR));
            }

            OMTracker omTracker = omTrackerFactory.make(apiClient.getOmEnabled() && advertisement.getOmEnabled());
            webClient.setWebViewObserver(omTracker);

            MRAIDAdPresenter presenter = new MRAIDAdPresenter(
                    advertisement,
                    placement,
                    repository,
                    new HandlerScheduler(),
                    adAnalytics,
                    webClient,
                    null,
                    assetDir,
                    omTracker,
                    request.getImpression());

            return new PresentationResultHolder(null, presenter, webClient);
        }

        @Override
        protected void onPostExecute(PresentationResultHolder result) {
            super.onPostExecute(result);
            if (!isCancelled() && viewCallback != null) {
                viewCallback.onResult(new Pair<>((WebAdContract.WebAdPresenter) result.advertisementPresenter, result.webClient), result.exception);
            }
        }
    }

    private static class NativeViewPresentationTask extends BaseTask {

        @SuppressLint("StaticFieldLeak")
        private Context context;
        @SuppressLint("StaticFieldLeak")
        private NativeAdLayout nativeAdLayout;
        private final AdRequest request;
        private final AdConfig adConfig;
        private final NativeViewCallback nativeViewCallback;
        private final Bundle savedState;
        private final JobRunner jobRunner;
        private final AdLoader adLoader;

        NativeViewPresentationTask(Context context,
                                   NativeAdLayout nativeAdLayout,
                                   AdRequest request,
                                   AdConfig adConfig,
                                   AdLoader adLoader,
                                   Repository repository,
                                   VungleStaticApi vungleStaticApi,
                                   JobRunner jobRunner,
                                   NativeViewCallback nativeViewCallback,
                                   Bundle savedState,
                                   OnModelLoadListener onModelLoadListener) {
            super(repository, vungleStaticApi, onModelLoadListener);
            this.context = context;
            this.nativeAdLayout = nativeAdLayout;
            this.request = request;
            this.adConfig = adConfig;
            this.nativeViewCallback = nativeViewCallback;
            this.savedState = savedState;
            this.jobRunner = jobRunner;
            this.adLoader = adLoader;
        }

        @Override
        void clear() {
            super.clear();
            context = null;
            nativeAdLayout = null;
        }

        @Override
        protected PresentationResultHolder doInBackground(Void... voids) {
            Pair<Advertisement, Placement> data;
            try {
                data = loadPresentationData(request, savedState);
            } catch (VungleException e) {
                return new PresentationResultHolder(e);
            }

            Advertisement advertisement = data.first;

            if (advertisement.getAdType() != Advertisement.TYPE_VUNGLE_MRAID) {
                Log.e(TAG, "Invalid Ad Type for Native Ad.");
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            Placement placement = data.second;

            if (!adLoader.canPlayAd(advertisement)) {
                Log.e(TAG, "Advertisement is null or assets are missing");
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            Cookie configCookie = repository.load(Cookie.CONFIG_COOKIE, Cookie.class).get();
            boolean isAdDownloadOptEnabled = configCookie != null && configCookie.getBoolean("isAdDownloadOptEnabled");
            //if feature is enabled and not all assets downloaded then we need to query AdAsset
            //to ensure if theres any more Assets that are downloaded
            if(isAdDownloadOptEnabled && !advertisement.assetsFullyDownloaded) {
                List<AdAsset> adAssets = repository.loadAllAdAssetByStatus(advertisement.getId(),
                        AdAsset.Status.DOWNLOAD_SUCCESS);
                if(!adAssets.isEmpty()) {
                    advertisement.updateMRAIDTokensFromAssetDB(adAssets);
                    try {
                        repository.save(advertisement);
                    } catch (DatabaseHelper.DBException e) {
                        Log.e(TAG, "Unable to update tokens");
                    }
                }
            }

            AdAnalytics adAnalytics = new JobDelegateAnalytics(jobRunner);

            File assetDir = repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
            if (assetDir == null || !assetDir.isDirectory()) {
                Log.e(TAG, "Advertisement assets dir is missing");
                return new PresentationResultHolder(new VungleException(DB_ERROR));
            }

            if (!advertisement.isNativeTemplateType()) {
                return new PresentationResultHolder(new VungleException(AD_UNABLE_TO_PLAY));
            }

            advertisement.configure(adConfig);
            try {
                repository.save(advertisement);
            } catch (DatabaseHelper.DBException e) {
                return new PresentationResultHolder(new VungleException(DB_ERROR));
            }

            NativeAdPresenter presenter = new NativeAdPresenter(
                    advertisement,
                    placement,
                    repository,
                    new HandlerScheduler(),
                    adAnalytics,
                    null,
                    request.getImpression()
            );
            NativeAdView nativeAdView = new NativeAdView(context, nativeAdLayout);
            return new PresentationResultHolder(nativeAdView, presenter, null);
        }

        @Override
        protected void onPostExecute(PresentationResultHolder result) {
            super.onPostExecute(result);
            if (!isCancelled() && nativeViewCallback != null) {
                nativeViewCallback.onResult(new Pair<>((NativeAdContract.NativeView) result.adView,
                        (NativeAdContract.NativePresenter) result.advertisementPresenter), result.exception);
            }
        }
    }

    private abstract static class BaseTask extends AsyncTask<Void, Void, PresentationResultHolder> {

        protected final Repository repository;
        protected final VungleStaticApi vungleStaticApi;
        private OnModelLoadListener onDataLoadedListener;
        private AtomicReference<Advertisement> adRef = new AtomicReference<>();
        private AtomicReference<Placement> plRef = new AtomicReference<>();
        private AdLoader adLoader;
        private Downloader downloader;

        BaseTask(Repository repository, VungleStaticApi vungleStaticApi, OnModelLoadListener onModelLoadListener) {
            this.repository = repository;
            this.vungleStaticApi = vungleStaticApi;
            this.onDataLoadedListener = onModelLoadListener;
            Context context = Vungle.appContext();
            if (context != null) {
                ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
                adLoader = serviceLocator.getService(AdLoader.class);
                downloader = serviceLocator.getService(Downloader.class);
            }
        }

        Pair<Advertisement, Placement> loadPresentationData(AdRequest request, Bundle savedInstanceState) throws VungleException {
            if (!vungleStaticApi.isInitialized()) {

                SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                        .setEvent(SessionEvent.PLAY_AD)
                        .addData(SessionAttribute.SUCCESS, false)
                        .build());

                throw new VungleException(VUNGLE_NOT_INTIALIZED);
            }

            if (request == null || TextUtils.isEmpty(request.getPlacementId())) {

                SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                        .setEvent(SessionEvent.PLAY_AD)
                        .addData(SessionAttribute.SUCCESS, false)
                        .build());


                throw new VungleException(AD_UNABLE_TO_PLAY);
            }

            Placement placement = repository.load(request.getPlacementId(), Placement.class).get();
            if (placement == null) {
                Log.e(TAG, "No Placement for ID");

                SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                        .setEvent(SessionEvent.PLAY_AD)
                        .addData(SessionAttribute.SUCCESS, false)
                        .build());


                throw new VungleException(PLACEMENT_NOT_FOUND);
            }

            if (placement.isMultipleHBPEnabled() && request.getEventId() == null) {

                SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                        .setEvent(SessionEvent.PLAY_AD)
                        .addData(SessionAttribute.SUCCESS, false)
                        .build());


                throw new VungleException(MISSING_HBP_EVENT_ID);
            }

            plRef.set(placement);

            Advertisement advertisement = null;
            if (savedInstanceState == null) {
                advertisement = repository.findValidAdvertisementForPlacement(request.getPlacementId(), request.getEventId()).get();
            } else {
                // Restore cached adv, it should be deleted from disk already
                String adId = savedInstanceState.getString(EXTRA_ADVERTISEMENT);
                if (!TextUtils.isEmpty(adId)) {
                    advertisement = repository.load(adId, Advertisement.class).get();
                }
            }

            if (advertisement == null) {

                SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                        .setEvent(SessionEvent.PLAY_AD)
                        .addData(SessionAttribute.SUCCESS, false)
                        .build());

                throw new VungleException(AD_UNABLE_TO_PLAY);
            }

            adRef.set(advertisement);

            File assetDir = repository.getAdvertisementAssetDirectory(advertisement.getId()).get();
            if (assetDir == null || !assetDir.isDirectory()) {
                Log.e(TAG, "Advertisement assets dir is missing");

                SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                        .setEvent(SessionEvent.PLAY_AD)
                        .addData(SessionAttribute.SUCCESS, false)
                        .addData(SessionAttribute.EVENT_ID, advertisement.getId())
                        .build());

                throw new VungleException(DB_ERROR);
            }

            // Try to cancel downloading asset requests for the optimization enabled advertisement.
            if (adLoader != null && downloader != null && adLoader.isAdLoadOptimizationEnabled(advertisement)) {
                Log.d(TAG, "Try to cancel downloading assets.");
                List<DownloadRequest> requests = downloader.getAllRequests();
                for (DownloadRequest req : requests) {
                    if (advertisement.getId().equals(req.getAdvertisementId())) {
                        Log.d(TAG, "Cancel downloading: " + req);
                        downloader.cancel(req);
                    }
                }
            }

            return new Pair<>(advertisement, placement);
        }

        @Override
        protected void onPostExecute(PresentationResultHolder presentationResultHolder) {
            super.onPostExecute(presentationResultHolder);
            if (onDataLoadedListener != null) {
                onDataLoadedListener.onLoad(adRef.get(), plRef.get());
            }
        }

        void clear() {
            onDataLoadedListener = null;
        }

        interface OnModelLoadListener {
            void onLoad(Advertisement advertisement, Placement placement);
        }
    }

    private BaseTask.OnModelLoadListener onModelLoadListener = new BaseTask.OnModelLoadListener() {

        @Override
        public void onLoad(Advertisement ad, Placement pl) {
            currentAdvertisement = ad;
        }
    };
}
