package com.vungle.warren;

import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.ExceptionCode;
import static com.vungle.warren.error.VungleException.INCORRECT_BANNER_API_USAGE;
import static com.vungle.warren.error.VungleException.INVALID_SIZE;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;
import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.AdConfig.AdSize;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.admarkup.AdMarkup;
import com.vungle.warren.persistence.FutureResult;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.AdMarkupDecoder;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.TimeoutProvider;

import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

/**
 * Banners API for loading banner ads and retrieving VungleBanner view instances
 */
public final class Banners {

    private static final String TAG = Banners.class.getSimpleName();

    /**
     * Check if we can play an advertisement for the given placement. This method checks out file
     * system to see if there are asset files for the given placement and adSize and returns true if we have
     * assets stored which have not expired.
     *
     * @param placementId The placement identifier.
     * @param adSize      The placement adSize
     * @return true if an advertisement can be played immediately, false otherwise.
     */
    public static boolean canPlayAd(@NonNull final String placementId, final @NonNull AdSize adSize) {
        return canPlayAd(placementId, null, adSize);
    }

    /**
     * Check if we can play an advertisement for the given placement. This method checks out file
     * system to see if there are asset files for the given placement and adSize and returns true if we have
     * assets stored which have not expired.
     *
     * @param placementId The placement identifier.
     * @param markup      The Ad markup.
     * @param adSize      The placement adSize
     * @return true if an advertisement can be played immediately, false otherwise.
     */
    @SuppressWarnings("squid:S2583")
    public static boolean canPlayAd(@NonNull final String placementId, @Nullable final String markup, final @NonNull AdSize adSize) {
        if (!AdSize.isBannerAdSize(adSize)) {
            Log.e(TAG, "Invalid Ad Size. Cannot check loaded status of non banner size placements.");
            return false;
        }

        final Context context = Vungle.appContext();

        if (context == null) {
            Log.e(TAG, "Context is null");
            return false;
        }

        if (TextUtils.isEmpty(placementId)) {
            Log.e(TAG, "PlacementId is null");
            return false;
        }
        final AdMarkup serializedAdMarkup = AdMarkupDecoder.decode(markup);
        if (markup != null && serializedAdMarkup == null) {
            Log.e(TAG, "Invalid AdMarkup");
            return false;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        FutureResult<Boolean> futureResult = new FutureResult<>(sdkExecutors.getApiExecutor()
                .submit(new Callable<Boolean>() {
                    @Override
                    public Boolean call() {

                        if (!Vungle.isInitialized()) {
                            Log.e(TAG, "Vungle is not initialized");
                            return false;
                        }
                        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
                        Repository repository = serviceLocator.getService(Repository.class);
                        String eventId = serializedAdMarkup != null ? serializedAdMarkup.getEventId() : null;

                        Placement placement = repository.load(placementId, Placement.class).get();
                        if (placement == null)
                            return false;

                        if (placement.isMultipleHBPEnabled() && eventId == null) {
                            return false;
                        }

                        Advertisement advertisement = repository
                                .findValidAdvertisementForPlacement(placementId, eventId).get();
                        if (advertisement == null) {
                            return false;
                        }

                        AdSize plSize = placement.getAdSize();
                        AdSize advSize = advertisement.getAdConfig().getAdSize();

                        //allow playing MREC Ad loaded with Vungle#loadAd API,
                        //different api calls san set different Placement and Advertisements sizes
                        boolean skipSizeCheck = adSize == AdSize.VUNGLE_MREC
                                && AdSize.isDefaultAdSize(plSize) && AdSize.isDefaultAdSize(advSize)
                                && placement.getPlacementAdType() == Placement.TYPE_VUNGLE_MREC;

                        /* Allow size mix-and-match, but only if it's between banners, transition
                         * between banner and mrec is not allowed in any direction */
                        if (placement.isMultipleHBPEnabled()
                                && AdSize.isNonMrecBannerAdSize(plSize)
                                && AdSize.isNonMrecBannerAdSize(advSize)
                                && AdSize.isNonMrecBannerAdSize(adSize)) {
                            skipSizeCheck = true;
                        }

                        if (!skipSizeCheck && !(adSize == plSize && adSize == advSize)) {
                            return false;
                        }

                        return Vungle.canPlayAd(advertisement);
                    }
                })
        );

        return Boolean.TRUE.equals(futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS));
    }

    /**
     * Retrieve a VungleBanner view that displays a banner ad if an ad is loaded for that placement.
     * DO NOT MODIFY LayoutParams.
     * <p>
     * Usage Example:
     * VungleBanner vungleBanner = Banners.getBanner(.....)
     * parentView.addView(vungleBanner)
     *
     * @param placementId    The placement identifier
     * @param adSize         Ad Size
     * @param playAdCallback Optional callback for ad state
     * @return VungleBanner View after calling load ad successfully.
     * @deprecated please use {@link #getBanner(String, BannerAdConfig, PlayAdCallback)} instead
     */
    @Nullable
    @Deprecated
    public static VungleBanner getBanner(@NonNull String placementId,
                                         @NonNull AdSize adSize,
                                         @Nullable PlayAdCallback playAdCallback) {
        return getBanner(placementId, new BannerAdConfig(adSize), playAdCallback);
    }

    /**
     * Retrieve a VungleBanner view that displays a banner ad if an ad is loaded for that placement.
     * DO NOT MODIFY LayoutParams.
     * <p>
     * Usage Example:
     * VungleBanner vungleBanner = Banners.getBanner(.....)
     * parentView.addView(vungleBanner)
     *
     * @param placementId    The placement identifier
     * @param bannerAdConfig Ad Config
     * @param playAdCallback Optional callback for ad state
     * @return VungleBanner View after calling load ad successfully.
     */
    @Nullable
    public static VungleBanner getBanner(@NonNull String placementId,
                                         @NonNull BannerAdConfig bannerAdConfig,
                                         @Nullable PlayAdCallback playAdCallback) {
        return getBanner(placementId, null, bannerAdConfig, playAdCallback);
    }

    /**
     * Retrieve a VungleBanner view that displays a banner ad if an ad is loaded for that placement.
     * DO NOT MODIFY LayoutParams.
     * <p>
     * Usage Example:
     * VungleBanner vungleBanner = Banners.getBanner(.....)
     * parentView.addView(vungleBanner)
     *
     * @param placementId    The placement identifier
     * @param markup         The Ad markup
     * @param bannerAdConfig Ad Config
     * @param playAdCallback Optional callback for ad state
     * @return VungleBanner View after calling load ad successfully.
     */
    @SuppressWarnings("squid:S2583")
    @Nullable
    public static VungleBanner getBanner(@NonNull final String placementId,
                                         @Nullable final String markup,
                                         @NonNull final BannerAdConfig bannerAdConfig,
                                         @Nullable PlayAdCallback playAdCallback) {
        VungleLogger.debug("VungleBanner#getBanner", "getBanner call invoked");
        Context appContext = Vungle.appContext();
        //AdSize is being requested in both loadAd and getBanner.
        if (appContext == null) {
            Log.e(TAG, "Vungle is not initialized, returned VungleBanner = null");
            onPlaybackError(placementId, playAdCallback, VUNGLE_NOT_INTIALIZED);
            return null;
        }

        final AdSize adSize = bannerAdConfig.getAdSize();
        final ServiceLocator serviceLocator = ServiceLocator.getInstance(appContext);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        final TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        final VungleSettings settings = ServiceLocator.getInstance(appContext)
                .getService(RuntimeValues.class)
                .settings
                .get();

        final PlayAdCallback listener = new PlayAdCallbackWrapper(sdkExecutors.getUIExecutor(), playAdCallback);
        FutureResult<Pair<Boolean, Placement>> futureResult = new FutureResult<>(sdkExecutors.getBackgroundExecutor()
                .submit(new Callable<Pair<Boolean, Placement>>() {

                    @Override
                    public Pair<Boolean, Placement> call() throws Exception {
                        if (!Vungle.isInitialized()) {
                            Log.e(TAG, "Vungle is not initialized.");
                            onPlaybackError(placementId, listener, VUNGLE_NOT_INTIALIZED);
                            return new Pair<>(false, null);
                        }

                        if (TextUtils.isEmpty(placementId)) {
                            onPlaybackError(placementId, listener, PLACEMENT_NOT_FOUND);
                            return new Pair<>(false, null);
                        }

                        Repository repository = serviceLocator.getService(Repository.class);
                        Placement placement = repository.load(placementId, Placement.class).get();
                        if (placement == null) {
                            onPlaybackError(placementId, listener, PLACEMENT_NOT_FOUND);
                            return new Pair<>(false, null);
                        }

                        if (!AdSize.isBannerAdSize(adSize)) {
                            onPlaybackError(placementId, listener, INCORRECT_BANNER_API_USAGE);
                            return new Pair<>(false, placement);
                        }

                        if (!canPlayAd(placementId, markup, adSize)) {
                            onPlaybackError(placementId, listener, AD_UNABLE_TO_PLAY);
                            return new Pair<>(false, placement);
                        }

                        return new Pair<>(true, placement);
                    }
                }));

        Pair<Boolean, Placement> placementWillPlay = futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS);
        if (placementWillPlay == null) { // timeout
            onPlaybackError(placementId, playAdCallback, PLACEMENT_NOT_FOUND);
            return null;
        }
        boolean canPlay = placementWillPlay.first;
        if (!canPlay) {
            return null;
        }

        int refreshDuration = adSize != AdSize.VUNGLE_MREC ? placementWillPlay.second.getAdRefreshDuration() : 0;
        if (settings != null && settings.getBannerRefreshDisabled()) {
            refreshDuration = 0;
        }

        return new VungleBanner(
                appContext,
                placementId,
                markup,
                refreshDuration,
                bannerAdConfig,
                listener
        );
    }

    /**
     * Request a banner ad from Vungle for a specific configuration.
     *
     * @param placementId    - Banner Placement Id
     * @param bannerAdConfig - config of ad to be requested. See {@link BaseAdConfig}
     * @param callback       - optional callback for successful and unsuccessful ad load
     */
    @SuppressWarnings("squid:S2583")
    public static void loadBanner(@NonNull final String placementId,
                                  @NonNull final BannerAdConfig bannerAdConfig,
                                  @Nullable final LoadAdCallback callback) {
        loadBanner(placementId, null, bannerAdConfig, callback);
    }

    /**
     * Request a banner ad from Vungle for a specific size.
     * <p>
     * Banners.loadBanner("BANNER-12345", AdConfig.AdSize.BANNER, new LoadAdCallback(){... });
     *
     * @param placementId - Banner Placement Id
     * @param adSize      - size of ad to be requested see {@link AdSize}
     * @param callback    - optional callback for successful and unsuccessful ad load
     * @deprecated Please use {@link #loadBanner(String, BannerAdConfig, LoadAdCallback)}
     */
    @SuppressWarnings("squid:S2583")
    @Deprecated
    public static void loadBanner(@NonNull final String placementId,
                                  @NonNull final AdSize adSize,
                                  @Nullable final LoadAdCallback callback) {
        //noinspection ConstantConditions
        if (adSize == null) {
            onLoadError(placementId, callback, INVALID_SIZE);
            return;
        }
        loadBanner(placementId, new BannerAdConfig(adSize), callback);
    }

    /**
     * Request a banner ad from Vungle with specific markup and configuration.
     * <p>
     *
     * @param placementId    - Banner Placement Id
     * @param markup         - Banner Ad markup
     * @param bannerAdConfig - size of ad to be requested see {@link AdSize}
     * @param callback       - optional callback for successful and unsuccessful ad load
     */
    @SuppressWarnings("squid:S2583")
    public static void loadBanner(@NonNull final String placementId,
                                  @Nullable final String markup,
                                  @NonNull final BannerAdConfig bannerAdConfig,
                                  @Nullable final LoadAdCallback callback) {
        VungleLogger.debug("Banners#loadBanner", "loadBanner API call invoked");
        Context appCtx = Vungle.appContext();
        if (appCtx == null || !Vungle.isInitialized()) {
            onLoadError(placementId, callback, VUNGLE_NOT_INTIALIZED);
            return;
        }

        final AdConfig adConfig = new AdConfig(bannerAdConfig);

        if (!AdSize.isBannerAdSize(adConfig.getAdSize())) {
            onLoadError(placementId, callback, INCORRECT_BANNER_API_USAGE);
            return;
        }

        Vungle.loadAdInternal(placementId, markup, adConfig, callback);
    }

    private static void onPlaybackError(@NonNull String placementId,
                                        @Nullable PlayAdCallback playAdCallback,
                                        @ExceptionCode int code) {
        VungleException ex = new VungleException(code);
        if (playAdCallback != null) {
            playAdCallback.onError(placementId, ex);
        }
        VungleLogger.error("Banners#onPlaybackError", "Banner play error: " + ex.getLocalizedMessage());
    }

    private static void onLoadError(@NonNull String placementId,
                                    @Nullable LoadAdCallback callback,
                                    @ExceptionCode int code) {
        VungleException ex = new VungleException(code);
        if (callback != null) {
            callback.onError(placementId, ex);
        }
        VungleLogger.error("Banners#onLoadError", "Banner load error: " + ex.getLocalizedMessage());
    }
}
