package com.vungle.warren;

import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;

import com.google.gson.annotations.SerializedName;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;

import static com.vungle.warren.AdConfig.AdSize.VUNGLE_DEFAULT;

/**
 * General settings and Configuration Class for playAd Requests
 * To prevent any issues, generate a new BaseAdConfig object per playAd() request.
 */
class BaseAdConfig {
    /**
     * Start this advertisement muted.
     */
    public static final int MUTED = 1;
    /**
     * Bitmask which stores the configuration settings for this advertisement.
     */
    @SerializedName("settings")
    protected int settings;
    /**
     * When set the SDK will check if it has an Ad that can be rendered in the specific size
     */
    @SerializedName("adSize")
    private AdConfig.AdSize adSize;

    protected boolean muteChangedByApi;

    public BaseAdConfig() {
    }

    public BaseAdConfig(AdConfig.AdSize adSize) {
        this.adSize = adSize;
    }

    public BaseAdConfig(BaseAdConfig baseAdConfig) {
        this(baseAdConfig.getAdSize());
        settings = baseAdConfig.getSettings();
    }


    /**
     * <p>Sets whether the video should start with its sound matching the
     * settings of your enclosing application <i>(the default</i> or muted.</p>
     * <p>
     * <p>Note that the user can still manually mute or unmute the sound during
     * playback.</p>
     *
     * @param isMuted <code>true</code> if the video should start with its
     *                audio settings matching those of your enclosing application;
     *                <code>false</code> if it should start muted.
     */
    public void setMuted(boolean isMuted) {
        if (isMuted) {
            settings = settings | MUTED;
        } else {
            settings = settings & ~MUTED;
        }
        muteChangedByApi = true;
    }

    /**
     * Returns AdSize
     *
     * @return AdSize
     */
    public AdConfig.AdSize getAdSize() {
        return adSize == null ? VUNGLE_DEFAULT : adSize;
    }

    /**
     * Sets the AdSize, currently only supported for MREC/Banner ad units
     * should be called immediately before getBannerAd
     *
     * @param adSize for the Ad
     */
    public void setAdSize(AdConfig.AdSize adSize) {
        this.adSize = adSize;
    }

    /**
     * returns the settings Bitmask which stores the configuration for this advertisement
     *
     * @return int
     */
    @RestrictTo(RestrictTo.Scope.LIBRARY)
    public int getSettings() {
        return settings;
    }

}
