package com.vungle.warren;

import static com.vungle.warren.PrivacyManager.COPPA.COPPA_NOTSET;

import android.os.Build;
import android.text.TextUtils;
import android.util.Base64;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.google.gson.Gson;
import com.vungle.warren.locale.LocaleInfo;
import com.vungle.warren.model.BidTokenV3;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.GdprCookie;
import com.vungle.warren.model.token.AndroidInfo;
import com.vungle.warren.model.token.Ccpa;
import com.vungle.warren.model.token.Consent;
import com.vungle.warren.model.token.Coppa;
import com.vungle.warren.model.token.Device;
import com.vungle.warren.model.token.Extension;
import com.vungle.warren.model.token.Gdpr;
import com.vungle.warren.model.token.Request;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.SDKExecutors;
import com.vungle.warren.utility.TimeoutProvider;
import com.vungle.warren.utility.platform.Platform;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPOutputStream;

public class BidTokenEncoder {
    private final Repository repository;
    private final TimeoutProvider timeoutProvider;
    private final Gson gson;
    private final Platform platform;
    private final LocaleInfo localeInfo;

    public BidTokenEncoder(
            Repository repository,
            TimeoutProvider timeoutProvider,
            LocaleInfo localeInfo,
            Platform platform,
            Gson gson,
            SDKExecutors sdkExecutors) {
        this.gson = gson;
        this.timeoutProvider = timeoutProvider;
        this.repository = repository;
        this.localeInfo = localeInfo;
        this.platform = platform;

        PrivacyManager.getInstance().init(sdkExecutors.getBackgroundExecutor(), repository);
    }

    public @Nullable
    String encode(String placementId, int maxBidTokenSizeBytes, int ordinalCount) {
        return V3BidToken(placementId, maxBidTokenSizeBytes, ordinalCount);
    }

    private String V3BidToken(String placementId, int maxBidTokenSizeBytes, int ordinalCount) {
        String token = constructV3Token(placementId, maxBidTokenSizeBytes, ordinalCount);
        ByteArrayOutputStream os = new ByteArrayOutputStream(token.length());
        try {
            GZIPOutputStream gos = new GZIPOutputStream(os);
            gos.write(token.getBytes());
            gos.close();
            byte[] compressed = os.toByteArray();
            String base64 = Base64.encodeToString(compressed, Base64.NO_WRAP);
            os.close();
            return "3:" + base64;
        } catch (IOException e) {
            return null;
        }
    }

    private String constructV3Token(String placementId, int maxBidTokenSizeBytes, int ordinalCount) {
        Cookie ccpaConsent = repository.load(Cookie.CCPA_COOKIE, Cookie.class)
                .get(timeoutProvider.getTimeout(), TimeUnit.MILLISECONDS);

        Ccpa ccpa = new Ccpa(getCCPAStatus(ccpaConsent));

        Gdpr gdpr = getGdpr();
        Coppa coppa = getCoppa();
        Consent consent = new Consent(ccpa, gdpr, coppa);

        Extension extension = new Extension(
                platform.getIsSideloaded(),
                platform.getIsSDCardPresent(),
                platform.getIsSoundEnabled()
        );
        String ifa = null;
        boolean isAmazonDevice = Platform.MANUFACTURER_AMAZON.equals(Build.MANUFACTURER);
        AndroidInfo androidInfo = isAmazonDevice ? null : new AndroidInfo();
        AndroidInfo amazonInfo = isAmazonDevice ? new AndroidInfo() : null;
        if (PrivacyManager.getInstance().shouldSendAdIds()) {
            String gaid = platform.getAdvertisingInfo().advertisingId;
            // Android id will be retrieved if gaid is empty.
            String androidId = TextUtils.isEmpty(gaid) ? platform.getAndroidId() : "";
            ifa = TextUtils.isEmpty(gaid) ? androidId : gaid;

            if (!TextUtils.isEmpty(androidId)) {
                if (isAmazonDevice) {
                    amazonInfo.android_id = androidId;
                } else {
                    androidInfo.android_id = androidId;
                }
            }
        }
        // App set id is always required.
        if (isAmazonDevice) {
            amazonInfo.app_set_id = platform.getAppSetId();
        } else {
            androidInfo.app_set_id = platform.getAppSetId();
        }
        Device device = new Device(
                platform.getIsBatterySaverEnabled(),
                localeInfo.getLanguage(),
                localeInfo.getTimeZoneId(),
                platform.getVolumeLevel(),
                ifa, amazonInfo, androidInfo,
                extension
        );
        Request request =
                new Request(
                        getConfigExtension(),
                        ordinalCount,
                        getAvailableBidTokens(placementId, maxBidTokenSizeBytes, ordinalCount),
                        VungleApiClient.getHeaderUa()
                );
        BidTokenV3 bidTokenV3 = new BidTokenV3(device, request, consent);
        return gson.toJson(bidTokenV3);
    }

    private List<String> getAvailableBidTokens(
            @Nullable String placementId,
            int maxBidTokenSizeBytes,
            int ordinalCount
    ) {
        final int bidTokenMaxBytesSanitized =
                maxBidTokenSizeBytes <= 0 ? Integer.MAX_VALUE - 1 : maxBidTokenSizeBytes;

        return repository.getAvailableBidTokens(
                placementId,
                getAvailableSizeForHBT(bidTokenMaxBytesSanitized,
                        "2",
                        Integer.toString(ordinalCount))
                , ",".getBytes().length).get();
    }

    private static @Ccpa.Status
    String getCCPAStatus(@Nullable Cookie ccpaConsent) {
        // Intentionally, should send opt-in when there is no value available.
        if (ccpaConsent == null) {
            return Ccpa.OPTED_IN;
        }
        return Cookie.CONSENT_STATUS_OPTED_OUT.equals(ccpaConsent.getString(Cookie.CCPA_CONSENT_STATUS)) ? Ccpa.OPTED_OUT : Ccpa.OPTED_IN;
    }

    private @Nullable
    Coppa getCoppa() {
        PrivacyManager privacyManager = PrivacyManager.getInstance();
        PrivacyManager.COPPA hasCoppa = privacyManager.getCoppaStatus();
        if (hasCoppa == COPPA_NOTSET) {
            return null;
        }
        return new Coppa(hasCoppa.getValue());
    }

    private Gdpr getGdpr() {
        GdprCookie gdprCookie = new GdprCookie(repository, timeoutProvider);
        String consentSource = gdprCookie.getSource();
        String gdprConsent = gdprCookie.getConsentStatus();
        Long timestamp = gdprCookie.getTimeStamp();
        String messageVersion = gdprCookie.getMessageVersion();
        return new Gdpr(gdprConsent, consentSource, messageVersion, timestamp);
    }

    @NonNull
    private String getConfigExtension() {
        Cookie cookie = repository.load(Cookie.CONFIG_EXTENSION, Cookie.class)
                .get(timeoutProvider.getTimeout(), TimeUnit.MILLISECONDS);
        String extension = "";
        if (cookie != null) {
            extension = cookie.getString(Cookie.CONFIG_EXTENSION);
        }
        return extension;
    }

    @VisibleForTesting
    static int getAvailableSizeForHBT(
            int bytesAvailable,
            String encodeVersion,
            String headerBiddingOrdinalCount
    ) {

        //math to calculate available bytes for tokens: solving for token_available_size
//        total_size = n + “:” + base64(n + “:” + token_available_size)
//        total_size - n  - “:”= 4/3(n + “:” + token_available_size)
//        0.75(total_size -n - “;”) = n + “:” + token_available_size
//        0.75(total_size -n - “;”)  n  - “:” = [token_available_size]

        int maxBytesAvailableForToken =
                (int) (3 * Math.floor((bytesAvailable - encodeVersion.getBytes().length - ":".getBytes().length) / 4)
                        - ":".getBytes().length - headerBiddingOrdinalCount.getBytes().length);

        //rounding to nearest 4 bit
        return (int) Math.max(Math.round((double) maxBytesAvailableForToken / 4) * 4, 0);
    }
}