package com.vungle.warren;

import android.os.Bundle;
import android.os.SystemClock;
import android.util.Log;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.vungle.warren.tasks.CacheBustJob;
import com.vungle.warren.tasks.JobInfo;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.utility.ActivityManager;

public class CacheBustManager {

    public static final String NEXT_CACHE_BUST = "next_cache_bust";
    public static final String CACHE_BUST_INTERVAL = "cache_bust_interval";
    public static final int NO_VALUE = Integer.MIN_VALUE;
    private JobRunner jobRunner;
    private static final int DEFAULT_REFRESH_RATE = 0;
    @VisibleForTesting
    long refreshRate = DEFAULT_REFRESH_RATE;
    private long overrideRefreshRate = NO_VALUE;
    @VisibleForTesting
    static final long MINIMUM_REFRESH_RATE = 15 * 60 * 1000; //15 mins

    private long startTrackingTime;
    private long refreshElapsed;

    private int status;

    @IntDef(value = {CacheBustManagerState.STOPPED, CacheBustManagerState.IN_PROGRESS})
    @interface CacheBustManagerState {
        int STOPPED = 0;
        int IN_PROGRESS = 1;
    }


    protected CacheBustManager(@NonNull JobRunner runner) {
        this.jobRunner = runner;
        if (ActivityManager.getInstance().isInitialized()) {
            setLifecycleListener();
        } else {
            Log.e(CacheBustManager.class.getSimpleName(), "No lifecycle listener set");
            VungleLogger.error(CacheBustManager.class.getSimpleName() + "#deliverError", "No lifecycle listener set");
        }
        status = CacheBustManagerState.STOPPED;

    }

    public synchronized void startBust() {
        if (status == CacheBustManagerState.IN_PROGRESS) {
            return;
        }
        status = CacheBustManagerState.IN_PROGRESS;
        if (refreshRate == 0) {
            jobRunner.execute(CacheBustJob.makeJobInfo());
        } else {
            Bundle bundle = new Bundle();
            bundle.putLong(CACHE_BUST_INTERVAL, refreshRate);
            bundle.putLong(CacheBustManager.NEXT_CACHE_BUST,
                    getCurrentTime() + refreshRate);
            jobRunner.execute(
                    CacheBustJob.makeJobInfo()
                            .setReschedulePolicy(refreshRate, JobInfo.ReschedulePolicy.LINEAR)
                            .setExtras(bundle)
            );
        }
        startTrackingTime = getCurrentTime();
    }

    @VisibleForTesting
    void overrideRefreshRate(long refreshRate) {
        overrideRefreshRate = refreshRate;
        this.refreshRate = refreshRate;
    }

    //on next set of refresh, if refresh is different update jobs
    public void setRefreshRate(long refreshRate) {
        if (overrideRefreshRate != NO_VALUE) {
            this.refreshRate = overrideRefreshRate;
            return;
        }
        long newRefreshRate = refreshRate > 0 ? Math.max(refreshRate, MINIMUM_REFRESH_RATE) : 0;
        if (newRefreshRate != this.refreshRate) {
            this.refreshRate = newRefreshRate;
            if (status == CacheBustManagerState.IN_PROGRESS) {
                jobRunner.cancelPendingJob(CacheBustJob.TAG);
                status = CacheBustManagerState.STOPPED;
                startBust();
            }
        }
    }

    private void setLifecycleListener(){
        ActivityManager.getInstance().addListener(new ActivityManager.LifeCycleCallback() {
            @Override
            public void onStart() {
                super.onStart();
                resumeCacheBust();
            }

            @Override
            public void onStop() {
                super.onStop();
                stopCacheBust();
            }
        });
    }

    void resumeCacheBust() {
        if ((status == CacheBustManagerState.STOPPED) && refreshRate != 0) {
            status = CacheBustManagerState.IN_PROGRESS;
            Bundle bundle = new Bundle();
            bundle.putLong(CACHE_BUST_INTERVAL, refreshRate);
            bundle.putLong(CacheBustManager.NEXT_CACHE_BUST,
                    getCurrentTime() + refreshRate);

            jobRunner.execute(
                    CacheBustJob.makeJobInfo()
                            .setDelay(refreshRate - refreshElapsed)
                            .setReschedulePolicy(refreshRate, JobInfo.ReschedulePolicy.LINEAR)
                            .setExtras(bundle)
            );

            refreshElapsed = 0;
            startTrackingTime = getCurrentTime();

        }
    }

    void stopCacheBust(){
        if (refreshRate != 0) {
            refreshElapsed = (getCurrentTime() - startTrackingTime) % refreshRate;
        }
        jobRunner.cancelPendingJob(CacheBustJob.TAG);
        status = CacheBustManagerState.STOPPED;
    }

    long getCurrentTime() {
        return SystemClock.elapsedRealtime();
    }
}


