package com.vungle.warren;

import android.annotation.SuppressLint;
import android.content.Context;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.view.MotionEvent;
import android.view.WindowManager;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

import com.vungle.warren.analytics.AdAnalytics;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.utility.ViewUtility;

import java.util.regex.Pattern;

import static com.vungle.warren.AdConfig.AdSize.VUNGLE_DEFAULT;

public class ClickCoordinateTracker {

    private static final int DEFAULT_WIDTH_HEIGHT = 0;

    private static final String MACRO_REQ_WIDTH = Pattern.quote("{{{req_width}}}");
    private static final String MACRO_REQ_HEIGHT = Pattern.quote("{{{req_height}}}");
    private static final String MACRO_WIDTH = Pattern.quote("{{{width}}}");
    private static final String MACRO_HEIGHT = Pattern.quote("{{{height}}}");
    private static final String MACRO_DOWN_X = Pattern.quote("{{{down_x}}}");
    private static final String MACRO_DOWN_Y = Pattern.quote("{{{down_y}}}");
    private static final String MACRO_UP_X = Pattern.quote("{{{up_x}}}");
    private static final String MACRO_UP_Y = Pattern.quote("{{{up_y}}}");

    private final Advertisement advertisement;

    private final AdAnalytics adAnalytics;

    @VisibleForTesting
    ClickCoordinate currentClick;

    public ClickCoordinateTracker(@NonNull Advertisement advertisement, @NonNull AdAnalytics analytics) {
        this.advertisement = advertisement;
        this.adAnalytics = analytics;
    }

    public void trackCoordinate(MotionEvent event) {

        if (!advertisement.isClickCoordinatesTrackingEnabled()) {
            return;
        }
        if (currentClick == null) {
            currentClick = new ClickCoordinate();
        }
        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                currentClick.setDownCoordinate(new Coordinate((int) event.getX(), (int) event.getY()));
                break;
            case MotionEvent.ACTION_UP:
                currentClick.setUpCoordinate(new Coordinate((int) event.getX(), (int) event.getY()));
                if (currentClick.ready()) {
                    sendClickCoordinates();
                }
                break;
            default:
                break;
        }

    }

    private void sendClickCoordinates() {
        if (adAnalytics != null) {
            String[] tpatUrls = advertisement.getTpatUrls(Advertisement.TPAT_CLICK_COORDINATES_URLS);
            if (tpatUrls == null || tpatUrls.length == 0) {
                return;
            }

            int reqWidth = getRequestWidth();
            int reqHeight = getRequestHeight();
            int adWidth = getRequestWidth();
            int adHeight = getRequestHeight();

            for (int i = 0; i < tpatUrls.length; i++) {
                String tpat = tpatUrls[i];
                if (TextUtils.isEmpty(tpat)) {
                    continue;
                }
                tpat = tpat.replaceAll(MACRO_REQ_WIDTH, Integer.toString(reqWidth))
                        .replaceAll(MACRO_REQ_HEIGHT, Integer.toString(reqHeight))
                        .replaceAll(MACRO_WIDTH, Integer.toString(adWidth))
                        .replaceAll(MACRO_HEIGHT, Integer.toString(adHeight))
                        .replaceAll(MACRO_DOWN_X, Integer.toString(currentClick.downCoordinate.x))
                        .replaceAll(MACRO_DOWN_Y, Integer.toString(currentClick.downCoordinate.y))
                        .replaceAll(MACRO_UP_X, Integer.toString(currentClick.upCoordinate.x))
                        .replaceAll(MACRO_UP_Y, Integer.toString(currentClick.upCoordinate.y));
                tpatUrls[i] = tpat;
            }
            adAnalytics.ping(tpatUrls);
        }
    }

    static class ClickCoordinate {
        Coordinate downCoordinate;
        Coordinate upCoordinate;

        public ClickCoordinate() {
            this.downCoordinate = new Coordinate(Integer.MIN_VALUE, Integer.MIN_VALUE);
            this.upCoordinate = new Coordinate(Integer.MIN_VALUE, Integer.MIN_VALUE);
        }

        public void setUpCoordinate(Coordinate coordinate) {
            this.upCoordinate = coordinate;
        }

        public void setDownCoordinate(Coordinate coordinate) {
            this.downCoordinate = coordinate;
        }

        public boolean ready() {
            return downCoordinate.x != Integer.MIN_VALUE && downCoordinate.y != Integer.MIN_VALUE &&
                    upCoordinate.x != Integer.MIN_VALUE && upCoordinate.y != Integer.MIN_VALUE;
        }
    }

    private int getRequestWidth() {
        if (Vungle.appContext() != null && advertisement.getAdConfig() != null) {
            AdConfig.AdSize adSize = advertisement.getAdConfig().getAdSize();
            if (adSize == VUNGLE_DEFAULT) {
                return getDeviceWidth();
            }
            return ViewUtility.dpToPixels(Vungle.appContext(), adSize.getWidth());
        }
        return DEFAULT_WIDTH_HEIGHT;
    }

    private int getRequestHeight() {
        if (Vungle.appContext() != null && advertisement.getAdConfig() != null) {
            AdConfig.AdSize adSize = advertisement.getAdConfig().getAdSize();
            if (adSize == VUNGLE_DEFAULT) {
                return getDeviceHeight();
            }
            return ViewUtility.dpToPixels(Vungle.appContext(), adSize.getHeight());
        }
        return DEFAULT_WIDTH_HEIGHT;
    }

    private int getDeviceWidth() {
        if (Vungle.appContext() != null) {
            return DeviceScreenInfo.getInstance(Vungle.appContext()).getDeviceWidth();
        }
        return DEFAULT_WIDTH_HEIGHT;
    }

    private int getDeviceHeight() {
        if (Vungle.appContext() != null) {
            return DeviceScreenInfo.getInstance(Vungle.appContext()).getDeviceHeight();
        }
        return DEFAULT_WIDTH_HEIGHT;
    }

    static class Coordinate {
        int x;
        int y;

        public Coordinate(int x, int y) {
            this.x = x;
            this.y = y;
        }
    }

    public static class DeviceScreenInfo {

        @SuppressLint("StaticFieldLeak")
        private static DeviceScreenInfo INSTANCE;
        private final Context context;
        private final DisplayMetrics dm;

        private DeviceScreenInfo(@NonNull Context context) {
            this.context = context.getApplicationContext();
            this.dm = new DisplayMetrics();
            WindowManager windowManager = (WindowManager) this.context.getSystemService(Context.WINDOW_SERVICE);
            windowManager.getDefaultDisplay().getMetrics(this.dm);
        }

        public static DeviceScreenInfo getInstance(@NonNull Context context) {
            if (INSTANCE == null) {
                INSTANCE = new DeviceScreenInfo(context);
            }
            return INSTANCE;
        }

        public int getDeviceWidth() {
            return dm.widthPixels;
        }

        public int getDeviceHeight() {
            return dm.heightPixels;
        }
    }
}
