package com.vungle.warren;

import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;

import android.content.Context;
import android.graphics.Bitmap;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.admarkup.AdMarkup;
import com.vungle.warren.persistence.FutureResult;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.ui.view.MediaView;
import com.vungle.warren.utility.AdMarkupDecoder;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.ImageLoader;
import com.vungle.warren.utility.ImpressionTracker;
import com.vungle.warren.utility.TimeoutProvider;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

/**
 * Vungle Native Ads Class
 */
public class NativeAd {

    private static final String TAG = NativeAd.class.getSimpleName();

    private static final String TOKEN_APP_NAME = "APP_NAME";
    private static final String TOKEN_APP_DESCRIPTION = "APP_DESCRIPTION";
    private static final String TOKEN_CTA_BUTTON_TEXT = "CTA_BUTTON_TEXT";
    public static final String TOKEN_CTA_BUTTON_URL = "CTA_BUTTON_URL";
    private static final String TOKEN_APP_RATING_VALUE = "APP_RATING_VALUE";
    private static final String TOKEN_SPONSORED_BY = "SPONSORED_BY";
    private static final String TOKEN_VUNGLE_PRIVACY_ICON_URL = "VUNGLE_PRIVACY_ICON_URL";
    public static final String TOKEN_VUNGLE_PRIVACY_URL = "VUNGLE_PRIVACY_URL";
    private static final String TOKEN_APP_ICON = "APP_ICON";
    private static final String TOKEN_MAIN_IMAGE = "MAIN_IMAGE";

    private static final int STATE_NEW = 1;
    private static final int STATE_READY = 2;
    private static final int STATE_VIEWING = 3;
    private static final int STATE_DESTROY = 4;
    private static final int STATE_ERROR = 5;

    private final Context context;
    private final String placementId;
    private String adMarkUp;
    private AdConfig adConfig;
    private Map<String, String> nativeAdAssetMap;
    private NativeAdListener nativeAdCallback;

    private NativeAdLayout rootNativeView;
    private ImageView iconView;
    @Nullable
    private MediaView contentView;
    private ImpressionTracker impressionTracker;
    private final ImageLoader imageLoader;
    private final Executor uiExecutor;

    // Root view for rendering our privacy icon.
    private FrameLayout adOptionsRootView;

    private NativeAdOptionsView adOptionsView;

    private List<View> clickableViews;

    // placement status of ad lifecycle
    private int adState;

    public NativeAd(@NonNull Context context, @NonNull String placementId) {
        this.context = context;
        this.placementId = placementId;

        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);

        uiExecutor = sdkExecutors.getUIExecutor();

        imageLoader = ImageLoader.getInstance();
        imageLoader.init(sdkExecutors.getIOExecutor());

        adState = STATE_NEW;
    }

    /**
     * Request a native ad from Vungle for a specific configuration.
     *
     * @param nativeAdConfig - config of ad to be requested. See {@link AdConfig}
     * @param callback       - optional callback for successful and unsuccessful ad load
     */
    public void loadAd(@Nullable AdConfig nativeAdConfig, @Nullable final NativeAdListener callback) {
        loadAd(nativeAdConfig, null, callback);
    }

    /**
     * Request a native ad from Vungle with specific markup and configuration.
     * <p>
     *
     * @param nativeAdConfig - config of ad to be requested. See {@link AdConfig}
     * @param markup         - Native Ad markup
     * @param callback       - optional callback for successful and unsuccessful ad load
     */
    public void loadAd(@Nullable AdConfig nativeAdConfig,
                       @Nullable final String markup, @Nullable final NativeAdListener callback) {
        VungleLogger.debug("NativeAd#loadAd", "loadAd API call invoked");
        if (!Vungle.isInitialized()) {
            onLoadError(placementId, callback, VUNGLE_NOT_INTIALIZED);
            return;
        }

        adState = STATE_NEW;

        adConfig = nativeAdConfig == null ? new AdConfig() : nativeAdConfig;
        adMarkUp = markup;
        nativeAdCallback = callback;

        Vungle.loadAdInternal(placementId, markup, adConfig, loadAdCallback);
    }

    private final LoadNativeAdCallback loadAdCallback = new LoadNativeAdCallback() {

        @Override
        public void onAdLoad(@Nullable Advertisement advertisement) {
            VungleLogger.debug(true, TAG, "NativeAd", "Native Ad Loaded : " + placementId);

            if (advertisement == null) {
                onLoadError(placementId, nativeAdCallback, VungleException.AD_FAILED_TO_DOWNLOAD);
                return;
            }

            adState = STATE_READY;

            nativeAdAssetMap = advertisement.getMRAIDArgsInMap();

            if (nativeAdCallback != null) {
                nativeAdCallback.onNativeAdLoaded(NativeAd.this);
            }
        }

        @Override
        public void onAdLoad(String placementId) {
            VungleLogger.error(true, TAG, "NativeAd", "Internal error! " +
                    "For native ads we should use onAdLoad(advertisement) callback.");
        }

        @Override
        public void onError(String placementId, VungleException exception) {
            VungleLogger.debug(true, TAG, "NativeAd",
                    "Native Ad Load Error : " + placementId + " Message : " + exception.getLocalizedMessage());
            onLoadError(placementId, nativeAdCallback, exception.getExceptionCode());
        }
    };

    private void onLoadError(@NonNull String placementId,
                             @Nullable NativeAdListener callback,
                             @VungleException.ExceptionCode int code) {
        adState = STATE_ERROR;

        VungleException ex = new VungleException(code);
        if (callback != null) {
            callback.onAdLoadError(placementId, ex);
        }
        VungleLogger.error("NativeAd#onLoadError", "NativeAd load error: " + ex.getLocalizedMessage());
    }

    /**
     * Check if we can play an advertisement for the given placement. This method checks out file
     * system to see if there are asset files for the given placement and returns true if we have
     * assets stored which have not expired.
     *
     * @return true if an advertisement can be played immediately, false otherwise.
     */
    public boolean canPlayAd() {
        if (TextUtils.isEmpty(placementId)) {
            VungleLogger.error(true, TAG, "NativeAd", "PlacementId is null");
            return false;
        }

        if (adState != STATE_READY) {
            Log.w(TAG, "Ad is not loaded or is displaying for placement: " + placementId);
            return false;
        }

        final AdMarkup serializedAdMarkup = AdMarkupDecoder.decode(adMarkUp);
        if (!TextUtils.isEmpty(adMarkUp) && serializedAdMarkup == null) {
            Log.e(TAG, "Invalid AdMarkup");
            return false;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        FutureResult<Boolean> futureResult = new FutureResult<>(sdkExecutors.getApiExecutor()
                .submit(new Callable<Boolean>() {
                    @Override
                    public Boolean call() {

                        if (!Vungle.isInitialized()) {
                            VungleLogger.error(true, TAG, "NativeAd", "Vungle is not initialized");
                            return false;
                        }

                        Repository repository = serviceLocator.getService(Repository.class);
                        AdRequest request = new AdRequest(placementId, AdMarkupDecoder.decode(adMarkUp), false);

                        Placement placement = repository.load(placementId, Placement.class).get();
                        if (placement == null)
                            return false;

                        if (placement.isMultipleHBPEnabled() && request.getEventId() == null) {
                            return false;
                        }

                        Advertisement advertisement = repository
                                .findValidAdvertisementForPlacement(placementId, request.getEventId()).get();
                        if (advertisement == null) {
                            return false;
                        }

                        return Vungle.canPlayAd(advertisement);
                    }
                })
        );

        return Boolean.TRUE.equals(futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS));
    }

    public String getPlacementId() {
        return placementId;
    }

    /**
     * Call when native ad is no longer needed.
     */
    public void destroy() {
        Log.d(TAG, "destroy()");
        adState = STATE_DESTROY;

        if (nativeAdAssetMap != null) {
            nativeAdAssetMap.clear();
            nativeAdAssetMap = null;
        }

        if (impressionTracker != null) {
            impressionTracker.destroy();
            impressionTracker = null;
        }

        if (iconView != null) {
            iconView.setImageDrawable(null);
            iconView = null;
        }

        if (contentView != null) {
            contentView.destroy();
            contentView = null;
        }

        if (adOptionsView != null) {
            adOptionsView.destroy();
            adOptionsView = null;
        }

        if (rootNativeView != null) {
            rootNativeView.finishDisplayingAdInternal(true);
            rootNativeView = null;
        }
    }

    /**
     * app icon url
     */
    @NonNull
    public String getAppIcon() {
        String ret = nativeAdAssetMap == null ? "" : nativeAdAssetMap.get(TOKEN_APP_ICON);
        return ret == null ? "" : ret;
    }

    /**
     * app name
     */
    @NonNull
    public String getAdTitle() {
        String ret = nativeAdAssetMap == null ? "" : nativeAdAssetMap.get(TOKEN_APP_NAME);
        return ret == null ? "" : ret;
    }

    /**
     * the ad description
     */
    @NonNull
    public String getAdBodyText() {
        String ret = nativeAdAssetMap == null ? "" : nativeAdAssetMap.get(TOKEN_APP_DESCRIPTION);
        return ret == null ? "" : ret;
    }

    /**
     * call to action phrase
     */
    @NonNull
    public String getAdCallToActionText() {
        String ret = nativeAdAssetMap == null ? "" : nativeAdAssetMap.get(TOKEN_CTA_BUTTON_TEXT);
        return ret == null ? "" : ret;
    }

    /**
     * app rating
     */
    @Nullable
    public Double getAdStarRating() {
        String ratingValue = nativeAdAssetMap == null ? null : nativeAdAssetMap.get(TOKEN_APP_RATING_VALUE);
        if (!TextUtils.isEmpty(ratingValue)) {
            try {
                return Double.valueOf(ratingValue);
            } catch (NumberFormatException e) {
                VungleLogger.error(true, TAG, "NativeAd", "Unable to parse " + ratingValue + " as double.");
                return null;
            }
        }

        return null;
    }

    /**
     * sponsored text
     */
    @NonNull
    public String getAdSponsoredText() {
        String ret = nativeAdAssetMap == null ? "" : nativeAdAssetMap.get(TOKEN_SPONSORED_BY);
        return ret == null ? "" : ret;
    }

    /**
     * Vungle privacy icon url
     */
    @NonNull
    String getPrivacyIconUrl() {
        String ret = nativeAdAssetMap == null ? "" : nativeAdAssetMap.get(TOKEN_VUNGLE_PRIVACY_ICON_URL);
        return ret == null ? "" : ret;
    }

    /**
     * Vungle privacy target url
     */
    @NonNull
    String getPrivacyUrl() {
        String ret = nativeAdAssetMap == null ? "" : nativeAdAssetMap.get(TOKEN_VUNGLE_PRIVACY_URL);
        return ret == null ? "" : ret;
    }

    /**
     * Call to action url
     */
    @NonNull
    String getCtaUrl() {
        String ret = nativeAdAssetMap == null ? "" : nativeAdAssetMap.get(TOKEN_CTA_BUTTON_URL);
        return ret == null ? "" : ret;
    }

    /**
     * Whether call to action url exists.
     */
    public boolean hasCallToAction() {
        return !TextUtils.isEmpty(getCtaUrl());
    }

    /**
     * Using custom root view instead of NativeAdLayout to render ad option view.
     * This API can only be used in mediation integration.
     *
     * @param adOptionsRootView for rendering our privacy icon
     */
    public void setAdOptionsRootView(FrameLayout adOptionsRootView) {
        if (VungleApiClient.WRAPPER_FRAMEWORK_SELECTED == null ||
                VungleApiClient.WRAPPER_FRAMEWORK_SELECTED == VungleApiClient.WrapperFramework.none) {
            Log.w(TAG, "You can NOT use this API to change the privacy icon parent view, " +
                    "please use NativeAdLayout as your native ad root view!");
            return;
        }

        this.adOptionsRootView = adOptionsRootView;
    }

    /**
     * Unregisters children view from NativeAd instance.
     * This method must be called when the NativeAd is no longer required or destroyed.
     */
    public void unregisterView() {
        if (adOptionsView != null && adOptionsView.getParent() != null) {
            ((ViewGroup) adOptionsView.getParent()).removeView(adOptionsView);
        }

        if (impressionTracker != null) {
            impressionTracker.clear();
        }

        if (clickableViews != null) {
            for (View view : clickableViews) {
                view.setOnClickListener(null);
            }
        } else if (contentView != null) {
            contentView.setOnClickListener(null);
        }
    }

    /**
     * Register the given view for this NativeAd to handle impressions and clicks.
     *
     * @param rootNativeView NativeAd root layout, should containing NativeAd for display
     * @param mediaView      NativeAd content view such as Image, Video
     * @param adIconView     App Icon view
     * @param clickableViews a list of all views that should handle taps event
     */
    public void registerViewForInteraction(@NonNull final NativeAdLayout rootNativeView, @NonNull MediaView mediaView,
                                           @Nullable final ImageView adIconView, @Nullable List<View> clickableViews) {
        if (!canPlayAd()) {
            playAdCallback.onError(placementId, new VungleException(VungleException.AD_UNABLE_TO_PLAY));
            return;
        }

        adState = STATE_VIEWING;

        this.rootNativeView = rootNativeView;
        this.contentView = mediaView;
        this.iconView = adIconView;
        this.clickableViews = clickableViews;

        if (adOptionsView != null) {
            adOptionsView.destroy();
        }
        adOptionsView = new NativeAdOptionsView(context);

        if (adOptionsRootView == null) {
            adOptionsRootView = rootNativeView;
        }
        adOptionsView.renderTo(this, adOptionsRootView, adConfig.getAdOptionsPosition());

        impressionTracker = new ImpressionTracker(context);

        rootNativeView.finishDisplayingAdInternal(false);

        impressionTracker.addView(adOptionsRootView, new ImpressionTracker.ImpressionListener() {
            @Override
            public void onImpression(View view) {
                rootNativeView.onImpression();
            }
        });

        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        AdRequest request = new AdRequest(placementId, AdMarkupDecoder.decode(adMarkUp), false);
        rootNativeView.register(context, this,
                serviceLocator.getService(PresentationFactory.class),
                Vungle.getEventListener(request, playAdCallback),
                adConfig,
                request);

        String mainImagePath = nativeAdAssetMap == null ? null : nativeAdAssetMap.get(TOKEN_MAIN_IMAGE);
        displayImage(mainImagePath, mediaView.getMainImage());
        if (adIconView != null) {
            String appIconPath = getAppIcon();
            displayImage(appIconPath, adIconView);
        }

        int event = NativeAdLayout.ViewEvent.CTA_CLICK;
        if (clickableViews != null && clickableViews.size() > 0) {
            for (View view : clickableViews) {
                registerClickEvent(view, event);
            }
        } else {
            registerClickEvent(mediaView, event);
        }
    }

    void registerClickEvent(@NonNull View view, @NativeAdLayout.ViewEvent final int event) {
        view.setClickable(true);
        view.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (rootNativeView != null) {
                    rootNativeView.onItemClicked(event);
                }
            }
        });
    }

    private final PlayAdCallback playAdCallback = new PlayAdCallback() {
        @Override
        public void creativeId(String creativeId) {
            if (nativeAdCallback != null) {
                nativeAdCallback.creativeId(creativeId);
            }
        }

        @Override
        public void onAdStart(String placementId) {
        }

        @Override
        public void onAdEnd(String placementId, boolean completed, boolean isCTAClicked) {
        }

        @Override
        public void onAdEnd(String placementId) {
        }

        @Override
        public void onAdClick(String placementId) {
            if (nativeAdCallback != null) {
                nativeAdCallback.onAdClick(placementId);
            }
        }

        @Override
        public void onAdRewarded(String placementId) {
        }

        @Override
        public void onAdLeftApplication(String placementId) {
            if (nativeAdCallback != null) {
                nativeAdCallback.onAdLeftApplication(placementId);
            }
        }

        @Override
        public void onError(String placementId, VungleException exception) {
            adState = STATE_ERROR;

            if (nativeAdCallback != null) {
                nativeAdCallback.onAdPlayError(placementId, exception);
            }
        }

        @Override
        public void onAdViewed(String placementId) {
            if (nativeAdCallback != null) {
                nativeAdCallback.onAdImpression(placementId);
            }
        }
    };

    void displayImage(@Nullable String path, @Nullable final ImageView iv) {
        imageLoader.displayImage(path, new ImageLoader.ImageLoaderListener() {
            @Override
            public void onImageLoaded(final Bitmap bitmap) {
                if (iv != null) {
                    uiExecutor.execute(new Runnable() {
                        @Override
                        public void run() {
                            iv.setImageBitmap(bitmap);
                        }
                    });
                }
            }
        });
    }

}
