package com.vungle.warren;

import android.annotation.TargetApi;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.util.AttributeSet;
import android.util.Log;
import android.util.Pair;
import android.view.View;
import android.widget.FrameLayout;

import com.vungle.warren.error.VungleException;
import com.vungle.warren.ui.contract.AdContract;
import com.vungle.warren.ui.contract.NativeAdContract;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

/**
 * NativeAdView is a wrapper for FrameLayout that is required to show the inline ad reporting menu for Native Ads.
 */
public class NativeAdLayout extends FrameLayout {

    private static final String TAG = NativeAdLayout.class.getSimpleName();

    private OnItemClickListener clickListener;
    private PresentationFactory presenterFactory;
    private NativeAdContract.NativePresenter presenter;
    private BroadcastReceiver broadcastReceiver;
    private AdContract.AdvertisementPresenter.EventListener listener;
    private AdRequest request;

    private final AtomicBoolean pendingStart = new AtomicBoolean(false);
    private final AtomicBoolean pendingImpression = new AtomicBoolean(false);
    private final AtomicReference<Boolean> isAdVisible = new AtomicReference<>();

    private boolean started = false;
    private boolean destroyed;

    @Nullable
    private NativeAd nativeAd;
    private Context context;

    private boolean disableRenderManagement;

    public NativeAdLayout(@NonNull Context context) {
        super(context);
        initView(context);
    }

    public NativeAdLayout(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        initView(context);
    }

    public NativeAdLayout(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initView(context);
    }

    @TargetApi(21)
    public NativeAdLayout(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        initView(context);
    }

    private void initView(@NonNull Context context) {
        this.context = context;
    }

    public void finishDisplayingAdInternal(boolean isFinishByExternalApi) {
        Log.d(TAG, "finishDisplayingAdInternal() " + isFinishByExternalApi + " " + hashCode());
        if (presenter != null) {
            int flag = AdContract.AdStopReason.IS_AD_FINISHING
                    | (isFinishByExternalApi ? AdContract.AdStopReason.IS_AD_FINISHED_BY_API : 0);
            presenter.detach(flag);
        } else if (presenterFactory != null) {
            presenterFactory.destroy();
            presenterFactory = null;
            listener.onError(new VungleException(VungleException.OPERATION_CANCELED), request.getPlacementId());
        }

        release();
    }

    public void onImpression() {
        Log.d(TAG, "onImpression() " + hashCode());
        if (presenter == null) {
            pendingImpression.set(true);
        } else {
            presenter.onProgressUpdate(1, 100);
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        Log.d(TAG, "onAttachedToWindow() " + hashCode());

        if (disableRenderManagement) {
            return;
        }

        renderNativeAd();
    }

    /**
     * ONLY USE IF {@link NativeAdLayout#disableLifeCycleManagement(boolean)} is set to TRUE
     * <p>
     * Starts rendering, must be invoked manually
     **/
    public void renderNativeAd() {
        Log.d(TAG, "renderNativeAd() " + hashCode());
        broadcastReceiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                String command = intent.getStringExtra(AdContract.AdvertisementBus.COMMAND);
                if (AdContract.AdvertisementBus.STOP_ALL.equalsIgnoreCase(command)) {
                    finishDisplayingAdInternal(false);
                } else {
                    VungleLogger.warn(NativeAdLayout.class.getSimpleName() + "#onAttachedToWindow", String.format("Receiving Invalid Broadcast: %1$s", command));
                }
            }
        };

        LocalBroadcastManager.getInstance(context).registerReceiver(broadcastReceiver, new IntentFilter(AdContract.AdvertisementBus.ACTION));

        start();
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        Log.d(TAG, "onDetachedFromWindow() " + hashCode());

        if (disableRenderManagement) {
            return;
        }

        finishNativeAd();
    }

    /**
     * ONLY USE IF {@link NativeAdLayout#disableLifeCycleManagement(boolean)} is set to TRUE
     * <p>
     * Call when this native ad is no longer needed. If {@link NativeAdLayout#disableLifeCycleManagement(boolean)}
     * is set to true, this function must be called when ad is no longer needed.
     * <p>
     * Can only be called after {@link NativeAdLayout#renderNativeAd()} is invoked first.
     */
    public void finishNativeAd() {
        Log.d(TAG, "finishNativeAd() " + hashCode());

        LocalBroadcastManager.getInstance(context).unregisterReceiver(broadcastReceiver);

        if (nativeAd != null) {
            nativeAd.destroy();
        } else {
            Log.d(TAG, "No need to destroy due to haven't played the ad.");
        }
    }

    @Override
    public void onWindowFocusChanged(boolean hasWindowFocus) {
        Log.d(TAG, "onWindowFocusChanged() hasWindowFocus=" + hasWindowFocus + " " + hashCode());
        super.onWindowFocusChanged(hasWindowFocus);

        setAdVisibility(hasWindowFocus);

        if (presenter != null && !started) {
            start();
        }
    }

    @Override
    protected void onVisibilityChanged(@NonNull View changedView, int visibility) {
        super.onVisibilityChanged(changedView, visibility);
        Log.d(TAG, "onVisibilityChanged() visibility=" + visibility + " " + hashCode());
        setAdVisibility(visibility == VISIBLE);
    }

    @Override
    protected void onWindowVisibilityChanged(int visibility) {
        super.onWindowVisibilityChanged(visibility);
        Log.d(TAG, "onWindowVisibilityChanged() visibility=" + visibility + " " + hashCode());
        setAdVisibility(visibility == VISIBLE);
    }

    private void setAdVisibility(boolean isVisible) {
        if (presenter != null) {
            presenter.setAdVisibility(isVisible);
        } else {
            isAdVisible.set(isVisible);
        }
    }

    private void start() {
        Log.d(TAG, "start() " + hashCode());
        if (presenter == null) {
            pendingStart.set(true);
        } else if (!started && hasWindowFocus()) {
            presenter.start();
            started = true;
        }
    }

    /**
     * For most uses cases when you retrieve a NativeAdLayout view, there are no additional methods or calls
     * that you will need to invoke for the NativeAdLayout to display an ad and release resources when it is
     * no longer needed. The NativeAdLayout in this case is in a MANAGED state, which is the default state.
     *
     * In special use cases, the mechanisms that NativeAdLayout uses to manage creation and teardown will not
     * always automatically work. For these cases, you will need to disable lifecycle management and handle it
     * manually. In theses cases setting true will set NativeAdLayout in an UN-MANAGED state.
     *
     * These cases include embedding a NativeAdLayout inside a recyclerview or listview. If not manually managed
     * ads will not correctly display or be served when the view is recycled.
     *
     *
     * {@link NativeAdLayout#renderNativeAd()} ()}} - invoked when the native ad should be rendered.
     * {@link NativeAdLayout#finishNativeAd()} ()} - invoked when the native ad is no longer needed.
     * {@link NativeAdLayout#setAdVisibility(boolean)} - call when native ad is no longer visible on screen
     *
     * @param disable - true or false
     */
    public void disableLifeCycleManagement(boolean disable) {
        //disable onAttached and onDetached Logic
        disableRenderManagement = disable;
    }

    public void register(@NonNull Context context,
                         @NonNull NativeAd nativeAd,
                         @NonNull PresentationFactory presentationFactory,
                         @NonNull AdContract.AdvertisementPresenter.EventListener eventListener,
                         @Nullable AdConfig adConfig,
                         @NonNull final AdRequest request) {
        this.presenterFactory = presentationFactory;
        this.listener = eventListener;
        this.request = request;
        this.nativeAd = nativeAd;
        if (presenter == null) {
            presentationFactory.getNativeViewPresentation(context, this, request, adConfig,
                    new PresentationFactory.NativeViewCallback() {

                        @Override
                        public void onResult(@NonNull Pair<NativeAdContract.NativeView, NativeAdContract.NativePresenter> result,
                                             @Nullable VungleException error) {
                            presenterFactory = null;
                            if (error != null) {
                                if (listener != null) {
                                    listener.onError(error, request.getPlacementId());
                                }
                                return;
                            }

                            NativeAdContract.NativeView nativeView = result.first;
                            presenter = result.second;
                            presenter.setEventListener(listener);
                            presenter.attach(nativeView, null);

                            if (pendingStart.getAndSet(false)) {
                                start();
                            }

                            if (pendingImpression.getAndSet(false)) {
                                presenter.onProgressUpdate(1, 100);
                            }

                            if (isAdVisible.get() != null) {
                                setAdVisibility(isAdVisible.get());
                            }

                            destroyed = false;
                        }
                    });
        }
    }

    public void setOnItemClickListener(OnItemClickListener clickListener) {
        this.clickListener = clickListener;
    }

    public void onItemClicked(@ViewEvent int code) {
        if (clickListener != null) {
            clickListener.onItemClicked(code);
        }
    }

    public void release() {
        if (destroyed)
            return;
        destroyed = true;
        presenter = null;
        presenterFactory = null;
    }

    public interface OnItemClickListener {
        void onItemClicked(@ViewEvent int code);
    }

    @IntDef(value = {ViewEvent.CTA_CLICK, ViewEvent.PRIVACY_CLICK})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ViewEvent {
        int CTA_CLICK = 1;
        int PRIVACY_CLICK = 2;
    }

}
