package com.vungle.warren;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.PriorityQueue;
import java.util.concurrent.atomic.AtomicInteger;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Wrapper around {@link PriorityQueue} to keep {@link com.vungle.warren.AdLoader.Operation}
 * order on reinserting.
 */
class OperationSequence {

    static class Entry {
        private static final AtomicInteger seq = new AtomicInteger();
        private final int order = seq.incrementAndGet();

        @NonNull
        AdLoader.Operation operation;

        Entry(@NonNull AdLoader.Operation operation) {
            this.operation = operation;
        }
    }

    private final PriorityQueue<Entry> queue = new PriorityQueue<>(11, new Comparator<Entry>() {
        //keep FIFO for Entry with same priority
        @Override
        public int compare(Entry e1, Entry e2) {
            int result = ((Integer) (e1.operation.priority)).compareTo(e2.operation.priority);
            if (result == 0) {
                return ((Integer) (e1.order)).compareTo(e2.order);
            } else {
                return result;
            }
        }
    });

    /**
     * Removes Existing entry in the queue or returns {@code null} if element is not found
     *
     * @param request {@link com.vungle.warren.AdLoader.Operation#request}
     * @return removed {@link Entry} entry
     */
    @Nullable
    Entry remove(AdRequest request) {
        Entry entry = null;
        for (Entry op : queue) {
            if (op.operation.request.equals(request)) {
                entry = op;
                break;
            }
        }
        queue.remove(entry);
        return entry;
    }

    boolean offer(Entry entry) {
        return queue.offer(entry);
    }

    Entry poll() {
        return queue.poll();
    }

    /**
     * Remove all operations on queue.
     *
     * @return List of {@link Entry}
     */
    List<Entry> removeAll() {
        ArrayList<Entry> ops = new ArrayList<>();
        while (!queue.isEmpty()) {
            Entry e = queue.poll();
            if (e != null) {
                ops.add(e);
            }
        }
        return ops;
    }
}
