package com.vungle.warren;

import com.vungle.warren.error.VungleException;

/**
 * Callback handler for playing a Vungle advertisement. This is given as a parameter to {@link Vungle#playAd(String, AdConfig, PlayAdCallback)}
 * and is triggered when the advertisement begins to play, when the advertisement ends, and when
 * any errors occur.
 */
public interface PlayAdCallback {

    /**
     * Called when the ad is just acquired and its creative ID can be queried. The first callback
     * to be notified
     * @param creativeId The Creative Id of the advertisement
     */
    void creativeId(String creativeId);

    /**
     * Called when the Vungle SDK has successfully launched the advertisement
     * and an advertisement will begin playing momentarily.
     *
     * @param placementId The Placement ID of the advertisement being played.
     */
    void onAdStart(String placementId);

    /**
     * Callback for an advertisement ending. The Vungle SDK has finished playing the advertisement and
     * the user has closed the advertisement.
     *
     * @param placementId The Placement ID of the advertisement that ended.
     * @param completed   Flag that indicates whether or not the user watched the advertisement to completion.
     * @deprecated Replaced by {@link #onAdEnd(String)} and {@link #onAdClick(String)} and
     * {@link #onAdRewarded(String)}
     */
    @Deprecated
    void onAdEnd(String placementId, boolean completed, boolean isCTAClicked);

    /**
     * Callback for an advertisement ending. The Vungle SDK has finished playing the advertisement and
     * the user has closed the advertisement.
     *
     * @param placementId The Placement ID of the advertisement that ended.
     */
    void onAdEnd(String placementId);

    /**
     * Callback for an advertisement tapped. Sent when the user has tapped on an ad.
     *
     * @param placementId The Placement ID of the advertisement that tapped.
     */
    void onAdClick(String placementId);

    /**
     * Callback for the user has watched the advertisement to completion. The Vungle SDK has finished playing the
     * advertisement and the user has closed the advertisement.
     *
     * @param placementId The Placement ID of the advertisement that ended.
     */
    void onAdRewarded(String placementId);

    /**
     * Callback when the user has left the app.
     *
     * @param placementId The Placement ID of the advertisement that tapped.
     */
    void onAdLeftApplication(String placementId);

    /**
     * Callback for an error that has occurred while playing an advertisement. If this is called, the
     * error was unrecoverable by the SDK and error handling should happen at the application layer.
     * If this callback is triggered, {@link #onAdStart(String)} and {@link #onAdEnd(String, boolean, boolean)} will
     * not be called. This indicates that the advertisement has finished.
     *
     * @param placementId The identifier for the advertisement placement for which the error occurred.
     * @param exception   The exception that prevented the advertisement from playing.
     */
    void onError(String placementId, VungleException exception);

    /**
     * Called when the ad is first rendered, please use this callback to track views.
     *
     * @param placementId The Placement Id of the advertisement shown
     */
    void onAdViewed(String placementId);
}
