package com.vungle.warren;

import androidx.annotation.VisibleForTesting;

import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.AnalyticUrl;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.CookieUtil;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Class to Get/Update privacy statuses
 */
public class PrivacyManager {

    private Repository repository;

    /**
     * COPPA Status stored from publisher
     */
    private final static AtomicReference<Boolean> coppaStatus = new AtomicReference<>();
    private final static AtomicReference<Boolean> disableAdId = new AtomicReference<>();

    private static PrivacyManager sInstance;

    private ExecutorService backgroundExecutor;

    public enum COPPA {
        COPPA_ENABLED(true),
        COPPA_DISABLED(false),
        COPPA_NOTSET(null);

        private Boolean value;

        COPPA(Boolean value) {
            this.value = value;
        }

        public boolean getValue() {
            if (value == null) {
                return true;
            }
            return this.value;
        }
    }

    private PrivacyManager() {
    }

    protected static synchronized PrivacyManager getInstance(){
        if (sInstance == null) {
            sInstance = new PrivacyManager();
        }
        return sInstance;
    }

    protected synchronized void init(ExecutorService backgroundExecutor, Repository repository) {
        this.repository = repository;
        this.backgroundExecutor = backgroundExecutor;
        //update in memory values
        Boolean storedCoppaValue = CookieUtil.getBoolean(repository, Cookie.COPPA_COOKIE, Cookie.COPPA_STATUS_KEY);
        if (coppaStatus.get() != null) {
            updateCoppaStatus(coppaStatus.get());
        } else if (storedCoppaValue != null) {
            coppaStatus.set(storedCoppaValue);
        }
    }

    protected void updateCoppaStatus(final Boolean newValue) {
        if (newValue != null) {
            coppaStatus.set(newValue);
            if (repository != null && backgroundExecutor != null) {
                backgroundExecutor.execute(new Runnable() {
                    @Override
                    public void run() {
                        CookieUtil.update(repository, Cookie.COPPA_COOKIE, Cookie.COPPA_STATUS_KEY, newValue);
                    }
                });
            }
        }
    }

    protected COPPA getCoppaStatus() {
        if (coppaStatus == null || coppaStatus.get() == null) {
            return COPPA.COPPA_NOTSET;
        } else if (coppaStatus.get() == true) {
            return COPPA.COPPA_ENABLED;
        } else if (coppaStatus.get() == false) {
            return COPPA.COPPA_DISABLED;
        }
        return COPPA.COPPA_NOTSET;
    }

    protected void updateDisableAdId(boolean newValue) {
        disableAdId.set(newValue);

        if (repository == null) {
            return;
        }

        Boolean previousDisableId = CookieUtil.getBoolean(repository, Cookie.COPPA_COOKIE, Cookie.COPPA_DISABLE_AD_ID);
        if ((previousDisableId == null || previousDisableId == false) && newValue) {
            deleteItemsWithAdIds();
        }

        //Need to store value to know if we need to delete cached ads later (if previous = false, current is true)
        //We need to delete cached ads
        CookieUtil.update(repository, Cookie.COPPA_COOKIE, Cookie.COPPA_DISABLE_AD_ID, newValue);
    }

    protected boolean shouldSendAdIds() {
        if (disableAdId == null || disableAdId.get() == null) {
            return false;
        }
        return !disableAdId.get();
    }

    //Use only for Testing
    @VisibleForTesting
    protected void clean() {
        disableAdId.set(null);
        coppaStatus.set(null);
    }

    private void deleteItemsWithAdIds() {
        //delete ads, cleanup job will take care of asset cache after
        repository.deleteAll(Advertisement.class);
        //delete analytic urls
        repository.deleteAll(AnalyticUrl.class);
    }

}
