package com.vungle.warren;

import static com.vungle.warren.AdConfig.AUTO_ROTATE;
import static com.vungle.warren.AdConfig.LANDSCAPE;
import static com.vungle.warren.AdConfig.MATCH_VIDEO;
import static com.vungle.warren.AdConfig.PORTRAIT;

import android.util.Log;

import androidx.annotation.VisibleForTesting;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.network.Response;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionConstants;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.utility.ActivityManager;
import com.vungle.warren.utility.HashUtility;
import com.vungle.warren.utility.ListUtility;
import com.vungle.warren.utility.UtilityResource;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Class used to track app sessions, sdk events to be sent to bi
 */
public class SessionTracker {

    private static final String TAG = SessionTracker.class.getSimpleName();

    private static SessionTracker _instance;
    private static long initTimestamp;

    private UtilityResource utilityResource;

    private ExecutorService sessionDataExecutor;

    /**
     * By default is not enabled
     */
    private boolean enabled = false;

    /**
     * Timeout to measure for AppSession Tracking
     */
    private long appSessionTimeout;

    private SessionCallback sessionCallback;

    private final List<SessionData> pendingEvents = Collections.synchronizedList(new ArrayList<SessionData>());

    private final List<String> placementLoadTracker = new ArrayList<>();

    private final Map<String, SessionData> customVideoCacheMap = new HashMap<>();

    private VungleApiClient vungleApiClient;

    private int sendLimit = MAX_EVENTS_PER_REPORT;

    private AtomicInteger eventCount = new AtomicInteger();

    private static final int MAX_EVENTS_PER_REPORT = 40;

    private int initCounter;

    private Repository repository;

    private SessionTracker() {
    }

    protected void init(SessionCallback sessionCallback,
                        UtilityResource utilityResource,
                        final Repository repository,
                        ExecutorService sessionDataExecutor,
                        VungleApiClient vungleApiClient,
                        final boolean enabled, final int batchSizeLimit) {
        this.sessionCallback = sessionCallback;
        this.utilityResource = utilityResource;
        this.sessionDataExecutor = sessionDataExecutor;
        this.repository = repository;
        this.enabled = enabled;
        this.vungleApiClient = vungleApiClient;

        sendLimit = batchSizeLimit <= 0 ? MAX_EVENTS_PER_REPORT : batchSizeLimit;

        if (!enabled) {
            clearTracking();
            return;
        }

        // Check for existing events
        // While searching for events do not allow new events to be tracked
        // create a lock
        sessionDataExecutor.submit(new Runnable() {
            @Override
            public void run() {

                if (!pendingEvents.isEmpty() && enabled) {
                    for (SessionData event: pendingEvents) {
                        trackEvent(event);
                    }
                }
                pendingEvents.clear();

                List<SessionData> existingEvents = repository.loadAll(SessionData.class).get();
                List<List<SessionData>> batchedSessionData = ListUtility.partition(
                        existingEvents,
                        sendLimit);
                for (List<SessionData> data: batchedSessionData) {
                    if (data.size() >= sendLimit) {
                        try {
                            sendData(data);
                        } catch (DatabaseHelper.DBException e) {
                            Log.e(TAG, "Unable to retrieve data to send " + e.getLocalizedMessage());
                        }
                    } else {
                        // its the last block
                        eventCount.set(data.size());
                    }
                }
            }
        });
    }

    @VisibleForTesting
    protected int getSendLimit() {
        return sendLimit;
    }

    public void setInitTimestamp(long millis) {
        initTimestamp = millis;
    }

    public static SessionTracker getInstance() {
        if (_instance == null) {
            _instance = new SessionTracker();
        }
        return _instance;
    }

    public long getInitTimestamp() {
        return initTimestamp;
    }

    public void trackAdConfig(BannerAdConfig adConfig) {
        if (adConfig != null && adConfig.muteChangedByApi) {
            trackEvent(new SessionData.Builder().setEvent(SessionEvent.MUTE)
                    .addData(SessionAttribute.MUTED, (adConfig.getSettings() & AdConfig.MUTED) == 1? true : false)
                    .build());
        }
    }


    public void trackAdConfig(AdConfig adConfig) {
        if (adConfig != null && adConfig.muteChangedByApi) {
            trackEvent(new SessionData.Builder().setEvent(SessionEvent.MUTE)
                    .addData(SessionAttribute.MUTED, (adConfig.getSettings() & AdConfig.MUTED) == 1? true : false)
                    .build());
        }

        if (adConfig != null && adConfig.orientationChangedByApi) {
            trackEvent(new SessionData.Builder()
                    .setEvent(SessionEvent.ORIENTATION)
                    .addData(SessionAttribute.ORIENTATION, getOrientation(adConfig.getAdOrientation()))
                    .build());
        }
    }

    //possible to be invoked from different threads
    public synchronized void trackEvent(final SessionData sessionData) {
        if (sessionData == null) {
            return;
        }
        if (!enabled) {
            //store pending events
            pendingEvents.add(sessionData);
            return;
        }

        if (!handleCustomRules(sessionData)) {
            storeEvent(sessionData);
        }
    }

    private synchronized void storeEvent(final SessionData sessionData) {
        if (sessionDataExecutor == null) {
            return;
        }
        sessionDataExecutor.submit(new Runnable() {
            @Override
            public void run() {
                try {
                    if (repository == null || sessionData == null) {
                        return;
                    }
                    repository.save(sessionData);
                    eventCount.incrementAndGet();
                    Log.d(TAG, "Session Count: " + eventCount + " " + sessionData.sessionEvent);
                    if (eventCount.get() >= sendLimit) {
                        sendData(repository.loadAll(SessionData.class).get());
                        Log.d(TAG, "SendData " + eventCount);
                    }
                } catch (DatabaseHelper.DBException e) {
                    VungleLogger.error(SessionTracker.TAG, "Could not save event to DB");
                }
            }
        });
    }

    protected void clearTracking() {
        pendingEvents.clear();
    }

    /**
     * Custom logic since product only wants to track placement loading events for explicit API calls
     * This method exists so that we don't keep any custom logic in AdRequest for how an event was tracked
     *
     * Additionally have to handle checking for video cached if CDN or CACHED
     *
     * @return true if event is to be skipped
     */
    protected synchronized boolean handleCustomRules(SessionData sessionData) {
        //Ensure that init end comes in pairs
        if (SessionEvent.INIT == sessionData.sessionEvent) {
            initCounter++;
            return false;
        }

        if (SessionEvent.INIT_END == sessionData.sessionEvent) {
            if (initCounter > 0) {
                initCounter--;
                return false;
            }
            return true;
        }

        //Allow explicit load ads
        if (SessionEvent.LOAD_AD == sessionData.sessionEvent) {
            placementLoadTracker.add(sessionData.getStringAttribute(SessionAttribute.PLACEMENT_ID));
            return false;
        }

        //Only allow load ad ends that have been explicitly been called
        if (SessionEvent.LOAD_AD_END == sessionData.sessionEvent) {
            if (placementLoadTracker.contains(sessionData.getStringAttribute(SessionAttribute.PLACEMENT_ID))) {
                placementLoadTracker.remove(sessionData.getStringAttribute(SessionAttribute.PLACEMENT_ID));
                return false;
            }
            return true;
        }

        // Only save the video and event id just for cached video assets
        // In order to tie an event id from ad loader and the assetdownloader call (which tells us whether asset is from CDN or Cache)
        // We have to call ADS_CACHED twice and tie both events into a single event
        if (SessionEvent.ADS_CACHED == sessionData.sessionEvent) {
            if (sessionData.getStringAttribute(SessionAttribute.VIDEO_CACHED) == null) {
                customVideoCacheMap.put(sessionData.getStringAttribute(SessionAttribute.URL), sessionData);
            } else {
                SessionData data = customVideoCacheMap.get(sessionData.getStringAttribute(SessionAttribute.URL));
                if (data != null) {
                    customVideoCacheMap.remove(sessionData.getStringAttribute(SessionAttribute.URL));
                    sessionData.removeEvent(SessionAttribute.URL);
                    sessionData.addAttribute(SessionAttribute.EVENT_ID, data.getStringAttribute(SessionAttribute.EVENT_ID));
                    return false;
                }
                // Should sent event when an ad has no file asset cache. Can happen for programmatic
                // ads.
                return !sessionData.getStringAttribute(SessionAttribute.VIDEO_CACHED).equals(SessionConstants.NONE);
            }
            return true;
        }

        return false;
    }

    /**
     * @return timeout returned by server for when we should reset the ordinal count. This is
     * the amount of time that an app is in background after user returns from app
     */
    public long getAppSessionTimeout() {
        return appSessionTimeout;
    }

    public void setAppSessionTimeout(long appSessionTimeout) {
        this.appSessionTimeout = appSessionTimeout;
    }

    private synchronized void sendData(List<SessionData> items) throws DatabaseHelper.DBException {
        if (!enabled || items.isEmpty()) {
            return;
        }

        JsonArray data = new JsonArray();
        for (SessionData sessionData : items) {
            JsonElement element = JsonParser.parseString(sessionData.getAsJsonString());
            if (element != null && element.isJsonObject()) {
                data.add(element.getAsJsonObject());
            }
        }

        try {
            Response<JsonObject> sendAnalyticsResponse =
                    vungleApiClient.sendSessionDataAnalytics(data)
                            .execute();
            for (SessionData sessionData : items) {
                if (sendAnalyticsResponse.isSuccessful() ||
                        sessionData.getSendAttempts() >= sendLimit) {
                    repository.delete(sessionData);
                } else {
                    sessionData.incrementSendAttempt();
                    repository.save(sessionData);
                }
            }
        } catch (IOException e) {
            Log.e(TAG, "Sending session analytics failed "  + e.getLocalizedMessage());
        }

        eventCount.set(0);
    }

    public void observe() {
        ActivityManager.getInstance().addListener(appLifeCycleCallback);
    }

    @VisibleForTesting
    public ActivityManager.LifeCycleCallback appLifeCycleCallback = new ActivityManager.LifeCycleCallback() {

        private long lastStoppedTimestamp;

        @Override
        public void onStart() {
            if (lastStoppedTimestamp <= 0) {
                return;
            }

            long duration = utilityResource.getSystemTimeMillis() - lastStoppedTimestamp;
            if (getAppSessionTimeout() > -1 &&
                    duration > 0 &&
                    duration >= getAppSessionTimeout() * 1000 &&
                    sessionCallback != null) {
                sessionCallback.onSessionTimeout();
            }
            trackEvent(new SessionData.Builder().setEvent(SessionEvent.APP_FOREGROUND).build());
        }

        @Override
        public void onStop() {
            trackEvent(new SessionData.Builder().setEvent(SessionEvent.APP_BACKGROUND).build());
            lastStoppedTimestamp = utilityResource.getSystemTimeMillis();
        }
    };

    public String getOrientation(int orientation) {
        switch (orientation) {
            case PORTRAIT:
                return "portrait";
            case AUTO_ROTATE:
                return "auto_rotate";
            case LANDSCAPE:
                return "landscape";
            case MATCH_VIDEO:
                return "match_video";
            default:
                return "none";
        }
    }

    public interface SessionCallback {
        void onSessionTimeout();
    }

}