package com.vungle.warren;

import static androidx.core.content.PermissionChecker.checkCallingOrSelfPermission;
import static com.vungle.warren.error.VungleException.AD_PAST_EXPIRATION;
import static com.vungle.warren.error.VungleException.AD_UNABLE_TO_PLAY;
import static com.vungle.warren.error.VungleException.APPLICATION_CONTEXT_REQUIRED;
import static com.vungle.warren.error.VungleException.CONFIGURATION_ERROR;
import static com.vungle.warren.error.VungleException.DB_ERROR;
import static com.vungle.warren.error.VungleException.INCORRECT_DEFAULT_API_USAGE;
import static com.vungle.warren.error.VungleException.INCORRECT_DEFAULT_API_USAGE_NATIVE;
import static com.vungle.warren.error.VungleException.INVALID_SIZE;
import static com.vungle.warren.error.VungleException.MISSING_HBP_EVENT_ID;
import static com.vungle.warren.error.VungleException.MISSING_REQUIRED_ARGUMENTS_FOR_INIT;
import static com.vungle.warren.error.VungleException.NETWORK_PERMISSIONS_NOT_GRANTED;
import static com.vungle.warren.error.VungleException.NETWORK_UNREACHABLE;
import static com.vungle.warren.error.VungleException.NO_SERVE;
import static com.vungle.warren.error.VungleException.NO_SPACE_TO_INIT;
import static com.vungle.warren.error.VungleException.OPERATION_ONGOING;
import static com.vungle.warren.error.VungleException.PLACEMENT_NOT_FOUND;
import static com.vungle.warren.error.VungleException.SDK_VERSION_BELOW_REQUIRED_VERSION;
import static com.vungle.warren.error.VungleException.SERVER_RETRY_ERROR;
import static com.vungle.warren.error.VungleException.UNKNOWN_ERROR;
import static com.vungle.warren.error.VungleException.VUNGLE_NOT_INTIALIZED;
import static com.vungle.warren.model.Advertisement.ERROR;
import static com.vungle.warren.model.Advertisement.NEW;
import static com.vungle.warren.model.Advertisement.READY;
import static com.vungle.warren.model.Cookie.CONSENT_COOKIE;
import static java.lang.Boolean.TRUE;

import android.Manifest;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.PermissionChecker;
import androidx.localbroadcastmanager.content.LocalBroadcastManager;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.vungle.warren.downloader.DownloadRequest;
import com.vungle.warren.downloader.Downloader;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.log.LogManager;
import com.vungle.warren.model.Advertisement;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.GdprCookie;
import com.vungle.warren.model.JsonUtil;
import com.vungle.warren.model.Placement;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.model.admarkup.AdMarkup;
import com.vungle.warren.network.Call;
import com.vungle.warren.network.Callback;
import com.vungle.warren.network.HttpException;
import com.vungle.warren.network.Response;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.FilePreferences;
import com.vungle.warren.persistence.FutureResult;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.tasks.AnalyticsJob;
import com.vungle.warren.tasks.CleanupJob;
import com.vungle.warren.tasks.JobInfo;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.tasks.ReconfigJob;
import com.vungle.warren.tasks.SendLogsJob;
import com.vungle.warren.tasks.SendReportsJob;
import com.vungle.warren.ui.VungleActivity;
import com.vungle.warren.ui.contract.AdContract.AdvertisementBus;
import com.vungle.warren.ui.view.VungleBannerView;
import com.vungle.warren.utility.ActivityManager;
import com.vungle.warren.utility.AdMarkupDecoder;
import com.vungle.warren.utility.Constants;
import com.vungle.warren.utility.Executors;
import com.vungle.warren.utility.SDKExecutors;
import com.vungle.warren.utility.TimeoutProvider;
import com.vungle.warren.utility.UtilityResource;
import com.vungle.warren.utility.platform.Platform;
import com.vungle.warren.vision.VisionConfig;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Main interface with the Vungle ad network.
 */
@Keep
public class Vungle {
    static final Vungle _instance = new Vungle();

    private static final String TAG = Vungle.class.getCanonicalName();

    /**
     * Consent to be maintained if publisher set the consent before calling init.
     * Also to avoid blocking call for {@link Vungle#updateConsentStatus(Consent, String)}
     */
    private final AtomicReference<Consent> consent = new AtomicReference<>();

    /**
     * Consent version to be maintained
     */
    private volatile String consentVersion;

    /**
     * CCPAStatus to be maintained if publisher set the ccpa status before calling init.
     * Also to avoid blocking call for {@link Vungle#updateCCPAStatus(Consent)}
     */
    private final AtomicReference<Consent> ccpaStatus = new AtomicReference<>();

    /**
     * State tracking field which contains true if a current placement is playing an ad.
     */
    private Map<String, Boolean> playOperations = new ConcurrentHashMap<>();

    /**
     * The current application ID. We need to store this in order to reconfigure the SDK as necessary.
     */
    volatile String appID;

    /**
     * A weak reference to the application context. Once this goes away, we can consider the SDK to be
     * dead, since the host application has lost application context. Holding a strong reference to
     * the context would be a memory leak.
     */
    private Context context;

    private static volatile boolean isInitialized;
    private static AtomicBoolean isInitializing = new AtomicBoolean(false);
    private static AtomicBoolean isDepInit = new AtomicBoolean(false);

    private static Gson gson = new GsonBuilder().create();

    //Header Bidding ordinal view count
    private AtomicInteger hbpOrdinalViewCount = new AtomicInteger(0);

    private static final int DEFAULT_SESSION_TIMEOUT = 900;

    /**
     * Private no-arg constructor to prevent instance generation of this class.
     */
    private Vungle() {
    }

    static Context getAppContext() {
        return _instance.context;
    }

    /**
     * Initialize the Vungle SDK
     * Please use {@link #init(String, Context, InitCallback)}
     * This method exists for backward compatibility. Placements pass in this method are ignored by the SDK.
     * Placements will be filled in from the dashboard to get a list of Valid Placements, call
     * {@link #getValidPlacements()} after Vungle is initialized successfully
     *
     * @param placements Placements pass here will no longer be passed into SDK. Placements will be passed in from the dashboard.
     * @param appId      The identifier for the publisher application. This can be found in the vungle
     *                   dashboard.
     * @param context    Application context.
     * @param callback   Callback that will be triggered once initialization has completed, or if any errors occur.
     * @throws IllegalArgumentException if InitCallback is null
     */
    @Deprecated
    public static void init(@NonNull final Collection<String> placements,
                            @NonNull String appId,
                            @NonNull Context context,
                            @NonNull InitCallback callback) throws IllegalArgumentException {
        init(appId, context, callback, new VungleSettings.Builder().build());
    }

    /**
     * Initialize the Vungle SDK
     *
     * @param appId    The identifier for the publisher application. This can be found in the vungle
     *                 dashboard.
     * @param context  Application context.
     * @param callback Callback that will be triggered once initialization has completed, or if any errors occur.
     * @throws IllegalArgumentException if InitCallback is null
     */
    public static void init(@NonNull final String appId,
                            @NonNull final Context context,
                            @NonNull final InitCallback callback) throws IllegalArgumentException {
        init(appId, context, callback, new VungleSettings.Builder().build());
    }

    /**
     * Initialize the Vungle SDK
     *
     * @param appId    The identifier for the publisher application. This can be found in the vungle
     *                 dashboard.
     * @param context  Application context.
     * @param callback Callback that will be triggered once initialization has completed, or if any errors occur.
     * @param settings Vungle's settings
     * @throws IllegalArgumentException if InitCallback is null
     */
    @SuppressWarnings("squid:S2583")
    public static void init(@NonNull final String appId,
                            @NonNull final Context context,
                            @NonNull final InitCallback callback,
                            @NonNull final VungleSettings settings) throws IllegalArgumentException {

        //String for conforming to Google's versioning logic, DO NOT remove or put this anywhere else
        String VUNGLE_VERSION_STRING = "!SDK-VERSION-STRING!:com.vungle:publisher-sdk-android:" + BuildConfig.VERSION_NAME;
        VungleLogger.debug("Vungle#init", "init request");

        SessionTracker.getInstance().trackEvent(
                new SessionData.Builder().setEvent(SessionEvent.INIT).build());

        if (callback == null) {
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END).addData(SessionAttribute.SUCCESS, false).build());
            throw new IllegalArgumentException("A valid InitCallback required to ensure API calls are being made after initialize is successful");
        }

        if (context == null) {
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END).addData(SessionAttribute.SUCCESS, false).build());
            callback.onError(new VungleException(MISSING_REQUIRED_ARGUMENTS_FOR_INIT));
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        final Platform platform = serviceLocator.getService(Platform.class);

        if (!platform.isAtLeastMinimumSDK()) {
            Log.e(TAG, "SDK is supported only for API versions 21 and above");
            callback.onError(new VungleException(SDK_VERSION_BELOW_REQUIRED_VERSION));
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END).addData(SessionAttribute.SUCCESS, false).build());
            return;
        }

        final RuntimeValues runtimeValues = ServiceLocator.getInstance(context).getService(RuntimeValues.class);
        runtimeValues.settings.set(settings);

        Executors sdkExecutors = serviceLocator.getService(Executors.class);

        InitCallback initCallback = callback instanceof InitCallbackWrapper
                ? callback
                : new InitCallbackWrapper(sdkExecutors.getUIExecutor(), callback);

        if ((appId == null || appId.isEmpty())) {
            initCallback.onError(new VungleException(MISSING_REQUIRED_ARGUMENTS_FOR_INIT));
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END).addData(SessionAttribute.SUCCESS, false).build());
            return;
        }

        if (!(context instanceof Application)) {
            initCallback.onError(new VungleException(APPLICATION_CONTEXT_REQUIRED));
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END).addData(SessionAttribute.SUCCESS, false).build());
            return;
        }

        if (isInitialized()) {
            Log.d(TAG, "init already complete");
            initCallback.onSuccess();
            VungleLogger.debug("Vungle#init", "init already complete");
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END).addData(SessionAttribute.SUCCESS, false).build());
            return;
        }

        if (isInitializing.getAndSet(true)) {
            Log.d(TAG, "init ongoing");
            onInitError(initCallback, new VungleException(OPERATION_ONGOING));
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END).addData(SessionAttribute.SUCCESS, false).build());
            return;
        }

        if (checkCallingOrSelfPermission(context, Manifest.permission.ACCESS_NETWORK_STATE)
                != PermissionChecker.PERMISSION_GRANTED
                || checkCallingOrSelfPermission(context, Manifest.permission.INTERNET)
                != PermissionChecker.PERMISSION_GRANTED) {
            Log.e(TAG, "Network permissions not granted");
            onInitError(initCallback, new VungleException(NETWORK_PERMISSIONS_NOT_GRANTED));
            isInitializing.set(false);
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END).addData(SessionAttribute.SUCCESS, false).build());
            return;
        }

        SessionTracker.getInstance().setInitTimestamp(System.currentTimeMillis());

        runtimeValues.initCallback.set(initCallback);

        sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {

                _instance.appID = appId;
                final InitCallback initCallback = runtimeValues.initCallback.get();
                if (!isDepInit.getAndSet(true)) {
                    LogManager logManager = serviceLocator.getService(LogManager.class);
                    VungleLogger.setupLoggerWithLogLevel(logManager, VungleLogger.LoggerLevel.DEBUG, VungleLogger.LOGGER_MAX_ENTRIES);

                    CacheManager cacheManager = serviceLocator.getService(CacheManager.class);
                    final VungleSettings settings = runtimeValues.settings.get();

                    if (settings != null && cacheManager.getBytesAvailable() < settings.getMinimumSpaceForInit()) {
                        onInitError(initCallback, new VungleException(NO_SPACE_TO_INIT));
                        deInit();
                        return;
                    }

                    cacheManager.addListener(cacheListener);

                    /// Save the context reference to track application state.
                    _instance.context = context;

                    //init first
                    Repository repository = serviceLocator.getService(Repository.class);
                    try {
                        repository.init();
                    } catch (DatabaseHelper.DBException e) {
                        onInitError(initCallback, new VungleException(DB_ERROR));
                        deInit();
                        return;
                    }

                    Executors sdkExecutors = serviceLocator.getService(Executors.class);

                    PrivacyManager.getInstance().init(sdkExecutors.getBackgroundExecutor(), repository);

                    VungleApiClient vungleApiClient = serviceLocator.getService(VungleApiClient.class);
                    vungleApiClient.init();

                    if (settings != null) {
                        platform.setAndroidIdFallbackDisabled(settings.getAndroidIdOptOut());
                    }

                    JobRunner jobRunner = serviceLocator.getService(JobRunner.class);
                    AdLoader adLoader = serviceLocator.getService(AdLoader.class);
                    adLoader.init(jobRunner);

                    //Restore the consent if publisher set it before calling init
                    if (_instance.consent.get() != null) {
                        saveGDPRConsent(repository, _instance.consent.get(), _instance.consentVersion);
                    } else {
                        //Restore it from storage
                        Cookie gdprConsent = repository.load(Cookie.CONSENT_COOKIE, Cookie.class).get();
                        if (gdprConsent == null) {
                            _instance.consent.set(null);
                            _instance.consentVersion = null;
                        } else {
                            _instance.consent.set(getConsent(gdprConsent));
                            _instance.consentVersion = getConsentMessageVersion(gdprConsent);
                        }
                    }

                    //Restore the CCPA if publisher set it before calling init
                    if (_instance.ccpaStatus.get() != null) {
                        updateCCPAStatus(repository, _instance.ccpaStatus.get());
                    } else {
                        //Restore it from storage
                        Cookie ccpaConsent = repository.load(Cookie.CCPA_COOKIE, Cookie.class).get();
                        _instance.ccpaStatus.set(getCCPAStatus(ccpaConsent));
                    }
                }
                Repository repository = serviceLocator.getService(Repository.class);
                Cookie appIdCookie = repository.load(Cookie.APP_ID, Cookie.class).get();
                if (appIdCookie == null) {
                    appIdCookie = new Cookie(Cookie.APP_ID);
                }
                appIdCookie.putValue(Constants.APP_ID, appId);
                try {
                    repository.save(appIdCookie);
                } catch (DatabaseHelper.DBException e) {
                    if (initCallback != null) {
                        onInitError(initCallback, new VungleException(DB_ERROR));
                    }
                    isInitializing.set(false);
                    return;
                }

                /// Configure the instance.
                _instance.configure(initCallback, false);

                // Send any pending TPATs
                JobRunner jobRunner = serviceLocator.getService(JobRunner.class);

                //grab all analytics urls first
                jobRunner.execute(AnalyticsJob.makeJob(AnalyticsJob.Action.RETRY_UNSENT, null,
                        null, JobInfo.NetworkType.CONNECTED));
            }
        }, new Runnable() {
            @Override
            public void run() {
                onInitError(callback, new VungleException(VungleException.OUT_OF_MEMORY));
            }
        });
    }

    private static void onInitError(InitCallback initCallback, VungleException e) {
        if (initCallback != null) {
            initCallback.onError(e);
        }
        if (e != null) {
            String exMsg = (e.getLocalizedMessage() != null && e.getLocalizedMessage().isEmpty()) ?
                    e.getLocalizedMessage() : Integer.toString(e.getExceptionCode());
            VungleLogger.error("Vungle#init", exMsg);
        }
    }

    static void reConfigure() {
        if (_instance.context == null)
            return;

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        final RuntimeValues runtimeValues = serviceLocator.getService(RuntimeValues.class);

        if (isInitialized()) {
            sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                @Override
                public void run() {
                    _instance.configure(runtimeValues.initCallback.get(), true);
                }
            }, new Runnable() {
                @Override
                public void run() {
                    onInitError(runtimeValues.initCallback.get(), new VungleException(VungleException.OUT_OF_MEMORY));
                }
            });
        } else {
            init(_instance.appID, _instance.context, runtimeValues.initCallback.get());
        }
    }

    /**
     * Request a configuration from the server. This updates the current list of placements and
     * schedules another configuration job in the future. It also has the side-effect of loading the
     * auto_cached ad if it is not downloaded currently.
     *
     * @param callback Callback that will be called when initialization has completed or failed.
     */
    private void configure(@NonNull final InitCallback callback, boolean isReconfig) {
        /// Request a configuration from the server. This happens asynchronously on the network thread.
        try {
            if (context == null)
                throw new IllegalArgumentException("Context is null");

            ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
            VungleApiClient vungleApiClient = serviceLocator.getService(VungleApiClient.class);
            vungleApiClient.setAppId(appID);
            Repository repository = serviceLocator.getService(Repository.class);
            JobRunner jobRunner = serviceLocator.getService(JobRunner.class);
            RuntimeValues runtimeValues = serviceLocator.getService(RuntimeValues.class);

            Response<JsonObject> response = vungleApiClient.config();

            if (response == null) {
                onInitError(callback, new VungleException(UNKNOWN_ERROR));
                isInitializing.set(false);
                return;
            }

            if (!response.isSuccessful()) {
                long retryAfterHeaderValue = vungleApiClient.getRetryAfterHeaderValue(response);
                if (retryAfterHeaderValue > 0) {
                    jobRunner.execute(ReconfigJob.makeJobInfo(_instance.appID).setDelay(retryAfterHeaderValue));
                    onInitError(callback, new VungleException(SERVER_RETRY_ERROR));
                    isInitializing.set(false);
                    return;
                }
                onInitError(callback, new VungleException(CONFIGURATION_ERROR));
                isInitializing.set(false);
                return;
            }

            final FilePreferences preferences = serviceLocator.getService(FilePreferences.class);

            JsonObject jsonObject = response.body();
            /// Parse out the placements
            JsonArray placementsData = jsonObject.getAsJsonArray("placements");

            if (placementsData == null) {
                onInitError(callback, new VungleException(CONFIGURATION_ERROR));
                isInitializing.set(false);
                return;
            }

            CleverCacheSettings settings = CleverCacheSettings.fromJson(jsonObject);
            Downloader downloader = serviceLocator.getService(Downloader.class);

            if (settings != null) {
                CleverCacheSettings currentCacheSettings = CleverCacheSettings.deserializeFromString(
                        preferences.getString(CleverCacheSettings.KEY_CLEVER_CACHE, null));

                boolean timestampChanged = currentCacheSettings == null ||
                        currentCacheSettings.getTimestamp() != settings.getTimestamp();

                if (!settings.isEnabled() || timestampChanged) {
                    downloader.clearCache();
                }

                downloader.setCacheEnabled(settings.isEnabled());

                preferences.put(CleverCacheSettings.KEY_CLEVER_CACHE, settings.serializeToString())
                        .apply();
            } else {
                downloader.setCacheEnabled(true);
            }

            final AdLoader adLoader = serviceLocator.getService(AdLoader.class);
//            playOperations.clearCache();
//            adLoader.clearCache();

            List<Placement> newPlacements = new ArrayList<>();
            for (JsonElement jsonElement : placementsData) {
                newPlacements.add(new Placement(jsonElement.getAsJsonObject()));
            }

            repository.setValidPlacements(newPlacements);

            if (jsonObject.has("session")) {
                JsonObject sessionData = jsonObject.getAsJsonObject("session");
                boolean sessionDataEnabled = JsonUtil.hasNonNull(sessionData, "enabled") && sessionData.get("enabled").getAsBoolean();

                int sendLimit = JsonUtil.getAsInt(sessionData, "limit", 0);

                SessionTracker.getInstance().init(
                        new SessionTracker.SessionCallback() {
                            @Override
                            public void onSessionTimeout() {
                                _instance.hbpOrdinalViewCount.set(0);
                            }
                        }, new UtilityResource(),
                        serviceLocator.getService(Repository.class),
                        serviceLocator.getService(SDKExecutors.class).getSessionDataExecutor(),
                        serviceLocator.getService(VungleApiClient.class),
                        sessionDataEnabled,
                        sendLimit
                );

                SessionTracker.getInstance().setAppSessionTimeout(JsonUtil.getAsInt(sessionData, "timeout",
                        DEFAULT_SESSION_TIMEOUT));
            }

            if (jsonObject.has("gdpr")) {
                // create cooking regardless of whether user is in CountryDataProtected or not
                TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);
                GdprCookie gdprCookie = new GdprCookie(repository, provider);
                JsonObject gdprJsonObject = jsonObject.getAsJsonObject("gdpr");
                gdprCookie.save(gdprJsonObject);
            }

            if (jsonObject.has("logging")) {
                LogManager logManager = serviceLocator.getService(LogManager.class);
                JsonObject attributionLogging = jsonObject.getAsJsonObject("logging");
                boolean enabled = JsonUtil.hasNonNull(attributionLogging, "enabled") ? attributionLogging.get("enabled").getAsBoolean() : LogManager.DEFAULT_LOGGING_ENABLED;
                logManager.setLoggingEnabled(enabled);
            }

            // crash
            if (jsonObject.has("crash_report")) {
                LogManager logManager = serviceLocator.getService(LogManager.class);
                JsonObject crashConfig = jsonObject.getAsJsonObject("crash_report");
                boolean enabled = JsonUtil.hasNonNull(crashConfig, "enabled") ? crashConfig.get("enabled").getAsBoolean() : LogManager.DEFAULT_CRASH_COLLECT_ENABLED;
                String filter = JsonUtil.hasNonNull(crashConfig, "collect_filter") ? crashConfig.get("collect_filter").getAsString() : LogManager.sDefaultCollectFilter;
                int batchMax = JsonUtil.hasNonNull(crashConfig, "max_send_amount") ? crashConfig.get("max_send_amount").getAsInt() : LogManager.DEFAULT_CRASH_SEND_BATCH_MAX;
                logManager.updateCrashReportConfig(enabled, filter, batchMax);
            }

            boolean cacheBustEnabled = false;
            int cacheBustInterval = 0;
            if (jsonObject.has("cache_bust")) {
                JsonObject cacheBustConfig = jsonObject.getAsJsonObject("cache_bust");
                if (cacheBustConfig.has("enabled")) {
                    cacheBustEnabled = cacheBustConfig.get("enabled").getAsBoolean();
                }
                if (cacheBustConfig.has("interval")) {
                    //interval received in seconds, need to covert to millis
                    cacheBustInterval = cacheBustConfig.get("interval").getAsInt() * 1000;
                }
            }

            Cookie configCookie = repository.load(Cookie.CONFIG_COOKIE, Cookie.class).get();
            if (configCookie == null) {
                configCookie = new Cookie(Cookie.CONFIG_COOKIE);
            }

            JsonObject adLoadOptObject = jsonObject.getAsJsonObject("ad_load_optimization");
            boolean isAdDownloadOptEnabled = JsonUtil.getAsBoolean(adLoadOptObject, "enabled",
                    AdLoader.DEFAULT_LOAD_OPTIMIZATION_ENABLED);
            adLoader.setAdLoadOptimizationEnabled(isAdDownloadOptEnabled);
            configCookie.putValue("isAdDownloadOptEnabled", isAdDownloadOptEnabled);

            if (jsonObject.has("ri")) {
                boolean isReportIncentivizedEnabled = jsonObject.getAsJsonObject("ri").get("enabled").getAsBoolean();
                configCookie.putValue("isReportIncentivizedEnabled", isReportIncentivizedEnabled);
            }

            //do not modify, default value is true
            boolean disableAdId = JsonUtil.getAsBoolean(jsonObject, "disable_ad_id", true);

            PrivacyManager.getInstance().updateDisableAdId(disableAdId);

            repository.save(configCookie);

            saveConfigExtension(repository, jsonObject);

            /// Schedule the reconfig job
            if (jsonObject.has("config")) {
                long sleep = jsonObject.getAsJsonObject("config").get("refresh_time").getAsLong();
                jobRunner.execute(ReconfigJob.makeJobInfo(appID).setDelay(sleep));
            }

            try {
                serviceLocator.getService(VisionController.class).setConfig(JsonUtil.hasNonNull(jsonObject, VisionController.VISION) ?
                        gson.fromJson(jsonObject.getAsJsonObject(VisionController.VISION), VisionConfig.class) : new VisionConfig());
            } catch (DatabaseHelper.DBException dbException) {
                Log.e(TAG, "not able to apply vision data config");
            }

            //this is earliest we think that SDK basic init is success
            isInitialized = true;

            /// Inform the publisher that initialization has succeeded.
            callback.onSuccess();
            VungleLogger.debug("Vungle#init", "onSuccess");
            isInitializing.set(false);

            SessionTracker.getInstance().observe();

            final Collection<Placement> placements = repository.loadValidPlacements().get();

            /// Clean up the asset and metadata caches
            jobRunner.execute(CleanupJob.makeJobInfo());

            /// Download assets for the auto-cached placement immediately. If assets are already
            /// available, this will do nothing except inform the publisher that the auto-cached
            /// placement is ready.
            if (placements != null) {
                final List<Placement> placementList = new ArrayList<>(placements);
                final VungleSettings vungleSettings = runtimeValues.settings.get();
                Collections.sort(placementList, new Comparator<Placement>() {
                    @Override
                    public int compare(Placement o1, Placement o2) {
                        if (vungleSettings != null) {
                            if (o1.getId().equals(vungleSettings.getPriorityPlacement())) {
                                return -1;
                            }
                            if (o2.getId().equals(vungleSettings.getPriorityPlacement())) {
                                return 1;
                            }
                        }
                        return ((Integer) o1.getAutoCachePriority()).compareTo(o2.getAutoCachePriority());
                    }
                });
                Log.d(TAG, "starting jobs for autocached advs");

                ExecutorService uiExecutor = serviceLocator.getService(Executors.class).getUIExecutor();
                uiExecutor.execute(new Runnable() {
                    @Override
                    public void run() {
                        for (final Placement placement : placementList) {
                            adLoader.loadEndlessIfNeeded(placement, placement.getAdSize(), 0, false);
                        }
                    }
                });
            }

            if (cacheBustEnabled) {
                CacheBustManager cacheBustManager = serviceLocator.getService(CacheBustManager.class);
                cacheBustManager.setRefreshRate(cacheBustInterval);
                cacheBustManager.startBust();
            }

            /// Send any pending ad reports
            jobRunner.execute(SendReportsJob.makeJobInfo(!isReconfig));

            // Send any pending logs
            jobRunner.execute(SendLogsJob.makeJobInfo());

            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder()
                            .setEvent(SessionEvent.INIT_END)
                            .addData(SessionAttribute.SUCCESS, true)
                            .build());

            /// If we have not reported this install, fire it off to the ad server
            if (!preferences.getBoolean("reported", false)) {
                vungleApiClient.reportNew().enqueue(new Callback<JsonObject>() {
                    @Override
                    public void onResponse(Call<JsonObject> call, Response<JsonObject> response) {
                        if (response.isSuccessful()) {
                            /// Save the reported state
                            preferences.put("reported", true);
                            preferences.apply();
                            Log.d(TAG, "Saving reported state to shared preferences");
                        }
                    }

                    @Override
                    public void onFailure(Call<JsonObject> call, Throwable throwable) {
                        /// Do nothing. Install will be reported on the next init.
                        /// TODO: Should this retryCount more?
                    }
                });
            }

        } catch (final Throwable throwable) {
            isInitialized = false;
            isInitializing.set(false);
            Log.e(TAG, Log.getStackTraceString(throwable));
            if (throwable instanceof HttpException) {
                onInitError(callback, new VungleException(CONFIGURATION_ERROR));
            } else if (throwable instanceof DatabaseHelper.DBException) {
                onInitError(callback, new VungleException(DB_ERROR));
            } else if (throwable instanceof UnknownHostException
                    || throwable instanceof SecurityException) {
                onInitError(callback, new VungleException(NETWORK_UNREACHABLE));
            } else {
                onInitError(callback, new VungleException(UNKNOWN_ERROR));
            }
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder().setEvent(SessionEvent.INIT_END)
                            .addData(SessionAttribute.SUCCESS, false).build());
        }
    }

    /**
     * Checks if Vungle's SDK is already in initialized state
     *
     * @return true if Vungle SDK is initialized, false otherwise.
     */
    public static boolean isInitialized() {
        // no need to check for number of placements, pubs can init for zero placements
        return isInitialized && (_instance.context != null);
    }

    /**
     * Overrides the previously-set incentivized fields which are used when warning the user if they
     * are attempting to exit an incentivized advertisement prematurely. If there are no local
     * values set, the SDK will use the values set in the Vungle Dashboard as the defaults. This method
     * must be called **before** {@link #playAd(String, AdConfig, PlayAdCallback)} is called.
     *
     * @param title        The dialog box title
     * @param body         The dialog box body text. Values longer than the space allows for will be ellipsised
     *                     at the end.
     * @param keepWatching Continue button text
     * @param close        Close button text.
     */
    public static void setIncentivizedFields(@Nullable final String userID,
                                             @Nullable final String title,
                                             final @Nullable String body,
                                             final @Nullable String keepWatching,
                                             final @Nullable String close) {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized, context is null");
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

        serviceLocator.getService(Executors.class).getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                if (!isInitialized()) {
                    Log.e(TAG, "Vungle is not initialized");
                    return;
                }

                Repository repository = serviceLocator.getService(Repository.class);

                Cookie incentivizedCookie = repository.load(Cookie.INCENTIVIZED_TEXT_COOKIE, Cookie.class).get();
                if (incentivizedCookie == null) {
                    incentivizedCookie = new Cookie(Cookie.INCENTIVIZED_TEXT_COOKIE);
                }

                String titleText = TextUtils.isEmpty(title) ? "" : title;
                String bodyText = TextUtils.isEmpty(body) ? "" : body;
                String continueText = TextUtils.isEmpty(keepWatching) ? "" : keepWatching;
                String closeText = TextUtils.isEmpty(close) ? "" : close;
                String userIdStr = TextUtils.isEmpty(userID) ? "" : userID;

                incentivizedCookie.putValue("title", titleText);
                incentivizedCookie.putValue("body", bodyText);
                incentivizedCookie.putValue("continue", continueText);
                incentivizedCookie.putValue("close", closeText);
                incentivizedCookie.putValue("userID", userIdStr);

                try {
                    repository.save(incentivizedCookie);
                } catch (DatabaseHelper.DBException e) {
                    Log.e(TAG, "Cannot save incentivized cookie", e);
                }
            }
        });
    }

    /**
     * Check if we can play an advertisement for the given placement. This method checks out file
     * system to see if there are asset files for the given placement and returns true if we have
     * assets stored which have not expired.
     *
     * @param placementId The placement identifier.
     * @return true if an advertisement can be played immediately, false otherwise.
     */
    public static boolean canPlayAd(@NonNull final String placementId) {
        return canPlayAd(placementId, null);
    }

    /**
     * Check if we can play an advertisement for the given placement. This method checks out file
     * system to see if there are asset files for the given placement and returns true if we have
     * assets stored which have not expired.
     *
     * @param placementId The placement identifier.
     * @param markup      The Ad markup.
     * @return true if an advertisement can be played immediately, false otherwise.
     */
    @SuppressWarnings("squid:S2583")
    public static boolean canPlayAd(@NonNull final String placementId, @Nullable final String markup) {
        final Context context = _instance.context;

        if (context == null) {
            Log.e(TAG, "Context is null");
            return false;
        }

        if (TextUtils.isEmpty(placementId)) {
            Log.e(TAG, "AdMarkup/PlacementId is null");
            return false;
        }

        final AdMarkup serializedAdMarkup = AdMarkupDecoder.decode(markup);
        if (markup != null && serializedAdMarkup == null) {
            Log.e(TAG, "Invalid AdMarkup");
            return false;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        FutureResult<Boolean> futureResult = new FutureResult<>(sdkExecutors.getApiExecutor()
                .submit(new Callable<Boolean>() {
                    @Override
                    public Boolean call() {

                        if (!isInitialized()) {
                            Log.e(TAG, "Vungle is not initialized");
                            return false;
                        }

                        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
                        Repository repository = serviceLocator.getService(Repository.class);

                        AdMarkup adMarkup = AdMarkupDecoder.decode(markup);
                        String eventId = adMarkup != null ? adMarkup.getEventId() : null;
                        Placement placement = repository
                                .load(placementId, Placement.class)
                                .get();
                        if (placement == null || !placement.isValid()) {
                            return false;
                        }

                        if (placement.isMultipleHBPEnabled() && eventId == null) {
                            return false;
                        }

                        final Advertisement advertisement = repository
                                .findValidAdvertisementForPlacement(placementId, eventId)
                                .get();

                        if (advertisement == null) {
                            return false;
                        }

                        if (placement.getPlacementAdType() == Placement.TYPE_VUNGLE_BANNER
                                || !AdConfig.AdSize.isDefaultAdSize(placement.getAdSize())
                                && !placement.getAdSize().equals(advertisement.getAdConfig().getAdSize())) {
                            return false;
                        }

                        return canPlayAd(advertisement);
                    }
                })
        );

        return Boolean.TRUE.equals(futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS));

    }

    static boolean canPlayAd(final Advertisement advertisement) {
        if (_instance.context == null)
            return false;


        AdLoader adLoader = ServiceLocator.getInstance(_instance.context).getService(AdLoader.class);
        return adLoader.canPlayAd(advertisement);
    }

    private static void onPlayError(String placementId, PlayAdCallback playAdCallback, VungleException e) {
        if (playAdCallback != null) {
            playAdCallback.onError(placementId, e);
        }
        if (e != null) {
            String exMsg = (e.getLocalizedMessage() != null && e.getLocalizedMessage().isEmpty()) ?
                    e.getLocalizedMessage() : Integer.toString(e.getExceptionCode());
            VungleLogger.error("Vungle#playAd", exMsg);
        }
        SessionTracker.getInstance().trackEvent(new SessionData.Builder()
                .setEvent(SessionEvent.PLAY_AD)
                .addData(SessionAttribute.SUCCESS, false)
                .build());
    }

    /**
     * Play an ad for the given placement ID. If this placement ID is valid and an advertisement is
     * ready to be played, this will cause the {@link VungleActivity} to start and the advertisement
     * will be rendered.
     *
     * @param placementId The placement identifier.
     * @param settings    Optional settings for playing the advertisement. The full list can be found
     *                    at {@link AdConfig} documentation.
     * @param callback    Optional, though strongly encouraged, event listener. This object will be
     *                    notified of the advertisement starting, ending, and any errors that occur
     *                    during rendering.
     */
    public static void playAd(@NonNull final String placementId, final AdConfig settings, @Nullable final PlayAdCallback callback) {
        playAd(placementId, null, settings, callback);
    }

    /**
     * Play an ad for the given placement ID. If this placement ID is valid and an advertisement is
     * ready to be played, this will cause the {@link VungleActivity} to start and the advertisement
     * will be rendered.
     *
     * @param placementId The placement identifier.
     * @param markup      The Ad markup.
     * @param settings    Optional settings for playing the advertisement. The full list can be found
     *                    at {@link AdConfig} documentation.
     * @param callback    Optional, though strongly encouraged, event listener. This object will be
     *                    notified of the advertisement starting, ending, and any errors that occur
     *                    during rendering.
     */
    public static void playAd(@NonNull final String placementId, @Nullable final String markup, final AdConfig settings, @Nullable final PlayAdCallback callback) {
        VungleLogger.debug("Vungle#playAd", "playAd call invoked");

        SessionTracker.getInstance().trackAdConfig(settings);

        if (!isInitialized()) {
            Log.e(TAG, "Locator is not initialized");
            if (callback != null) {
                onPlayError(placementId, callback, new VungleException(VUNGLE_NOT_INTIALIZED));
            }
            return;
        }

        if (TextUtils.isEmpty(placementId)) {
            onPlayError(placementId, callback, new VungleException(PLACEMENT_NOT_FOUND));
            return;
        }

        final AdMarkup serializedAdMarkup = AdMarkupDecoder.decode(markup);
        if (markup != null && serializedAdMarkup == null) {
            onPlayError(placementId, callback, new VungleException(MISSING_HBP_EVENT_ID));
            return;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

        final Executors sdkExecutors = serviceLocator.getService(Executors.class);
        final Repository repository = serviceLocator.getService(Repository.class);
        final AdLoader adLoader = serviceLocator.getService(AdLoader.class);
        final VungleApiClient vungleApiClient = serviceLocator.getService(VungleApiClient.class);

        final PlayAdCallback listener = new PlayAdCallbackWrapper(sdkExecutors.getUIExecutor(), callback);
        final Runnable OOMRunnable = new Runnable() {
            @Override
            public void run() {
                onPlayError(placementId, listener, new VungleException(VungleException.OUT_OF_MEMORY));
            }
        };

        sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                AdMarkup adMarkup = AdMarkupDecoder.decode(markup);
                final AdRequest request = new AdRequest(placementId, adMarkup, false);
                //Don't allow Ad to play when either loading or playing
                if (TRUE.equals(_instance.playOperations.get(placementId)) || adLoader.isLoading(request)) {
                    onPlayError(placementId, listener, new VungleException(OPERATION_ONGOING));
                    return;
                }

                final Placement placement = repository.load(placementId, Placement.class).get();
                if (placement == null) {
                    onPlayError(placementId, listener, new VungleException(PLACEMENT_NOT_FOUND));
                    return;
                }

                if (AdConfig.AdSize.isBannerAdSize(placement.getAdSize())) {
                    Log.e(TAG, "Incorrect API for Banners and MREC");
                    onPlayError(placementId, listener, new VungleException(INVALID_SIZE));
                    return;
                }

                // Check if an Ad is valid
                boolean streamingOnly = false;

                // Check if an Ad for this placement exists expired or not.
                Advertisement advertisement =
                        repository.findPotentiallyExpiredAd(placementId, request.getEventId()).get();


                try {
                    long currentTime = System.currentTimeMillis() / 1000L;
                    boolean hasExpired = (advertisement != null) && (advertisement.getExpireTime() < currentTime);

                    if (!canPlayAd(advertisement) || hasExpired) {
                        // Even if we don't have a cached Ad prepared, we can still play a streaming Ad if available.
                        streamingOnly = true;

                        if (advertisement != null && (advertisement.getState() == READY || hasExpired)) {
                            //assets were somehow deleted
                            repository.saveAndApplyState(advertisement, placementId, ERROR);
                            adLoader.loadEndlessIfNeeded(placement, placement.getAdSize(), 0, false);
                        }

                        if (hasExpired) {
                            SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.AD_EXPIRED)
                                    .addData(SessionAttribute.EVENT_ID, advertisement.getId()).build());

                            onPlayError(
                                    request.getPlacementId(),
                                    listener,
                                    new VungleException(AD_PAST_EXPIRATION)
                            );
                            return;
                        }

                    } else {
                        // Ad we have cached is valid and can be used, apply the settings and save them
                        advertisement.configure(settings);
                        repository.save(advertisement);
                    }
                } catch (DatabaseHelper.DBException ignored) {
                    onPlayError(placementId, listener, new VungleException(DB_ERROR));
                    return;
                }

                if (_instance.context != null) {
                    /// Inform the Ad Server that we are about to play an Ad.
                    /// If there is a better Ad available, it will be substituted here.
                    final boolean finalStreamingOnly = streamingOnly;
                    final Advertisement finalAdvertisement = advertisement;

                    if (vungleApiClient.canCallWillPlayAd()) {
                        vungleApiClient.willPlayAd(placement.getId(), placement.isAutoCached(),
                                streamingOnly ? "" : advertisement.getAdToken()).enqueue(new Callback<JsonObject>() {

                            @Override
                            public void onResponse(Call<JsonObject> call, final Response<JsonObject> response) {
                                //Lets do it in separate thread, since there are few IO calls involved.
                                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                                    @Override
                                    public void run() {
                                        Advertisement streamingAd = null;
                                        if (response.isSuccessful()) {
                                            JsonObject responseBody = response.body();
                                            if (responseBody != null && responseBody.has("ad")) {
                                                try {
                                                    JsonObject adJson = responseBody.getAsJsonObject("ad");
                                                    streamingAd = new Advertisement(adJson);

                                                    /// Update the settings for this advertisement
                                                    streamingAd.configure(settings);

                                                    /// If we make it here, it means that there is a replacement streaming ad for
                                                    /// this placement. So we update the metadata and save it so the activity can
                                                    /// use the fresh values.
                                                    repository.saveAndApplyState(streamingAd, placementId, NEW);
                                                } catch (IllegalArgumentException e) {
                                                    VungleLogger.debug("Vungle#playAd", "streaming ads IllegalArgumentException");
                                                    Log.v(TAG, "Will Play Ad did not respond with a replacement. Move on.");
                                                } catch (Exception e) {
                                                    VungleLogger.error("Vungle#playAd", "streaming ads Exception :" + e.getLocalizedMessage());
                                                    Log.e(TAG, "Error using will_play_ad!", e);
                                                }
                                            }
                                        }

                                        if (finalStreamingOnly) {
                                            if (streamingAd == null) {
                                                onPlayError(placementId, listener, new VungleException(NO_SERVE));
                                            } else {
                                                renderAd(request, listener, placement, streamingAd);
                                            }
                                        } else {
                                            renderAd(request, listener, placement, finalAdvertisement);
                                        }
                                    }
                                }, OOMRunnable);
                            }

                            @Override
                            public void onFailure(Call<JsonObject> call, Throwable throwable) {
                                //Lets do it in separate thread, since there are few IO calls involved.
                                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                                    @Override
                                    public void run() {
                                        if (finalStreamingOnly) {
                                            onPlayError(placementId, listener, new VungleException(NO_SERVE));
                                        } else {
                                            renderAd(request, listener, placement, finalAdvertisement);
                                        }
                                    }
                                }, OOMRunnable);
                            }
                        });
                    } else {
                        if (finalStreamingOnly) {
                            onPlayError(placementId, listener, new VungleException(NO_SERVE));
                        } else {
                            renderAd(request, listener, placement, finalAdvertisement);
                        }
                    }
                }
            }
        }, OOMRunnable);
    }

    /**
     * Private helper method to reduce callback hell, this method starts playing the given advertisement
     * in a {@link VungleActivity}. It also creates an event listener in order to control the flow
     * of events once the advertisement ends.
     *
     * @param request       The Ad request identifier
     * @param listener      The optional listener for playback events
     * @param placement     The placement in which the advertisement is being rendered
     * @param advertisement The advertisement metadata for the advertisement that will be played.
     */
    private static synchronized void renderAd(@NonNull final AdRequest request,
                                              @Nullable final PlayAdCallback listener,
                                              final Placement placement,
                                              final Advertisement advertisement) {
        /// Subscribe to the event bus of the activity before starting the activity, otherwise
        /// the Publisher notices it has no subscribers and does not emit the start value.
        if (!isInitialized()) {
            Log.e(TAG, "Sdk is not initialized");
            return;
        }
        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

        VungleActivity.setEventListener(new AdEventListener(
                request,
                _instance.playOperations,
                listener,
                serviceLocator.getService(Repository.class),
                serviceLocator.getService(AdLoader.class),
                serviceLocator.getService(JobRunner.class),
                serviceLocator.getService(VisionController.class),
                placement,
                advertisement
        ) {
            @Override
            protected void onFinished() {
                super.onFinished();
                VungleActivity.setEventListener(null);
            }
        });

        /// Start the activity, and if there are any extras that have been overridden by the application, apply them.
        Intent intent = AdActivity.createIntent(_instance.context, request);

        ActivityManager.startWhenForeground(_instance.context, null, intent, null);
    }

    /**
     * Get ad event listener for native ads.
     *
     * @param request  the native ad request
     * @param listener the playback listener
     * @return event listener
     */
    static AdEventListener getEventListener(@NonNull AdRequest request, @Nullable PlayAdCallback listener) {
        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        return new AdEventListener(
                request,
                _instance.playOperations,
                listener,
                serviceLocator.getService(Repository.class),
                serviceLocator.getService(AdLoader.class),
                serviceLocator.getService(JobRunner.class),
                serviceLocator.getService(VisionController.class),
                null,
                null
        );
    }

    private static void onLoadError(String placementId, @Nullable LoadAdCallback loadAdCallback, VungleException e) {
        if (loadAdCallback != null) {
            loadAdCallback.onError(placementId, e);
        }
        if (e != null) {
            String exMsg = (e.getLocalizedMessage() != null && e.getLocalizedMessage().isEmpty()) ?
                    e.getLocalizedMessage() : Integer.toString(e.getExceptionCode());
            VungleLogger.error("Vungle#loadAd", exMsg);
        }
    }

    /**
     * Request the Vungle SDK to load the assets for an advertisement by the placement identifier.
     * This will cause us to request an ad from the Ad Server, and if the bid is filled, the assets
     * will be downloaded. It is possible for the bid to get no responses and therefore no assets
     * will be loaded. In this case, the SDK will automatically retry at a later time, specified by
     * the Vungle Server. The callback will be notified that the assets are pending download.
     *
     * @param placementId The placement identifier for which assets should be loaded.
     * @param callback    The optional callback which will be notified when the assets are loaded, or
     *                    if they are deferred. Errors will also be sent through this callback.
     */
    public static void loadAd(@NonNull final String placementId, @Nullable final LoadAdCallback callback) {
        loadAd(placementId, new AdConfig(), callback);
    }

    /**
     * Request the Vungle SDK to load the assets for an advertisement by the placement identifier.
     * This will cause us to request an ad from the Ad Server, and if the bid is filled, the assets
     * will be downloaded. It is possible for the bid to get no responses and therefore no assets
     * will be loaded. In this case, the SDK will automatically retry at a later time, specified by
     * the Vungle Server. The callback will be notified that the assets are pending download.
     *
     * @param placementId The placement identifier for which assets should be loaded.
     * @param adConfig    Optional AdConfig field to set ad size and playback options
     * @param callback    The optional callback which will be notified when the assets are loaded, or
     *                    if they are deferred. Errors will also be sent through this callback.
     */
    public static void loadAd(@NonNull final String placementId,
                              @Nullable final AdConfig adConfig,
                              @Nullable final LoadAdCallback callback) {
        loadAd(placementId, null, adConfig, callback);
    }

    /**
     * Request the Vungle SDK to load the assets for an advertisement by the placement identifier.
     * This will cause us to request an ad from the Ad Server, and if the bid is filled, the assets
     * will be downloaded. It is possible for the bid to get no responses and therefore no assets
     * will be loaded. In this case, the SDK will automatically retry at a later time, specified by
     * the Vungle Server. The callback will be notified that the assets are pending download.
     *
     * @param placementId The placement identifier for which assets should be loaded.
     * @param markup      The placement markup for which assets should be loaded.
     * @param adConfig    Optional AdConfig field to set ad size and playback options
     * @param callback    The optional callback which will be notified when the assets are loaded, or
     *                    if they are deferred. Errors will also be sent through this callback.
     */
    public static void loadAd(@NonNull final String placementId,
                              @Nullable final String markup,
                              @Nullable final AdConfig adConfig,
                              @Nullable final LoadAdCallback callback) {

        VungleLogger.debug("Vungle#loadAd", "loadAd API call invoked");
        /// Validate SDK State
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized");
            onLoadError(placementId, callback, new VungleException(VUNGLE_NOT_INTIALIZED));
            return;
        }

        if (adConfig != null && !AdConfig.AdSize.isDefaultAdSize(adConfig.getAdSize())) {
            onLoadError(placementId, callback, new VungleException(INCORRECT_DEFAULT_API_USAGE));
            return;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Repository repository = serviceLocator.getService(Repository.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        Placement placement = repository.load(placementId, Placement.class).get(provider.getTimeout(), TimeUnit.MILLISECONDS);
        if (placement != null && placement.getPlacementAdType() == Placement.TYPE_VUNGLE_NATIVE) {
            onLoadError(placementId, callback, new VungleException(INCORRECT_DEFAULT_API_USAGE_NATIVE));
            return;
        }

        loadAdInternal(placementId, markup, adConfig, callback);
    }

    static void loadAdInternal(@NonNull final String placementId,
                               @Nullable final String markup,
                               @Nullable AdConfig adConfig,
                               @Nullable final LoadAdCallback callback) {
        /// Validate SDK State
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized");
            onLoadError(placementId, callback, new VungleException(VUNGLE_NOT_INTIALIZED));
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

        LoadAdCallback listener;
        if (callback instanceof LoadNativeAdCallback) {
            listener = new LoadNativeAdCallbackWrapper(serviceLocator
                    .getService(Executors.class)
                    .getUIExecutor(), (LoadNativeAdCallback) callback);
        } else {
            listener = new LoadAdCallbackWrapper(serviceLocator
                    .getService(Executors.class)
                    .getUIExecutor(), callback);
        }

        final AdMarkup serializedAdMarkup = AdMarkupDecoder.decode(markup);
        if (!TextUtils.isEmpty(markup) && serializedAdMarkup == null) {
            onLoadError(placementId, callback, new VungleException(MISSING_HBP_EVENT_ID));
            return;
        }

        /*
         * If adConfig set by the publisher is null set to default
         */
        AdMarkup adMarkup = AdMarkupDecoder.decode(markup);
        AdLoader adLoader = serviceLocator.getService(AdLoader.class);
        if (adConfig == null) {
            adConfig = new AdConfig();
        }
        adLoader.load(new AdRequest(placementId, adMarkup, true), adConfig, listener);
    }

    /**
     * Clear the internal caches and files. This triggers a config call to happen again.
     * This method is private but used in test app using reflection.
     */
    private static void clearCache() {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized");
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        serviceLocator.getService(Executors.class).getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                //Clear the persisted data
                serviceLocator.getService(Downloader.class).cancelAll();
                serviceLocator.getService(AdLoader.class).clear();
                serviceLocator.getService(Repository.class).clearAllData();
                _instance.playOperations.clear();

                //Reset ccpa status due to database dropped.
                _instance.ccpaStatus.set(null);

                /// Configure again, which will auto-load the auto-cached placement and hydrate our metadata.
                _instance.configure(serviceLocator.getService(RuntimeValues.class).initCallback.get(), true);
            }
        });
    }

    /**
     * Delete all existing advertisement.
     * This method is private but used in test app using reflection.
     */
    private static void clearAdvertisements() {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized");
            return;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        serviceLocator.getService(Executors.class).getBackgroundExecutor().execute(new Runnable() {
            @Override
            public void run() {
                //Clear the persisted data
                serviceLocator.getService(Downloader.class).cancelAll();
                serviceLocator.getService(AdLoader.class).clear();
                final Repository repository = serviceLocator.getService(Repository.class);
                Executors sdkExecutors = serviceLocator.getService(Executors.class);
                sdkExecutors.getBackgroundExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        List<Advertisement> ads = repository.loadAll(Advertisement.class).get();
                        if (ads != null) {
                            for (Advertisement ad : ads) {
                                try {
                                    repository.deleteAdvertisement(ad.getId());
                                } catch (DatabaseHelper.DBException ignored) {
                                }
                            }
                        }
                    }
                });
            }
        });
    }

    @Nullable
    static VungleBannerView getBannerViewInternal(String placementId, AdMarkup markup,
                                                  AdConfig adConfig, final PlayAdCallback playAdCallback) {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized, returned VungleBannerView = null");
            onPlayError(placementId, playAdCallback, new VungleException(VUNGLE_NOT_INTIALIZED));
            return null;
        }

        if (TextUtils.isEmpty(placementId)) {
            onPlayError(placementId, playAdCallback, new VungleException(PLACEMENT_NOT_FOUND));
            return null;
        }

        final ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        final AdLoader adLoader = serviceLocator.getService(AdLoader.class);
        AdRequest request = new AdRequest(placementId, markup, true);

        //todo allow same placement on multiple banners for simple integration?
        boolean isLoading = adLoader.isLoading(request);
        if (TRUE.equals(_instance.playOperations.get(placementId)) || isLoading) {
            Log.e(TAG, "Playing or Loading operation ongoing. Playing "
                    + _instance.playOperations.get(request.getPlacementId()) + " Loading: " + isLoading);
            onPlayError(placementId, playAdCallback, new VungleException(OPERATION_ONGOING));
            return null;
        }
        try {
            return new VungleBannerView(
                    _instance.context.getApplicationContext(),
                    request,
                    adConfig,
                    serviceLocator.getService(PresentationFactory.class),
                    new AdEventListener(
                            request,
                            _instance.playOperations,
                            playAdCallback,
                            serviceLocator.getService(Repository.class),
                            adLoader,
                            serviceLocator.getService(JobRunner.class),
                            serviceLocator.getService(VisionController.class),
                            null,
                            null
                    )
            );
        } catch (Exception e) {
            VungleLogger.error("Vungle#playAd", "Vungle banner ad fail: " + e.getLocalizedMessage());
            if (playAdCallback != null) {
                playAdCallback.onError(placementId, new VungleException(AD_UNABLE_TO_PLAY));
            }
            if (BuildConfig.DEBUG) {
                throw new RuntimeException(e);
            }

        }
        return null;
    }

    /**
     * If Vungle is initialized, will return list of placement identifiers associated with
     * it's corresponding app id
     *
     * @return a list of valid placement identifiers
     */
    public static Collection<String> getValidPlacements() {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized return empty placements list");
            return Collections.emptyList();
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Repository repository = serviceLocator.getService(Repository.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        Collection<String> placements = repository.getValidPlacementIds().get(provider.getTimeout(), TimeUnit.MILLISECONDS);
        if (placements == null) {
            return Collections.emptyList();
        } else {
            return placements;
        }
    }


    /**
     * If Vungle is initialized, will return list of all placements
     * This method is protected but used in test app using reflection.
     *
     * @return a list of valid placements
     */
    @VisibleForTesting
    static Collection<Placement> getValidPlacementModels() {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized return empty placements list");
            return Collections.emptyList();
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Repository repository = serviceLocator.getService(Repository.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        Collection<Placement> placements = repository.loadValidPlacements().get(provider.getTimeout(), TimeUnit.MILLISECONDS);
        if (placements == null) {
            return Collections.emptyList();
        } else {
            return placements;
        }
    }

    /**
     * If Vungle is initialized, will return list of all Advertisements for placement
     * This method is protected but used in test app using reflection.
     *
     * @return a list of valid placements
     */
    @VisibleForTesting
    static Collection<Advertisement> getValidAdvertisementModels(@NonNull String placementId) {
        if (!isInitialized()) {
            Log.e(TAG, "Vungle is not initialized return empty placements list");
            return Collections.emptyList();
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Repository repository = serviceLocator.getService(Repository.class);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);

        Collection<Advertisement> advertisements = repository.findValidAdvertisementsForPlacement(placementId, null).get(provider.getTimeout(), TimeUnit.MILLISECONDS);
        if (advertisements == null) {
            return Collections.emptyList();
        } else {
            return advertisements;
        }
    }


    /**
     * GDPR or CCPA Consent Status
     */
    @Keep
    public enum Consent {
        OPTED_IN,
        OPTED_OUT
    }

    /**
     * If handling GDPR Consent dialog with own implementation, use this dialog to update Vungle on
     * user consent status.
     * <p>
     * Updates the data-gathering consent status of the user. This is gathered by the publisher
     * and provided to Vungle. It is then stored on disk and used whenever we are sending information
     * to the ad server.
     *
     * @param status                If {@link Consent#OPTED_IN}, the user has consented to us gathering data about their device. Cannot/shouldn't be <code>null</code>.
     * @param consentMessageVersion Optional version string that can be passed indicating the version
     *                              of your shown consent message users acted on
     */
    @SuppressWarnings("squid:S2583")
    public static void updateConsentStatus(@NonNull final Consent status,
                                           @Nullable final String consentMessageVersion) {
        //Additional check since method is public
        if (status == null) {
            Log.e(TAG, "Cannot set consent with a null consent, please check your code");
            return;
        }

        _instance.consent.set(status);
        _instance.consentVersion = consentMessageVersion;

        if (isInitialized() && isDepInit.get()) {
            ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
            final Repository repository = serviceLocator.getService(Repository.class);
            saveGDPRConsent(repository, _instance.consent.get(), _instance.consentVersion);
        } else {
            Log.e(TAG, "Vungle is not initialized");
        }

    }

    private static void saveGDPRConsent(@NonNull final Repository repository,
                                        @NonNull final Consent status,
                                        @Nullable final String consentMessageVersion) {
        /// If there is already a cookie on disk for consent status, re-use it.
        repository.load(Cookie.CONSENT_COOKIE, Cookie.class, new Repository.LoadCallback<Cookie>() {
            @Override
            public void onLoaded(Cookie gdprConsent) {
                if (gdprConsent == null) {
                    /// Otherwise, create a new one
                    gdprConsent = new Cookie(Cookie.CONSENT_COOKIE);
                }
                gdprConsent.putValue("consent_status", status == Consent.OPTED_IN ? "opted_in" : "opted_out");
                gdprConsent.putValue("timestamp", System.currentTimeMillis() / 1000); /// Server requires seconds.
                gdprConsent.putValue("consent_source", "publisher");
                gdprConsent.putValue("consent_message_version", consentMessageVersion == null ? "" : consentMessageVersion);
                repository.save(gdprConsent, null, false);
            }
        });
    }

    /**
     * @return Whether a user for Vungle has Accepted GDPR Consent. Returns null if user has had the opportunity yet to make opt in or out.
     */
    @Nullable
    public static Consent getConsentStatus() {
        if (isInitialized() && isDepInit.get()) {
            Cookie consentCookie = getGDPRConsent();
            return getConsentStatus(consentCookie);

        }
        return _instance.consent.get();
    }

    private static Consent getConsentStatus(Cookie cookie) {
        if (cookie == null) {
            return null;
        } else {
            String consentString = cookie.getString("consent_status");
            switch (consentString) {
                case "opted_out_by_timeout":
                case "opted_out":
                    _instance.consent.set(Consent.OPTED_OUT);
                    return Consent.OPTED_OUT;
                case "opted_in":
                    _instance.consent.set(Consent.OPTED_IN);
                    return Consent.OPTED_IN;
                default:
                    return null;
            }
        }
    }

    @Nullable
    private static Cookie getGDPRConsent() {
        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);
        return serviceLocator.getService(Repository.class)
                .load(CONSENT_COOKIE, Cookie.class).get(provider.getTimeout(), TimeUnit.MILLISECONDS);
    }

    /**
     * @return Whether a user for Vungle has Accepted GDPR Consent Message Version. {@link Vungle#updateConsentStatus(Consent, String)}
     */
    public static String getConsentMessageVersion() {
        return _instance.consentVersion;
    }

    private static Consent getConsent(Cookie gdprConsent) {
        if (gdprConsent == null) {
            return null;
        }
        return "opted_in".equals(gdprConsent.getString("consent_status")) ? Consent.OPTED_IN : Consent.OPTED_OUT;

    }

    private static String getConsentMessageVersion(Cookie gdprConsent) {
        if (gdprConsent == null) {
            return null;
        } else {
            return gdprConsent.getString("consent_message_version");
        }
    }

    private static String getConsentSource(Cookie gdprConsent) {
        if (gdprConsent == null) {
            return null;
        } else {
            return gdprConsent.getString("consent_source");
        }
    }

    /**
     * If handling CCPA Consent dialog with own implementation, use this dialog to update Vungle on
     * user consent status.
     * <p>
     * Updates the data-gathering consent status of the user. This is gathered by the publisher
     * and provided to Vungle. It is then stored on disk and used whenever we are sending information
     * to the ad server.
     *
     * @param status If {@link Consent#OPTED_IN}, the user has consented to us  data about their
     *               device. Cannot/shouldn't be <code>null</code>.
     */
    @SuppressWarnings("squid:S2583")
    public static void updateCCPAStatus(@NonNull final Consent status) {
        if (status == null) {
            Log.e(TAG, "Unable to update CCPA status, Invalid input parameter.");
            return;
        }

        //To ensure update state should be maintained.
        _instance.ccpaStatus.set(status);

        if (!isInitialized() || !isDepInit.get()) {
            Log.e(TAG, "Vungle is not initialized");
            return;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        Repository repository = serviceLocator.getService(Repository.class);

        updateCCPAStatus(repository, status);
    }

    private static void updateCCPAStatus(@NonNull final Repository repository,
                                         @NonNull final Consent status) {
        repository.load(Cookie.CCPA_COOKIE, Cookie.class, new Repository.LoadCallback<Cookie>() {
            @Override
            public void onLoaded(Cookie ccpaConsent) {
                if (ccpaConsent == null) {
                    ccpaConsent = new Cookie(Cookie.CCPA_COOKIE);
                }
                ccpaConsent.putValue(Cookie.CCPA_CONSENT_STATUS, status == Consent.OPTED_OUT
                        ? Cookie.CONSENT_STATUS_OPTED_OUT : Cookie.CONSENT_STATUS_OPTED_IN);
                repository.save(ccpaConsent, null, false);
            }
        });
    }

    /**
     * @return Whether a user for Vungle has Accepted CCPA Consent.
     * Returns null if SDK has not been called earlier with {@link Vungle#updateCCPAStatus(Consent)}.
     */
    @Nullable
    public static Consent getCCPAStatus() {
        return _instance.ccpaStatus.get();
    }

    private static Consent getCCPAStatus(@Nullable Cookie ccpaConsent) {
        if (ccpaConsent == null) {
            return null;
        }
        return Cookie.CONSENT_STATUS_OPTED_OUT.equals(ccpaConsent.getString(Cookie.CCPA_CONSENT_STATUS)) ? Consent.OPTED_OUT : Consent.OPTED_IN;
    }

    /**
     * Method to register Header bidding callback.
     *
     * @param headerBiddingCallback {@link HeaderBiddingCallback} instance to get notified about
     *                              bidding token and ad availability.
     */
    public static void setHeaderBiddingCallback(HeaderBiddingCallback headerBiddingCallback) {
        if (_instance.context == null)
            return;

        ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
        RuntimeValues runtimeValues = serviceLocator.getService(RuntimeValues.class);

        runtimeValues.headerBiddingCallback.set(new HeaderBiddingCallbackWrapper(
                serviceLocator.getService(Executors.class).getUIExecutor(),
                headerBiddingCallback));
    }

    /**
     * Will return an encoded string of placements bid token.
     * This method might be called from adapter side. By default this will return all bid tokens available.
     *
     * @param context application context.
     * @return an encoded string contains available bid tokens. In rare cases, this can be null.
     */
    public static String getAvailableBidTokens(@NonNull final Context context) {
        return getAvailableBidTokens(context, null, 0);
    }

    /**
     * Use to update Vungle on COPPA status of user. Indicate true if user is known to fall under COPPA regulations.
     * Call this before initialization the Vungle SDK
     * <p>
     * i.e
     * Vungle.updateUserCoppaStatus(true)
     * Vungle.init(...)
     *
     * @param isUserCoppa - true is user is known to be 13 and under to comply with COPPA regulations
     */
    public static void updateUserCoppaStatus(boolean isUserCoppa) {
        PrivacyManager.getInstance().updateCoppaStatus(isUserCoppa);

        if (isInitialized()) {
            Log.e(TAG, "COPPA status changes should be passed before SDK initialization, they will ONLY take into effect during future SDK initializations and sessions");
            return;
        }
    }

    /**
     * Will return an encoded string of placements bid token.
     * This method might be called from adapter side.
     *
     * @param context              application context.
     * @param maxBidTokenSizeBytes the bid tokens size limitation in bytes. Refer to {@link Vungle#getAvailableBidTokens(Context)} in case setting of max size is not required
     *                             Optional: Pass in 0 or negative value to make this filter no-op.
     * @deprecated replaced by {@link #getAvailableBidTokens(Context, String, int)}}
     * instead
     */
    @Deprecated
    public static String getAvailableBidTokensBySize(
            @NonNull final Context context, final int maxBidTokenSizeBytes
    ) {
        return getAvailableBidTokens(context, null, maxBidTokenSizeBytes);
    }

    /**
     * Will return an encoded string of advertisement bid tokens.
     * This method might be called from adapter side.
     *
     * @param context              application context.
     * @param placementId          Returns the list of bid tokens corresponding to placement ID.
     *                             Optional: Pass in null or empty String to make this filter return all encoded advertisements.
     * @param maxBidTokenSizeBytes the bid tokens size limitation in bytes. Refer to {@link Vungle#getAvailableBidTokens(Context)} in case setting of max size is not required
     *                             Optional: Pass in 0 or negative value to make this filter no-op.
     * @return an encoded string contains available bid tokens digest. In rare cases, this can return null value
     */
    @Nullable
    public static String getAvailableBidTokens(
            @NonNull final Context context,
            @Nullable final String placementId,
            final int maxBidTokenSizeBytes
    ) {
        if (context == null) {
            Log.e(TAG, "Context is null");
            return null;
        }

        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        Executors sdkExecutors = serviceLocator.getService(Executors.class);
        final TimeoutProvider provider = serviceLocator.getService(TimeoutProvider.class);
        final BidTokenEncoder bidTokenEncoder = serviceLocator.getService(BidTokenEncoder.class);

        FutureResult<String> futureResult = new FutureResult<>(sdkExecutors.getApiExecutor()
                .submit(new Callable<String>() {
                    @Override
                    public String call() {
                        String superToken = bidTokenEncoder.encode(
                                placementId,
                                maxBidTokenSizeBytes,
                                _instance.hbpOrdinalViewCount.incrementAndGet());
                        Log.d(TAG, "Supertoken is " + superToken);
                        return superToken;
                    }
                }));

        return futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS);
    }

    private void saveConfigExtension(Repository repository, JsonObject jsonObject) throws DatabaseHelper.DBException {
        Cookie configExt = new Cookie(Cookie.CONFIG_EXTENSION);
        String configExtension = "";
        if (jsonObject.has("config_extension")) {
            configExtension = JsonUtil.getAsString(jsonObject, Cookie.CONFIG_EXTENSION, "");
        }
        configExt.putValue("config_extension", configExtension);
        repository.save(configExt);
    }

    /**
     * Lite version of de-initialization, can be used in Unit Tests and {@link Vungle#init(String, Context, InitCallback)}
     */
    protected static void deInit() {
        if (_instance.context != null) {
            ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);
            if (serviceLocator.isCreated(CacheManager.class)) {
                serviceLocator.getService(CacheManager.class).removeListener(cacheListener);
            }
            if (serviceLocator.isCreated(Downloader.class)) {
                serviceLocator.getService(Downloader.class).cancelAll();
            }
            if (serviceLocator.isCreated(AdLoader.class)) {
                serviceLocator.getService(AdLoader.class).clear();
            }
            _instance.playOperations.clear();
        }

        ServiceLocator.deInit();

        isInitialized = false;
        isDepInit.set(false);
        isInitializing.set(false);
    }

    private static CacheManager.Listener cacheListener = new CacheManager.Listener() {
        @Override
        public void onCacheChanged() {
            if (_instance.context == null)
                return;

            stopPlaying();
            ServiceLocator serviceLocator = ServiceLocator.getInstance(_instance.context);

            CacheManager cacheManager = serviceLocator.getService(CacheManager.class);
            Downloader downloader = serviceLocator.getService(Downloader.class);
            if (cacheManager.getCache() != null) {
                List<DownloadRequest> requests = downloader.getAllRequests();
                String newPath = cacheManager.getCache().getPath();
                for (DownloadRequest request : requests) {
                    if (!request.path.startsWith(newPath)) {
                        downloader.cancel(request);
                    }
                }
            }
            downloader.init();
        }
    };

    private static void stopPlaying() {
        if (_instance.context == null)
            return;

        Intent broadcast = new Intent(AdvertisementBus.ACTION);
        broadcast.putExtra(AdvertisementBus.COMMAND, AdvertisementBus.STOP_ALL);
        LocalBroadcastManager.getInstance(_instance.context).sendBroadcast(broadcast);
    }

    @Nullable
    static Context appContext() {
        if (_instance != null)
            return _instance.context;
        return null;
    }
}