package com.vungle.warren;

import static com.vungle.warren.error.VungleException.CONFIGURATION_ERROR;
import static com.vungle.warren.model.Cookie.CCPA_COOKIE;
import static com.vungle.warren.model.Cookie.CONSENT_COOKIE;
import static com.vungle.warren.model.Cookie.IS_PLAY_SERVICE_AVAILABLE;
import static com.vungle.warren.model.Cookie.USER_AGENT_ID_COOKIE;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.UiModeManager;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.media.AudioManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.BatteryManager;
import android.os.Build;
import android.os.Environment;
import android.os.PowerManager;
import android.provider.Settings;
import android.security.NetworkSecurityPolicy;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.WindowManager;
import android.webkit.URLUtil;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringDef;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.PermissionChecker;
import androidx.core.util.Consumer;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailabilityLight;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.AdvertisingInfo;
import com.vungle.warren.model.CacheBust;
import com.vungle.warren.model.Cookie;
import com.vungle.warren.model.JsonUtil;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.network.APIFactory;
import com.vungle.warren.network.Call;
import com.vungle.warren.network.Response;
import com.vungle.warren.network.VungleApi;
import com.vungle.warren.omsdk.OMInjector;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.utility.TimeoutProvider;
import com.vungle.warren.utility.platform.Platform;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import okhttp3.HttpUrl;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Protocol;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import okio.Buffer;
import okio.BufferedSink;
import okio.GzipSink;
import okio.Okio;

/**
 * The HTTP Client for communicating with the Ad Server. All HTTP Requests should be routed through
 * this class.
 */
public class VungleApiClient {

    private static final String TAG = VungleApiClient.class.getCanonicalName();

    private static final String ID = "id";

    //ID FIELDS
    private static final String AMAZON_ADVERTISING_ID = "amazon_advertising_id";
    public static final String GAID = "gaid";
    public static final String ANDROID_ID = "android_id";
    public static final String IFA = "ifa";
    private final Platform platform;

    /**
     * Application context. Weak reference because we don't want to cause memory leaks if the
     * application is exiting.
     */
    private Context context;

    /**
     * Retrofit API client. The HTTP requests go through here ultimately.
     */
    private VungleApi api;

    /**
     * The paths of the endpoints that are defined by the response of the /config call.
     */
    private String newEndpoint,
            requestAdEndpoint,
            reportAdEndpoint,
            willPlayAdEndpoint,
            riEndpoint,
            logEndpoint,
            cacheBustEndpoint,
            biLoggingEndpoint;
    /**
     * Value of the "device" field that is sent on every POST request to the ad server.
     */
    private JsonObject baseDeviceInfo;

    /**
     * Value of the "app" field that is sent on every POST request to the server.
     */
    private JsonObject appBody;

    /**
     * If true, the ad server would like a chance to replace the ad that is about to be played and
     * /willPlayAd should be called during the playAd() flow. If false, this part of the flow is
     * skipped.
     */
    private boolean willPlayAdEnabled;

    /**
     * The number of milliseconds to set as the timeout for the /willPlayAd response.
     */
    private int willPlayAdTimeout;

    /**
     * The HTTP Client that makes the requests. We need to keep a reference to this in order to
     * create per-request clones. The per-request clones share the connection pool, dispatcher, and
     * configuration of the original client.
     */
    private OkHttpClient client;

    /**
     * Retrofit API Client that enforces a strict timeout on its requests. This is primarily used
     * to manage the willPlayAd timeout logic. This field is only populated if willPlayAdEnabled
     * true.
     * It is otherwise <code>null</code>.
     */
    private VungleApi timeoutApi;

    /**
     * Retrofit API client with GZIP encoding of request body.
     */
    private VungleApi gzipApi;

    /**
     * Config if we should globally initialize OM SDK on this device to track
     * viewability. If this is true we initialize OM SDK.
     * enableOm should override om enabled requests on ad responses.
     */
    private boolean enableOm;

    /**
     * Designer that manages the cache directory. The API requires that we send the number of
     * available bytes in every request, which the designer knows.
     */
    private CacheManager cacheManager;

    /**
     * Holds the value of is_google_play_services_available in Device->ext->vungle->android json
     */
    private Boolean isGooglePlayServicesAvailable;

    private TimeoutProvider timeoutProvider;

    private static String headerUa =
            (Platform.MANUFACTURER_AMAZON.equals(Build.MANUFACTURER) ? "VungleAmazon/" : "VungleDroid/") +
                    BuildConfig.VERSION_NAME;

    private static String BASE_URL = "https://ads.api.vungle.com/";

    protected static WrapperFramework WRAPPER_FRAMEWORK_SELECTED;

    private Map<String, Long> retryAfterDataMap = new ConcurrentHashMap<>();

    private Repository repository;

    /**
     * String to keep track of User-Agent String
     */
    private String uaString = System.getProperty("http.agent");

    private final OMInjector omInjector;

    /**
     * Private no-arg constructor ensure that we always have an api client.
     */
    VungleApiClient(
            @NonNull final Context context,
            @NonNull CacheManager cacheManager,
            @NonNull Repository repository,
            @NonNull OMInjector omInjector,
            @NonNull Platform platform) {
        this.cacheManager = cacheManager;
        this.context = context.getApplicationContext();
        this.repository = repository;
        this.omInjector = omInjector;
        this.platform = platform;

        /// Response Interceptor for Retry-After header value
        Interceptor responseInterceptor = new Interceptor() {

            @Override
            public okhttp3.Response intercept(Chain chain) throws IOException {
                Request request = chain.request();
                okhttp3.Response response;

                String urlPath = request.url().encodedPath();

                Long retryExpireTime = retryAfterDataMap.get(urlPath);
                if (retryExpireTime != null) {
                    long currentTimeStamp = System.currentTimeMillis();
                    long newRetryAfter = TimeUnit.MILLISECONDS.toSeconds(retryExpireTime - currentTimeStamp);
                    if (newRetryAfter > 0) {
                        return new okhttp3.Response.Builder()
                                .request(request)
                                .addHeader("Retry-After", String.valueOf(newRetryAfter))
                                .code(500)
                                .protocol(Protocol.HTTP_1_1)
                                .message("Server is busy")
                                .body(ResponseBody.create(MediaType.parse("application/json; charset=utf-8"), "{\"Error\":\"Retry-After\"}"))
                                .build();
                    } else {
                        retryAfterDataMap.remove(urlPath);            //Retry - After time passed, Remove url from MAP, allow request
                    }
                }

                response = chain.proceed(request);
                if (response != null) {
                    int responseCode = response.code();
                    if (responseCode == 429 || responseCode == 500 || responseCode == 502 || responseCode == 503) {                            //Checking for error code as per API docs
                        String retryAfterTimeStr = response.headers().get("Retry-After");
                        if (!TextUtils.isEmpty(retryAfterTimeStr)) {
                            try {
                                long retryAfterTimeValue = Long.parseLong(retryAfterTimeStr);
                                if (retryAfterTimeValue > 0) {
                                    retryAfterDataMap.put(urlPath, (retryAfterTimeValue * 1000) + System.currentTimeMillis());
                                }
                            } catch (NumberFormatException e) {
                                Log.d(TAG, "Retry-After value is not an valid value");
                            }
                        }
                    }
                }
                return response;
            }
        };

        /// Create the OkHttp Client
        OkHttpClient.Builder builder = new OkHttpClient.Builder()
                .addInterceptor(responseInterceptor);
        if (BuildConfig.DEBUG) {
            for (Interceptor interceptor : logInterceptors) {
                builder.addInterceptor(interceptor);                //SDK File logger works with addInterceptor()
            }

            for (Interceptor interceptor : networkInterceptors) {
                builder.addNetworkInterceptor(interceptor);         //Stetho requires addNetworkInterceptor()
            }
        }
        client = builder.build();

        OkHttpClient gzipClient = builder.addInterceptor(new GzipRequestInterceptor()).build();

        api = new APIFactory(client, BASE_URL).createAPI(Vungle._instance.appID);
        gzipApi = new APIFactory(gzipClient, BASE_URL).createAPI(Vungle._instance.appID);

        ServiceLocator serviceLocator = ServiceLocator.getInstance(context);
        timeoutProvider = serviceLocator.getService(TimeoutProvider.class);
    }

    public static String getHeaderUa() {
        return headerUa;
    }

    public static void setHeaderUa(String headerUa) {
        VungleApiClient.headerUa = headerUa;
    }

    public void init() {
        init(context);
    }

    static class GzipRequestInterceptor implements Interceptor {
        private static final String CONTENT_ENCODING = "Content-Encoding";
        private static final String GZIP = "gzip";

        @NonNull
        @Override
        public okhttp3.Response intercept(@NonNull Chain chain) throws IOException {
            okhttp3.Request originalRequest = chain.request();
            if (originalRequest.body() == null
                    || originalRequest.header(CONTENT_ENCODING) != null) {
                return chain.proceed(originalRequest);
            }

            okhttp3.Request compressedRequest = originalRequest.newBuilder()
                    .header(CONTENT_ENCODING, GZIP)
                    .method(originalRequest.method(), gzip(originalRequest.body()))
                    .build();
            return chain.proceed(compressedRequest);
        }

        private RequestBody gzip(final RequestBody requestBody) throws IOException {
            final Buffer output = new Buffer();
            BufferedSink gzipSink = Okio.buffer(new GzipSink(output));
            requestBody.writeTo(gzipSink);
            gzipSink.close();
            return new RequestBody() {
                @Override
                public MediaType contentType() {
                    return requestBody.contentType();
                }

                @Override
                public long contentLength() {
                    return output.size();
                }

                @Override
                public void writeTo(@NonNull BufferedSink sink) throws IOException {
                    sink.write(output.snapshot());
                }
            };
        }
    }

    /**
     * Initializes the Vungle API Client and creates the request body parts.
     *
     * @param context Application context
     */
    @VisibleForTesting
    synchronized void init(final Context context) {

        /// Create the device and app objects, they don't change through the lifecycle of the SDK App
        JsonObject app = new JsonObject();
        app.addProperty("bundle", context.getPackageName());
        String versionName = null;
        try {
            versionName = context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionName;
        } catch (PackageManager.NameNotFoundException e) {
            /// Unable to retrieve the application version, will default to 1.0
        }
        app.addProperty("ver", versionName != null ? versionName : "1.0");

        /// Device
        JsonObject device = new JsonObject();
        device.addProperty("make", Build.MANUFACTURER);
        device.addProperty("model", Build.MODEL);
        device.addProperty("osv", Build.VERSION.RELEASE);
        device.addProperty("carrier", ((TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE)).getNetworkOperatorName());
        device.addProperty("os", Platform.MANUFACTURER_AMAZON.equals(Build.MANUFACTURER) ? "amazon" : "android");
        DisplayMetrics dm = new DisplayMetrics();
        WindowManager windowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        windowManager.getDefaultDisplay().getMetrics(dm);
        device.addProperty("w", dm.widthPixels);
        device.addProperty("h", dm.heightPixels);

        try {
            //Lets first check in Cookie
            uaString = platform.getUserAgent();
            device.addProperty("ua", uaString);

            //might it has been updated, re-init UA in background. Since it should not block Vungle Executor or to avoid dead-lock
            initUserAgentLazy();
        } catch (Exception ex) {        /// Adding Generic Exception to avoid any WebView related crash
            Log.e(TAG, "Cannot Get UserAgent. Setting Default Device UserAgent." + ex.getLocalizedMessage());
        }

        baseDeviceInfo = device;

        /// Assign the values to the singleton instance.
        appBody = app;

        //try to get Play services availability and store it in Cookie for reuse from DB
        //as querying from GPS which in turn queries Pckg Mgr is not optimal and recently GPS also throws RuntimeException
        isGooglePlayServicesAvailable = getPlayServicesAvailabilityFromAPI();
    }

    void setAppId(String appId) {
        setAppId(appId, appBody);
    }

    private void setAppId(String appId, JsonObject requestBody) {
        requestBody.addProperty(ID, appId);
    }

    private void initUserAgentLazy() {
        platform.getUserAgentLazy(new Consumer<String>() {
            @Override
            public void accept(String uaString) {
                if (uaString == null) {
                    Log.e(TAG, "Cannot Get UserAgent. Setting Default Device UserAgent");
                    return;
                }
                VungleApiClient.this.uaString = uaString;
            }
        });
    }

    /**
     * Requests a configuration from the server. This will set the relevant fields within the API
     * Client for the dynamic endpoints, then callback.
     */
    public Response config() throws VungleException, IOException {

        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody(true));
        body.add("app", appBody);
        body.add("user", getUserBody());
        JsonObject extBody = getExtBody();
        if (extBody != null) {
            body.add("ext", extBody);
        }

        Response<JsonObject> response = api.config(getHeaderUa(), body).execute();

        if (!response.isSuccessful()) {
            /// Immediately propagate the response up to the caller. We cannot extract
            /// any useful information here, but perhaps some business logic can.
            return response;
        }

        JsonObject jsonObject = response.body();
        Log.d(TAG, "Config Response: " + jsonObject);
        if (JsonUtil.hasNonNull(jsonObject, "sleep")) {
            String errorMessage = JsonUtil.hasNonNull(jsonObject, "info") ? jsonObject.get("info").getAsString() : "";
            Log.e(TAG, "Error Initializing Vungle. Please try again. " + errorMessage);
            throw new VungleException(CONFIGURATION_ERROR);
        }

        //endpoints call be null and crash out application
        if (!JsonUtil.hasNonNull(jsonObject, "endpoints")) {
            Log.e(TAG, "Error Initializing Vungle. Please try again. ");
            throw new VungleException(CONFIGURATION_ERROR);
        }

        /// Parse out the endpoints
        JsonObject endpoints = jsonObject.getAsJsonObject("endpoints");

        HttpUrl newUrl = HttpUrl.parse(endpoints.get("new").getAsString());
        HttpUrl adsUrl = HttpUrl.parse(endpoints.get("ads").getAsString());
        HttpUrl willPlayAdUrl = HttpUrl.parse(endpoints.get("will_play_ad").getAsString());
        HttpUrl reportAdUrl = HttpUrl.parse(endpoints.get("report_ad").getAsString());
        HttpUrl reportIncentivized = HttpUrl.parse(endpoints.get("ri").getAsString());
        HttpUrl logUrl = HttpUrl.parse(endpoints.get("log").getAsString());
        HttpUrl cacheBustUrl = HttpUrl.parse(endpoints.get("cache_bust").getAsString());
        HttpUrl biLoggingUrl = HttpUrl.parse(endpoints.get("sdk_bi").getAsString());

        if (newUrl == null || adsUrl == null || willPlayAdUrl == null || reportAdUrl == null
                || reportIncentivized == null || logUrl == null || cacheBustUrl == null || biLoggingUrl == null) {
            Log.e(TAG, "Error Initializing Vungle. Please try again. ");
            throw new VungleException(CONFIGURATION_ERROR);
        }

        newEndpoint = newUrl.toString();
        requestAdEndpoint = adsUrl.toString();
        willPlayAdEndpoint = willPlayAdUrl.toString();
        reportAdEndpoint = reportAdUrl.toString();
        riEndpoint = reportIncentivized.toString();
        logEndpoint = logUrl.toString();
        cacheBustEndpoint = cacheBustUrl.toString();
        biLoggingEndpoint = biLoggingUrl.toString();

        /// Parse out the configuration variables
        JsonObject willPlayAd = jsonObject.getAsJsonObject("will_play_ad");
        willPlayAdTimeout = willPlayAd.get("request_timeout").getAsInt();
        willPlayAdEnabled = willPlayAd.get("enabled").getAsBoolean();

        //parse out viewability
        JsonObject viewability = jsonObject.getAsJsonObject("viewability");
        enableOm = JsonUtil.getAsBoolean(viewability, "om", false);

        if (willPlayAdEnabled) {
            /// Make a clone of the original client in order to override the timeout value only for
            /// this endpoint. It is not currently possible to override this value per
            /// request or per call. Followed best practices according to OkHTTP:
            /// https://github.com/square/okhttp/wiki/Recipes#per-call-configuration
            Log.v(TAG, "willPlayAd is enabled, generating a timeout client.");
            OkHttpClient timeoutClient = client.newBuilder()
                    .readTimeout(willPlayAdTimeout, TimeUnit.MILLISECONDS)
                    .build();
            APIFactory timeoutRetro = new APIFactory(timeoutClient,
                    "https://api.vungle.com/");

            timeoutApi = timeoutRetro.createAPI(Vungle._instance.appID);
        }

        if (getOmEnabled()) {
            omInjector.init();
        } else {
            SessionTracker.getInstance().trackEvent(
                    new SessionData.Builder()
                            .setEvent(SessionEvent.OM_SDK)
                            .addData(SessionAttribute.ENABLED, false)
                            .build());
        }

        return response;
    }

    /**
     * Only called the first time the SDK runs per application, informs the server that there is a new
     * installation.
     *
     * @return {@link Call} which will pass back the response body to the caller.
     * @throws IllegalStateException if this method is called before the Api Client has been initialized.
     */
    public Call<JsonObject> reportNew() throws IllegalStateException {
        if (newEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }
        HashMap<String, String> query = new HashMap<>(2);
        JsonElement idElement = appBody.get(ID);
        query.put("app_id", idElement != null ? idElement.getAsString() : "");

        //Device body after init is stale and will have ifa removed from deviceBody()
        JsonObject latestDeviceBody = getDeviceBody();
        if (PrivacyManager.getInstance().shouldSendAdIds()) {
            JsonElement ifaElement = latestDeviceBody.get("ifa");
            query.put("ifa", ifaElement != null ? ifaElement.getAsString() : "");
        }
        return api.reportNew(getHeaderUa(), newEndpoint, query);
    }

    /**
     * Request the ad information for a particular placement.
     *
     * @param placement             The identifier for the placement whose assets are being requested.
     * @param adSize                Size of Ad
     * @param isHeaderBiddingEnable Status of header bidding feature.
     * @param vision                Vision payload
     * @return {@link Call} which will pass back the response body to the caller.
     * @throws IllegalStateException If called before the API Client has been initialized
     */
    public Call<JsonObject> requestAd(String placement, String adSize, boolean isHeaderBiddingEnable, @Nullable JsonObject vision) throws IllegalStateException {
        if (requestAdEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }

        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        JsonObject userBody = getUserBody();
        if (vision != null) {
            userBody.add(VisionController.VISION, vision);
        }
        body.add("user", userBody);

        JsonObject extBody = getExtBody();
        if (extBody != null) {
            body.add("ext", extBody);
        }

        /// Create the request body
        JsonObject request = new JsonObject();
        JsonArray placementsArray = new JsonArray();
        placementsArray.add(placement);
        request.add("placements", placementsArray);
        request.addProperty("header_bidding", isHeaderBiddingEnable);

        if (!TextUtils.isEmpty(adSize)) {
            request.addProperty("ad_size", adSize);
        }

        body.add("request", request);

        /// Hack to work around the server not giving me ads.
        return gzipApi.ads(getHeaderUa(), requestAdEndpoint, body);
    }

    /**
     * Inform the server that we are about to play an ad, gives the back-end a chance to change the ad
     * that will be played if there is a better one available.
     *
     * @param adToken     The token identifying the advertisement bundle that is about to be played
     * @param autoCached  Whether or not this placement is auto-cached
     * @param placementID The placement identifier.
     * @return {@link Call} which will pass back the response body to the caller.
     */
    Call<JsonObject> willPlayAd(String placementID, boolean autoCached, String adToken) {
        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        body.add("user", getUserBody());

        /// Create the request body
        JsonObject request = new JsonObject();
        JsonObject placement = new JsonObject();
        placement.addProperty("reference_id", placementID);
        placement.addProperty("is_auto_cached", autoCached);

        request.add("placement", placement);
        request.addProperty("ad_token", adToken);

        body.add("request", request);

        return timeoutApi.willPlayAd(getHeaderUa(), willPlayAdEndpoint, body);
    }

    boolean canCallWillPlayAd() {
        return willPlayAdEnabled && !TextUtils.isEmpty(willPlayAdEndpoint);
    }

    /**
     * Report that an ad has been played to the server.
     *
     * @param request The request body
     * @return A {@link Call} object which can be used to execute or enqueue the request. It is expected
     * that the caller will handle the error scenarios.
     */
    public Call<JsonObject> reportAd(JsonObject request) {
        if (reportAdEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }
        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        body.add("request", request);
        body.add("user", getUserBody());
        JsonObject extBody = getExtBody();
        if (extBody != null) {
            body.add("ext", extBody);
        }

        return gzipApi.reportAd(getHeaderUa(), reportAdEndpoint, body);
    }

    /**
     * Send log document files to the server.
     *
     * @param request The request body
     * @return A {@link Call} object which can be used to execute or enqueue the request. It is expected
     * that the caller will handle the error scenarios.
     */
    public Call<JsonObject> sendLog(JsonObject request) {
        if (logEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }

        return gzipApi.sendLog(getHeaderUa(), logEndpoint, request);
    }

    /**
     * <pre>
     * "request": {
     * "placement_reference_id": "string",// reserved for utilize placement information to reward callback<br>
     * "app_id": "string",
     * "adStartTime": 0,// Make sure its value is same as the one in /report_ad request payload<br>
     * "user": "string",
     * "name": "string"
     * }
     * </pre>
     *
     * @param request request parameters packed in {@link JsonObject}
     * @return {@link Call} to /ri API which will pass back the response body to the caller.
     */

    public Call<JsonObject> ri(JsonObject request) {
        if (riEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }
        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        body.add("request", request);
        body.add("user", getUserBody());
        JsonObject extBody = getExtBody();
        if (extBody != null) {
            body.add("ext", extBody);
        }

        return api.ri(getHeaderUa(), riEndpoint, body);
    }

    /**
     * Ping TPAT
     *
     * @param url - TPAT Url
     * @return false if request failed but can retry
     */
    public boolean pingTPAT(final String url) throws ClearTextTrafficException, MalformedURLException {
        if (!TextUtils.isEmpty(url) && HttpUrl.parse(url) != null) {    //Url is empty or invalid, No need to hit tpat
            /*
             * From Android M, developer can use clearTextTraffic flag to set the clearTextTraffic allow or not.
             * Below Android M, clearTextTraffic is always allowed and developer can not disable it.
             *
             * By default allow clearTextTrafficEnabled is true, below api Android P and upto Android M
             * By default allow clearTextTrafficEnabled is false, for api Android P and above
             *
             * So the below condition is used to check whether cleartext network traffic allowed or not for App
             * https://developer.android.com/reference/android/security/NetworkSecurityPolicy.html#isCleartextTrafficPermitted()
             */
            boolean clearTextTrafficPermitted;
            String host;                                                //Checking host
            try {
                host = (new URL(url)).getHost();
            } catch (MalformedURLException e) {
                SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.TPAT)
                        .addData(SessionAttribute.SUCCESS, false)
                        .addData(SessionAttribute.REASON, "Invalid URL")
                        .addData(SessionAttribute.URL, url)
                        .build()
                );
                throw new MalformedURLException("Invalid URL : " + url);
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {       //Checking clearTextTrafficPermitted
                clearTextTrafficPermitted = NetworkSecurityPolicy.getInstance().isCleartextTrafficPermitted(host);  //Above Android N check with host
            } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                clearTextTrafficPermitted = NetworkSecurityPolicy.getInstance().isCleartextTrafficPermitted();      //Android M check normally
            } else {
                clearTextTrafficPermitted = true;                                                                   //Older always permitted
            }

            if (!clearTextTrafficPermitted && URLUtil.isHttpUrl(url)) {
                SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.TPAT)
                        .addData(SessionAttribute.SUCCESS, false)
                        .addData(SessionAttribute.REASON, "Clear Text Traffic is blocked")
                        .addData(SessionAttribute.URL, url)
                        .build());

                //If not permitted and url is http throw exc
                throw new ClearTextTrafficException("Clear Text Traffic is blocked");
            }

            try {
                Response<Void> response = api.pingTPAT(uaString, url).execute();

                if (response == null) {
                    SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.TPAT)
                            .addData(SessionAttribute.SUCCESS, false)
                            .addData(SessionAttribute.REASON, "Error on pinging TPAT")
                            .addData(SessionAttribute.URL, url)
                            .build());
                } else if (!response.isSuccessful()) {
                    SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.TPAT)
                            .addData(SessionAttribute.SUCCESS, false)
                            .addData(SessionAttribute.REASON, response.code() + ": " + response.message())
                            .addData(SessionAttribute.URL, url)
                            .build());
                }
            } catch (IOException e) {
                SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.TPAT)
                        .addData(SessionAttribute.SUCCESS, false)
                        .addData(SessionAttribute.REASON, e.getMessage())
                        .addData(SessionAttribute.URL, url)
                        .build());
                Log.d(TAG, "Error on pinging TPAT");
                return false;
            }
        } else {
            SessionTracker.getInstance().trackEvent(new SessionData.Builder().setEvent(SessionEvent.TPAT)
                    .addData(SessionAttribute.SUCCESS, false)
                    .addData(SessionAttribute.REASON, "Invalid URL")
                    .addData(SessionAttribute.URL, url)
                    .build());
            throw new MalformedURLException("Invalid URL : " + url);
        }
        return true;
    }

    /**
     * @param lastCacheBustTime last updated
     * @return
     */
    public Call<JsonObject> cacheBust(long lastCacheBustTime) {
        if (cacheBustEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }

        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);
        JsonObject userBody = getUserBody();
        body.add("user", userBody);

        /// Create the request body
        JsonObject request = new JsonObject();
        request.addProperty("last_cache_bust", lastCacheBustTime);

        body.add("request", request);

        return gzipApi.cacheBust(getHeaderUa(), cacheBustEndpoint, body);
    }

    /**
     * Sends analytics regarding cache bust
     *
     * @return {@link Call} with {@link JsonObject} to contain result
     */
    public Call<JsonObject> sendAnalytics(Collection<CacheBust> busts) {
        if (biLoggingEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }

        if (busts == null || busts.isEmpty()) {
            throw new IllegalArgumentException("Cannot send analytics when bust and session data is empty");
        }

        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);

        /// Create the request body
        JsonObject request = new JsonObject();
        JsonArray bustsArray = new JsonArray(busts.size());
        for (CacheBust bust : busts) {
            JsonObject bustElement;
            for (int i = 0; i < bust.getEventIds().length; i++) {
                bustElement = new JsonObject();
                bustElement.addProperty("target",
                        bust.getIdType() == CacheBust.EVENT_TYPE_CAMPAIGN ? "campaign" : "creative");
                bustElement.addProperty("id", bust.getId());
                bustElement.addProperty("event_id", bust.getEventIds()[i]);
                bustsArray.add(bustElement);
            }
        }
        if (bustsArray.size() > 0) {
            request.add("cache_bust", bustsArray);
        }

        body.add("request", request);

        return gzipApi.sendBiAnalytics(getHeaderUa(), biLoggingEndpoint, body);
    }

    /**
     * Sends session event analytics
     *
     * @return {@link Call} with {@link JsonObject} to contain result
     */
    public Call<JsonObject> sendSessionDataAnalytics(@NonNull JsonArray sessionEvents) {
        if (biLoggingEndpoint == null) {
            throw new IllegalStateException("API Client not configured yet! Must call /config first.");
        }

        JsonObject body = new JsonObject();
        body.add("device", getDeviceBody());
        body.add("app", appBody);

        /// Create the request body
        JsonObject request = new JsonObject();

        request.add("session_events", sessionEvents);

        body.add("request", request);

        return gzipApi.sendBiAnalytics(getHeaderUa(), biLoggingEndpoint, body);
    }


    /**
     * Method that generates the device body. The device body is not because it includes up-to-date
     * battery and network information. Example:
     * <pre><code>
     *     "android_id": "68f19b937a5ef8da",
     *     "battery_level": 1,
     *     "battery_saver_enabled": 0,
     *     "battery_state": "BATTERY_PLUGGED_USB",
     *     "connection_type": "WIFI",
     *     "connection_type_detail": "WIFI",
     *     "data_saver_status": "NOT_APPLICABLE",
     *     "gaid": "68f19b937a5ef8da",
     *     "language": "en",
     *     "locale": "en_US",
     *     "network_metered": 0,
     *     "sd_card_available": 1,
     *     "sound_enabled": 0,
     *     "time_zone": "America/Los_Angeles",
     *     "volume_level": 0
     * </code></pre>
     *
     * @return JsonObject that includes the up-to-date android device information.
     */
    @SuppressLint({"HardwareIds", "NewApi"})
    @SuppressWarnings("squid:S5322")
    private JsonObject getDeviceBody() throws IllegalStateException {
        return getDeviceBody(false);
    }

    /**
     * Method that generates the device body. The device body is not because it includes up-to-date
     * battery and network information. Example:
     * <pre><code>
     *     "android_id": "68f19b937a5ef8da",
     *     "battery_level": 1,
     *     "battery_saver_enabled": 0,
     *     "battery_state": "BATTERY_PLUGGED_USB",
     *     "connection_type": "WIFI",
     *     "connection_type_detail": "WIFI",
     *     "data_saver_status": "NOT_APPLICABLE",
     *     "gaid": "68f19b937a5ef8da",
     *     "language": "en",
     *     "locale": "en_US",
     *     "network_metered": 0,
     *     "sd_card_available": 1,
     *     "sound_enabled": 0,
     *     "time_zone": "America/Los_Angeles",
     *     "volume_level": 0
     * </code></pre>
     *
     * @param explicitBlock Should only be used by config to override the disable_ad_id
     * @return JsonObject that includes the up-to-date android device information.
     */
    @SuppressLint({"HardwareIds", "NewApi"})
    @SuppressWarnings("squid:S5322")
    private synchronized JsonObject getDeviceBody(boolean explicitBlock) throws IllegalStateException {
        JsonObject deviceBody = baseDeviceInfo.deepCopy();

        JsonObject android = new JsonObject();

        /// Advertising Identifier
        String advertId;
        boolean limitAdTracking;

        AdvertisingInfo advertisingInfo = platform.getAdvertisingInfo();
        limitAdTracking = advertisingInfo.limitAdTracking;
        advertId = advertisingInfo.advertisingId;

        if (PrivacyManager.getInstance().shouldSendAdIds()) {
            if (advertId != null) {
                android.addProperty(Platform.MANUFACTURER_AMAZON.equals(Build.MANUFACTURER) ? AMAZON_ADVERTISING_ID : GAID, advertId);
                deviceBody.addProperty(IFA, advertId);
            } else {
                /// If the google advertising ID is not available, we fall back to the android_id
                String androidID = platform.getAndroidId();
                deviceBody.addProperty(IFA, !TextUtils.isEmpty(androidID) ? androidID : "");
                if (!TextUtils.isEmpty(androidID)) {
                    android.addProperty(ANDROID_ID, androidID);
                }
            }
        }

        //Remove Ad Ids from in memory (android and deviceBody) when device switches from true -> false
        //Do not remove if Amazon device by default value will be zero'd by amazon: https://developer.amazon.com/docs/policy-center/advertising-id.html
        if (!PrivacyManager.getInstance().shouldSendAdIds() || explicitBlock) {
            deviceBody.remove(IFA); //remove it if value changes in between requests
            android.remove(ANDROID_ID);
            android.remove(GAID);
            android.remove(AMAZON_ADVERTISING_ID);
        }

        //this lmt value is legally required to be passed upward
        deviceBody.addProperty("lmt", limitAdTracking ? 1 : 0);

        boolean isGooglePlaySvcAvailable = Boolean.TRUE.equals(isGooglePlayServicesAvailable());
        android.addProperty("is_google_play_services_available", isGooglePlaySvcAvailable);

        String appSetId = platform.getAppSetId();
        if (!TextUtils.isEmpty(appSetId)) {
            android.addProperty("app_set_id", appSetId);
        }

        /// Battery
        Intent batteryStatus = (context != null) ?
                context.registerReceiver(null, new IntentFilter(Intent.ACTION_BATTERY_CHANGED)) : null;
        String batteryState;
        if (batteryStatus != null) {
            int level = 0;
            level = batteryStatus.getIntExtra(BatteryManager.EXTRA_LEVEL, -1);
            int scale = batteryStatus.getIntExtra(BatteryManager.EXTRA_SCALE, -1);
            if (level > 0 && scale > 0) {
                android.addProperty("battery_level", level / (float) scale);
            }
            int status = batteryStatus.getIntExtra(BatteryManager.EXTRA_STATUS, -1);

            if (status == -1) {
                batteryState = "UNKNOWN";
            } else if (status == BatteryManager.BATTERY_STATUS_CHARGING || status == BatteryManager.BATTERY_STATUS_FULL) {
                switch (batteryStatus.getIntExtra(BatteryManager.EXTRA_PLUGGED, -1)) {
                    case BatteryManager.BATTERY_PLUGGED_USB:
                        batteryState = "BATTERY_PLUGGED_USB";
                        break;
                    case BatteryManager.BATTERY_PLUGGED_AC:
                        batteryState = "BATTERY_PLUGGED_AC";
                        break;
                    case BatteryManager.BATTERY_PLUGGED_WIRELESS:
                        batteryState = "BATTERY_PLUGGED_WIRELESS";
                        break;
                    default:
                        batteryState = "BATTERY_PLUGGED_OTHERS";
                }
            } else {
                batteryState = "NOT_CHARGING";
            }
        } else {
            batteryState = "UNKNOWN";
        }

        android.addProperty("battery_state", batteryState);

        /// Battery saver (only available from Lollipop onward)
        PowerManager powerManager = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
        android.addProperty(
                "battery_saver_enabled",
                (powerManager != null && powerManager.isPowerSaveMode()) ? 1 : 0
        );

        /// Network Connection
        if (PermissionChecker.checkCallingOrSelfPermission(context, Manifest.permission.ACCESS_NETWORK_STATE) == PermissionChecker.PERMISSION_GRANTED) {
            String connectionType = "NONE";
            String connectionTypeDetail = ConnectionTypeDetail.UNKNOWN;

            ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (cm != null) {
                NetworkInfo info = cm.getActiveNetworkInfo();
                if (info != null) {
                    switch (info.getType()) {
                        case ConnectivityManager.TYPE_BLUETOOTH:
                            connectionType = "BLUETOOTH";
                            break;
                        case ConnectivityManager.TYPE_ETHERNET:
                            connectionType = "ETHERNET";
                            break;
                        case ConnectivityManager.TYPE_MOBILE:
                            connectionType = "MOBILE";
                            connectionTypeDetail = getConnectionTypeDetail(info.getSubtype());
                            break;
                        case ConnectivityManager.TYPE_WIFI:
                        case ConnectivityManager.TYPE_WIMAX:
                            connectionType = "WIFI";
                            break;
                        default:
                            connectionType = "UNKNOWN";
                    }
                }
            }

            android.addProperty("connection_type", connectionType);
            android.addProperty("connection_type_detail", connectionTypeDetail);

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                if (cm.isActiveNetworkMetered()) {
                    String dataSaverStatus;
                    switch (cm.getRestrictBackgroundStatus()) {
                        case ConnectivityManager.RESTRICT_BACKGROUND_STATUS_ENABLED:
                            // Background data usage is blocked for this app. If
                            // possible, sdk should use less data in the foreground.
                            dataSaverStatus = "ENABLED";
                            break;
                        case ConnectivityManager.RESTRICT_BACKGROUND_STATUS_WHITELISTED:
                            // Current app is whitelisted. If possible, sdk should use
                            // less data in the foreground and background.
                            dataSaverStatus = "WHITELISTED";
                            break;
                        case ConnectivityManager.RESTRICT_BACKGROUND_STATUS_DISABLED:
                            // Data Saver is disabled. Since the device is on a metered
                            // network, if possible, the SDK should use less data.
                            dataSaverStatus = "DISABLED";
                            break;
                        default:
                            dataSaverStatus = "UNKNOWN";
                            break;
                    }
                    android.addProperty("data_saver_status", dataSaverStatus);
                    android.addProperty("network_metered", 1);
                } else {
                    android.addProperty("data_saver_status", "NOT_APPLICABLE");
                    android.addProperty("network_metered", 0);
                }
            }
        }

        /// Language/Locale
        android.addProperty("locale", Locale.getDefault().toString());
        android.addProperty("language", Locale.getDefault().getLanguage()); /// ISO-639-1-alpha-2
        android.addProperty("time_zone", TimeZone.getDefault().getID());

        /// Audio Values
        AudioManager audio = (AudioManager) context.getSystemService(Context.AUDIO_SERVICE);
        int max, current;
        float vol;
        if (audio != null) {
            max = audio.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
            current = audio.getStreamVolume(AudioManager.STREAM_MUSIC);
            vol = (float) current / (float) max;
            android.addProperty("volume_level", vol);
            android.addProperty("sound_enabled", current > 0 ? 1 : 0);
        }

        /// Storage Values
        final File cacheDirectory = cacheManager.getCache();
        final String cachePath = cacheDirectory.getPath();

        if (cacheDirectory.exists() && cacheDirectory.isDirectory()) {
            android.addProperty("storage_bytes_available", cacheManager.getBytesAvailable());
        }

        /// TV Values
        boolean isTV;
        if (Platform.MANUFACTURER_AMAZON.equals(Build.MANUFACTURER)) {
            final String AMAZON_FEATURE_FIRE_TV = "amazon.hardware.fire_tv";
            isTV = context.getApplicationContext().getPackageManager().hasSystemFeature(AMAZON_FEATURE_FIRE_TV);
        } else {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                // in later API versions, this is better check as some handheld devices could be connected to TV screen
                // and run in TV mode
                UiModeManager uiModeManager = (UiModeManager) context.getSystemService(Context.UI_MODE_SERVICE);
                isTV = (uiModeManager.getCurrentModeType() == Configuration.UI_MODE_TYPE_TELEVISION);
            } else {
                final String FEATURE_ANDROID_TV = "com.google.android.tv";
                final String FEATURE_HW_TOUCHSCREEN = "android.hardware.touchscreen";
                //has feature flag for Android TV OR Does Not have H/W Feature Touchscreen
                isTV = context.getApplicationContext().getPackageManager().hasSystemFeature(FEATURE_ANDROID_TV) ||
                        !context.getApplicationContext().getPackageManager().hasSystemFeature(FEATURE_HW_TOUCHSCREEN);
            }
        }
        android.addProperty("is_tv", isTV);
        android.addProperty("os_api_level", Build.VERSION.SDK_INT);
        android.addProperty("app_target_sdk_version", context.getApplicationInfo().targetSdkVersion);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            android.addProperty("app_min_sdk_version", context.getApplicationInfo().minSdkVersion);
        }

        /// Non Market Install Values
        boolean canInstallNonMarket = false;
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                if (PackageManager.PERMISSION_GRANTED ==
                        context.checkCallingOrSelfPermission(Manifest.permission.REQUEST_INSTALL_PACKAGES)) {
                    canInstallNonMarket = context.getApplicationContext().getPackageManager().canRequestPackageInstalls();
                }
            } else {
                canInstallNonMarket = (Settings.Secure.getInt(context.getContentResolver(), Settings.Secure.INSTALL_NON_MARKET_APPS) == 1);
            }
        } catch (Settings.SettingNotFoundException e) {
            Log.e(TAG, "isInstallNonMarketAppsEnabled Settings not found", e);
        }
        android.addProperty("is_sideload_enabled", canInstallNonMarket);

        boolean isSDPresent = Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
        android.addProperty("sd_card_available", isSDPresent ? 1 : 0);
        android.addProperty("os_name", Build.FINGERPRINT);

        android.addProperty("vduid", "");

        deviceBody.addProperty("ua", uaString);

        // Device Extension
        JsonObject ext = new JsonObject();
        JsonObject vungle = new JsonObject();
        ext.add("vungle", vungle);
        deviceBody.add("ext", ext);

        vungle.add(Platform.MANUFACTURER_AMAZON.equals(Build.MANUFACTURER) ? "amazon" : "android", android);

        return deviceBody;
    }

    /**
     * Find and returns the sub type of mobile connection detail as per the of mobile network
     *
     * @return String sub type of mobile connection like GPRS, EVDO, LTE, etc.
     */
    private String getConnectionTypeDetail(int type) {
        switch (type) {
            case TelephonyManager.NETWORK_TYPE_1xRTT:
                return ConnectionTypeDetail.CDMA_1XRTT;
            case TelephonyManager.NETWORK_TYPE_CDMA:
                return ConnectionTypeDetail.WCDMA;
            case TelephonyManager.NETWORK_TYPE_EDGE:
                return ConnectionTypeDetail.EDGE;
            case TelephonyManager.NETWORK_TYPE_EHRPD:
                return ConnectionTypeDetail.HRPD;
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
                return ConnectionTypeDetail.CDMA_EVDO_0;
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
                return ConnectionTypeDetail.CDMA_EVDO_A;
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
                return ConnectionTypeDetail.CDMA_EVDO_B;
            case TelephonyManager.NETWORK_TYPE_GPRS:
                return ConnectionTypeDetail.GPRS;
            case TelephonyManager.NETWORK_TYPE_HSDPA:
                return ConnectionTypeDetail.HSDPA;
            case TelephonyManager.NETWORK_TYPE_HSUPA:
                return ConnectionTypeDetail.HSUPA;
            case TelephonyManager.NETWORK_TYPE_LTE:
                return ConnectionTypeDetail.LTE;
            case TelephonyManager.NETWORK_TYPE_NR:
            case TelephonyManager.NETWORK_TYPE_UNKNOWN:
            default:
                return ConnectionTypeDetail.UNKNOWN;
        }
    }

    /**
     * Generates and returns the user portion of the request body, which contains data-gathering
     * consent information and other user state.
     *
     * @return JsonObject which includes all the user information required to make the request.
     */
    private JsonObject getUserBody() {
        JsonObject userBody = new JsonObject();

        /// The consent status is saved on disk in a special cookie, retrieve it and extract the data.
        String status, source, messageVersion;
        long timestamp;

        Cookie consentCookie = repository.load(CONSENT_COOKIE, Cookie.class)
                .get(timeoutProvider.getTimeout(), TimeUnit.MILLISECONDS);

        //return status even if not gdpr
        if (consentCookie != null) {
            status = consentCookie.getString("consent_status");
            source = consentCookie.getString("consent_source");
            timestamp = consentCookie.getLong("timestamp");
            messageVersion = consentCookie.getString("consent_message_version");
        } else {
            /// If we have no consent status, default to unknown values. We use lack of data to infer
            /// this status.
            status = "unknown";
            source = "no_interaction";
            timestamp = 0L;
            //return empty string for non updated consent status/vungle default
            //returning a number will break jaeger
            messageVersion = "";
        }

        JsonObject gdpr = new JsonObject();

        /*
         * Status of consent regarding General Data Protection Regulation (GDPR). There are three
         * options for this field: ‘opted_in’ which means the user allows the use of personal data,
         * ‘opted_out’ which means the user denied the use of personal data, and ‘unknown’ which means
         * the user did not specify the use of personal data.
         */
        gdpr.addProperty("consent_status", status);

        /*
         * Source of consent regarding General Data Protection Regulation (GDPR). There are three
         * options for this field: ‘publisher’ which means the consent came from the publisher,
         * ‘vungle_modal’ which means the consent came from our vungle modal, and ‘no_interaction’
         * which means there was no interaction from the user to the vungle modal.
         */
        gdpr.addProperty("consent_source", source);

        /*
         * Timestamp in unix epoch seconds at the moment that consent was recorded.
         */
        gdpr.addProperty("consent_timestamp", timestamp);

        gdpr.addProperty("consent_message_version", TextUtils.isEmpty(messageVersion) ? "" : messageVersion);

        /// Include the GDPR state in the user model
        userBody.add("gdpr", gdpr);

        // The ccpa status is saved on disk in a special cookie, retrieve it and extract the data.
        String ccpaStatus;
        Cookie ccpaCookie = repository.load(CCPA_COOKIE, Cookie.class).get();
        //return status even if not ccpa
        if (ccpaCookie != null) {
            ccpaStatus = ccpaCookie.getString(Cookie.CCPA_CONSENT_STATUS);
        } else {
            // If we have no ccpa status, default to opted_in values. We use lack of data to infer this status.
            ccpaStatus = Cookie.CONSENT_STATUS_OPTED_IN;
        }

        JsonObject ccpa = new JsonObject();

        /*
         * Status of consent regarding California Consumer Privacy Act (CCPA). There are two
         * options for this field: ‘opted_in’ which means the user allows the use of personal data,
         * ‘opted_out’ which means the user denied the use of personal data.
         */
        ccpa.addProperty("status", ccpaStatus);

        // Include the CCPA state in the user model
        userBody.add("ccpa", ccpa);

        if (PrivacyManager.getInstance().getCoppaStatus() != PrivacyManager.COPPA.COPPA_NOTSET) {
            JsonObject coppaStatusJson = new JsonObject();
            coppaStatusJson.addProperty(Cookie.COPPA_STATUS_KEY, PrivacyManager.getInstance().getCoppaStatus().getValue());
            userBody.add(Cookie.COPPA_KEY, coppaStatusJson);
        }

        return userBody;
    }

    public boolean getOmEnabled() {
        return enableOm;
    }

    private String getUserAgentFromCookie() {
        String tempUserAgent;

        Cookie cookie = repository.load(USER_AGENT_ID_COOKIE, Cookie.class).get();
        if (cookie == null) {                            //Very first time or after clearing app data
            tempUserAgent = System.getProperty("http.agent");
        } else {
            tempUserAgent = cookie.getString("userAgent");
            if (TextUtils.isEmpty(tempUserAgent)) {        //Generally not possible but for safer side
                tempUserAgent = System.getProperty("http.agent");
            }
        }

        return tempUserAgent;
    }

    public long getRetryAfterHeaderValue(Response response) {
        String header = response.headers().get("Retry-After");
        try {
            return Long.parseLong(header) * 1000;
        } catch (NumberFormatException ex) {
            return 0;
        }
    }

    /**
     * External framework wrapping this SDK.
     */
    @Keep
    public enum WrapperFramework {
        admob,
        air,
        cocos2dx,
        corona,
        dfp,
        heyzap,
        marmalade,
        mopub,
        unity,
        fyber,
        ironsource,
        upsight,
        appodeal,
        aerserv,
        adtoapp,
        tapdaq,
        vunglehbs,
        max,
        none
    }

    @StringDef(value = {
            ConnectionTypeDetail.UNKNOWN,
            ConnectionTypeDetail.CDMA_1XRTT,
            ConnectionTypeDetail.WCDMA,
            ConnectionTypeDetail.EDGE,
            ConnectionTypeDetail.HRPD,
            ConnectionTypeDetail.CDMA_EVDO_0,
            ConnectionTypeDetail.CDMA_EVDO_A,
            ConnectionTypeDetail.CDMA_EVDO_B,
            ConnectionTypeDetail.GPRS,
            ConnectionTypeDetail.HSDPA,
            ConnectionTypeDetail.HSUPA,
            ConnectionTypeDetail.LTE
    })
    @Retention(RetentionPolicy.SOURCE)
    public @interface ConnectionTypeDetail {
        String UNKNOWN = "unknown";
        String CDMA_1XRTT = "cdma_1xrtt";
        String WCDMA = "wcdma";
        String EDGE = "edge";
        String HRPD = "hrpd";
        String CDMA_EVDO_0 = "cdma_evdo_0";
        String CDMA_EVDO_A = "cdma_evdo_a";
        String CDMA_EVDO_B = "cdma_evdo_b";
        String GPRS = "gprs";
        String HSDPA = "hsdpa";
        String HSUPA = "hsupa";
        String LTE = "LTE";
    }

    private static Set<Interceptor> networkInterceptors = new HashSet<>();

    private static Set<Interceptor> logInterceptors = new HashSet<>();

    public static class ClearTextTrafficException extends IOException {

        ClearTextTrafficException(String message) {
            super(message);
        }
    }

    @VisibleForTesting
    void overrideApi(VungleApi api) {
        this.api = api;
    }

    @VisibleForTesting
    public Boolean isGooglePlayServicesAvailable() {
        //to avoid checking for  Play service availability everytime from PackageMnagaer its optimal
        //to first check if we have a stored status, as this value is unlikely to change
        if (isGooglePlayServicesAvailable == null) {
            isGooglePlayServicesAvailable = getPlayServicesAvailabilityFromCookie();
        }
        if (isGooglePlayServicesAvailable == null) {
            isGooglePlayServicesAvailable = getPlayServicesAvailabilityFromAPI();
        }
        return isGooglePlayServicesAvailable;
    }

    @VisibleForTesting
    Boolean getPlayServicesAvailabilityFromCookie() {
        Boolean playSvcAvailability = null;
        Cookie cookie = repository.load(IS_PLAY_SERVICE_AVAILABLE, Cookie.class)
                .get(timeoutProvider.getTimeout(), TimeUnit.MILLISECONDS);
        if (cookie != null) {
            playSvcAvailability = cookie.getBoolean(IS_PLAY_SERVICE_AVAILABLE);
        }
        return playSvcAvailability;
    }

    @VisibleForTesting
    Boolean getPlayServicesAvailabilityFromAPI() {
        Boolean result = null;
        try {
            GoogleApiAvailabilityLight googleApiAvailabilityLight = GoogleApiAvailabilityLight.getInstance();
            if (googleApiAvailabilityLight != null) {
                result = googleApiAvailabilityLight.isGooglePlayServicesAvailable(context) == ConnectionResult.SUCCESS;
                addPlaySvcAvailabilityInCookie(result);
            }
        } catch (NoClassDefFoundError error) {
            Log.w(TAG, "Play services Not available");
            //this will be mostly due to integration issue where dependency is not present
            //or a device issue, in either case the status should not change
            result = false;
            try {
                addPlaySvcAvailabilityInCookie(result);
            } catch (DatabaseHelper.DBException e) {
                Log.w(TAG, "Failure to write GPS availability to DB");
            }
        } catch (Exception exception) {
            Log.w(TAG, "Unexpected exception from Play services lib.");
        }
        return result;
    }

    @VisibleForTesting
    void addPlaySvcAvailabilityInCookie(boolean isPlaySvcAvailable) throws DatabaseHelper.DBException {
        Cookie cookie = new Cookie(IS_PLAY_SERVICE_AVAILABLE);
        cookie.putValue(IS_PLAY_SERVICE_AVAILABLE, isPlaySvcAvailable);
        repository.save(cookie);
    }

    private JsonObject getExtBody() {
        Cookie cookie = repository.load(Cookie.CONFIG_EXTENSION, Cookie.class)
                .get(timeoutProvider.getTimeout(), TimeUnit.MILLISECONDS);
        String extension = "";
        if (cookie != null) {
            extension = cookie.getString(Cookie.CONFIG_EXTENSION);
        }
        if (TextUtils.isEmpty(extension)) {
            return null;
        }

        JsonObject extBody = new JsonObject();
        extBody.addProperty("config_extension", extension);
        return extBody;
    }
}
