package com.vungle.warren;

import android.content.Context;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.widget.RelativeLayout;

import com.vungle.warren.AdConfig.AdSize;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.admarkup.AdMarkup;
import com.vungle.warren.ui.view.VungleBannerView;
import com.vungle.warren.utility.AdMarkupDecoder;
import com.vungle.warren.utility.RefreshHandler;
import com.vungle.warren.utility.ViewUtility;
import com.vungle.warren.utility.WeakLoadAdCallback;
import com.vungle.warren.utility.WeakRunnable;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Vungle Banner Class
 */
public class VungleBanner extends RelativeLayout {

    //Summary:
    //There's no way for a developer to request a new size for a banner currently
    //if an ad is cached, dev will have to play new ad. close and then request.

    private static final String TAG = VungleBanner.class.getSimpleName();

    private String placementId;

    private int calculatedPixelWidth;
    private int calculatedPixelHeight;

    private boolean destroyed;
    private boolean renderWhenAvailable;
    private boolean disableRenderManagement;

    @Nullable
    private VungleBannerView bannerAd;
    private BannerAdConfig bannerAdConfig;

    private PlayAdCallback playAdCallback;
    private RefreshHandler refreshHandler;
    private boolean renderAdRequested;

    /**
     * Constructor for Banner Class, assumes that Banner ad has been loaded
     */
    protected VungleBanner(@NonNull Context context, String placementId, @Nullable String markup,
                           int adRefreshDuration, BannerAdConfig bannerAdConfig, PlayAdCallback callback) {
        super(context);
        long bannerAdStartTime = System.currentTimeMillis();
        VungleLogger.verbose(true, TAG, AdLoader.TT_DOWNLOAD_CONTEXT,
                String.format("Creating banner ad, id = %1$s, at: %2$d", placementId, bannerAdStartTime));
        this.placementId = placementId;
        this.bannerAdConfig = bannerAdConfig;
        AdSize adSize = bannerAdConfig.getAdSize();
        this.playAdCallback = callback;

        //set Ad Size
        calculatedPixelHeight = ViewUtility.dpToPixels(context, adSize.getHeight());
        calculatedPixelWidth = ViewUtility.dpToPixels(context, adSize.getWidth());

        SessionTracker.getInstance().trackAdConfig(bannerAdConfig);

        AdConfig adConfig = new AdConfig(bannerAdConfig);
        AdMarkup adMarkup = AdMarkupDecoder.decode(markup);
        bannerAd = Vungle.getBannerViewInternal(placementId, adMarkup, adConfig, this.playAdCallback);

        refreshHandler = new RefreshHandler(new WeakRunnable(refreshAdRunnable), adRefreshDuration * 1000L);

        VungleLogger.verbose(true, TAG, AdLoader.TT_DOWNLOAD_CONTEXT,
                String.format("Banner ad created, id = %1$s, elapsed time: %2$dms", placementId, System.currentTimeMillis() - bannerAdStartTime));
    }

    @Override
    protected void onWindowVisibilityChanged(int visibility) {
        super.onWindowVisibilityChanged(visibility);
        Log.d(TAG, "Banner onWindowVisibilityChanged: " + visibility);
        setAdVisibility(visibility == VISIBLE);
    }

    private Runnable refreshAdRunnable = new Runnable() {
        @Override
        public void run() {
            Log.d(TAG, "Refresh Timeout Reached");
            renderWhenAvailable = true;
            loadAdInternal();
        }
    };

    /**
     * Render Banner Ad as soon as its attach to window if available.
     */
    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        Log.d(TAG, "Banner onAttachedToWindow");

        if (disableRenderManagement) {
            return;
        }

        renderAd();
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();

        //TODO: Resolve case in which Banner View exists in a list/recyclerview
        // Ways to check if view being detached is not in a scrollview or listview
        // 1. Check if instance of listview/recyclerview
        // 2. use windowvisibility logic
        // Logic might also be that there must be a windowVisbilityChanged event right before
        // a finishbannercall; or last event must have been a 8 (gets invoked before this method)

        if (disableRenderManagement) {
            Log.d(TAG, "Banner onDetachedFromWindow: render management disabled, do nothing");
            return;
        }
        finishAdInternal(true);
    }

    /**
     * ONLY USE IF {@link VungleBanner#disableLifeCycleManagement(boolean)} is set to TRUE
     *
     * Starts rendering, must be invoked manually
     **/
    public void renderAd() {

        renderAdRequested = true;
        //If ad is not visible, delay rendering until onAttachedWindow/
        if (getVisibility() != View.VISIBLE) {
            return;
        }

        if (bannerAd == null) {
            if (canRender()) {
                renderWhenAvailable = true;
                loadAdInternal();
            }
            return;
        }

        View view = bannerAd.renderBannerView();
        if (view.getParent() != this) {
            //if there is no view added, render ad
            addView(view, calculatedPixelWidth, calculatedPixelHeight);
            Log.d(TAG, "Add VungleBannerView to Parent");
        }

        Log.d(TAG, "Rendering new ad for: " + placementId);

        //set both container sizes
        //in case developer sets the wrong layout params for the banner
        ViewGroup.LayoutParams bannerLayoutParams = getLayoutParams();
        if (bannerLayoutParams != null) {
            //Set Ad Size when Banner Ad is attached to Window
            bannerLayoutParams.height = calculatedPixelHeight;
            bannerLayoutParams.width = calculatedPixelWidth;
            requestLayout();
        }

        refreshHandler.start();
    }

    /**
     * ONLY USE IF {@link VungleBanner#disableLifeCycleManagement(boolean)} is set to TRUE
     *
     * Call when banner ad is no longer needed. If {@link VungleBanner#disableLifeCycleManagement(boolean)}
     * is set to true, this function must be called when ad is no longer needed.
     *
     * Can only be called after {@link VungleBanner#renderAd()} is invoked first.
     */
    public void finishAd() {
        finishAdInternal(true);
    }

    private void finishAdInternal(boolean isFinishedByApi) {
        synchronized (VungleBanner.this) {
            refreshHandler.clean();
            if (bannerAd != null) {
                bannerAd.finishDisplayingAdInternal(isFinishedByApi);
                bannerAd = null;
                try {
                    removeAllViews();
                } catch (Exception exception) {
                    Log.d(TAG, "Removing webview error: "  + exception.getLocalizedMessage());
                }
            }
        }
    }

    /**
     * ONLY USE IF {@link VungleBanner#disableLifeCycleManagement(boolean)} is set to TRUE
     *
     * Tears down the VungleBanner view: Banner instance can no longer be user once this method executes.
     * The parent Activity's onDestroy implementation is suggested to include a call to this method.
     *
     * This function must be called when ad is no longer needed.
     */
    public void destroyAd() {
        finishAdInternal(true);
        destroyed = true;
        playAdCallback = null;
    }

    /**
     * ONLY USE IF {@link VungleBanner#disableLifeCycleManagement(boolean)} is set to TRUE
     *
     * Invoke when ad is changing visibility (i.e. no longer in view). Required when view is being
     * recycled.
     *
     * This will tell us when to pause rendering and prevent usage of unnecessary resources
     * when ad is not visible.
     *
     * @param visible true is ad is in view, false if not (i.e. offscreen)
     */
    public void setAdVisibility(boolean visible) {
        if (visible && canRender()) {
            refreshHandler.start();
        } else {
            refreshHandler.pause();
        }

        if (bannerAd != null) {
            bannerAd.setAdVisibility(visible);
        }
    }

    @Override
    public void onWindowFocusChanged(boolean hasWindowFocus) {
        setAdVisibility(hasWindowFocus);
    }

    private boolean canRender() {
        return !destroyed && (!disableRenderManagement || renderAdRequested);
    }

    @Override
    protected void onVisibilityChanged(@NonNull View changedView, int visibility) {
        super.onVisibilityChanged(changedView, visibility);
        setAdVisibility(visibility == VISIBLE);
    }

    private LoadAdCallback loadAdCallback = new LoadAdCallback() {
        @Override
        public void onAdLoad(String placementId) {
            Log.d(TAG, "Ad Loaded : " + placementId);
            if (renderWhenAvailable && canRender()) {
                renderWhenAvailable = false;
                finishAdInternal(false);
                VungleBannerView ad = Vungle.getBannerViewInternal(VungleBanner.this.placementId,
                        (AdMarkup) null, new AdConfig(bannerAdConfig), playAdCallback);
                if (ad != null) {
                    bannerAd = ad;
                    renderAd();
                } else {
                    onError(VungleBanner.this.placementId, new VungleException(VungleException.AD_UNABLE_TO_PLAY));
                    VungleLogger.error(VungleBanner.class.getSimpleName() + "#loadAdCallback; onAdLoad", "VungleBannerView is null");
                }
            }
        }

        @Override
        public void onError(String placementId, VungleException exception) {
            Log.d(TAG, "Ad Load Error : " + placementId + " Message : " + exception.getLocalizedMessage());
            if (getVisibility() == View.VISIBLE && canRender()) {
                refreshHandler.start();
            }
        }
    };

    protected void loadAdInternal() {
        Log.d(TAG, "Loading Ad");
        Banners.loadBanner(placementId, bannerAdConfig, new WeakLoadAdCallback(loadAdCallback));
    }

    /**
     * For most uses cases when you retrieve a VungleBanner view, there are no additional methods or calls
     * that you will need to invoke for the VungleBanner to display an ad and release resources when it is
     * no longer needed. The VungleBanner in this case is in a MANAGED state, which is the default state.
     *
     * In special use cases, the mechanisms that VungleBanner uses to manage creation and teardown will not
     * always automatically work. For these cases, you will need to disable lifecycle management and handle it
     * manually. In theses cases setting true will set VungleBanner in an UN-MANAGED state.
     *
     * These cases include embedding a VungleBanner inside a recyclerview or listview. If not manually managed
     * ads will not correctly display or be served when the view is recycled.
     *
     *
     * {@link VungleBanner#renderAd()}} - invoked when the banner should be rendered.
     * {@link VungleBanner#finishAd()} - invoked when the banner is no longer needed.
     * {@link VungleBanner#setAdVisibility(boolean)} - call when banner is no longer visible on screen
     *
     * @param disable - true or false
     */
    public void disableLifeCycleManagement(boolean disable) {
        //disable onAttached and onDetached Logic
        disableRenderManagement = disable;
    }

}
