package com.vungle.warren;

import android.os.Handler;
import android.os.Looper;
import android.os.SystemClock;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.util.Log;

import com.vungle.warren.tasks.JobCreator;
import com.vungle.warren.tasks.JobInfo;
import com.vungle.warren.tasks.JobRunner;
import com.vungle.warren.tasks.runnable.JobRunnable;
import com.vungle.warren.tasks.utility.ThreadPriorityHelper;
import com.vungle.warren.utility.NetworkProvider;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;

/**
 * The VungleJobRunner is responsible for creating jobs.
 */

class VungleJobRunner implements JobRunner {

    private static Handler handler = new Handler(Looper.getMainLooper());
    private static final String TAG = VungleJobRunner.class.getSimpleName();
    private final ThreadPriorityHelper threadPriorityHelper;
    private final NetworkProvider networkProvider;
    private JobCreator creator;
    private Executor executor;
    private List<PendingJob> pendingJobs;
    private Runnable pendingRunnable;
    private long nextCheck = Long.MAX_VALUE;

    private static class PendingJob {
        private final long uptimeMillis;
        JobInfo info;

        PendingJob(long uptimeMillis, JobInfo info) {
            this.uptimeMillis = uptimeMillis;
            this.info = info;
        }
    }

    private static class PendingRunnable implements Runnable {
        WeakReference<VungleJobRunner> runner;

        PendingRunnable(WeakReference<VungleJobRunner> runner) {
            this.runner = runner;
        }

        @Override
        public void run() {
            VungleJobRunner ref = runner.get();
            if (ref != null) {
                ref.executePendingJobs();
            }
        }
    }

    VungleJobRunner(@NonNull JobCreator jobCreator,
                    @NonNull Executor executor,
                    @Nullable ThreadPriorityHelper threadPriorityHelper,
                    @NonNull NetworkProvider networkProvider) {
        this.creator = jobCreator;
        this.executor = executor;
        this.threadPriorityHelper = threadPriorityHelper;
        this.pendingJobs = new CopyOnWriteArrayList<>();
        this.networkProvider = networkProvider;
        this.pendingRunnable = new PendingRunnable(new WeakReference<>(this));
    }

    @Override
    public synchronized void execute(@NonNull JobInfo jobInfo) {
        // make a copy to avoid mutable side effects
        JobInfo jobInfoCopy = jobInfo.copy();

        String jobTag = jobInfoCopy.getJobTag();
        long delay = jobInfoCopy.getDelay();

        //clearing delay
        jobInfoCopy.setDelay(0);

        if (jobInfoCopy.getUpdateCurrent()) {
            for (PendingJob job : pendingJobs) {
                if (job.info.getJobTag().equals(jobTag)) {
                    Log.d(TAG, "replacing pending job with new " + jobTag);
                    pendingJobs.remove(job);
                }
            }
        }

        pendingJobs.add(new PendingJob(SystemClock.uptimeMillis() + delay, jobInfoCopy));
        executePendingJobs();
    }

    @Override
    public synchronized void cancelPendingJob(@NonNull String tag) {
        List<PendingJob> jobsToRemove = new ArrayList<>();
        for (PendingJob pendingJob : pendingJobs) {
            if (pendingJob.info.getJobTag().equals(tag)) {
                jobsToRemove.add(pendingJob);
            }
        }
        pendingJobs.removeAll(jobsToRemove);
    }

    private synchronized void executePendingJobs() {
        long now = SystemClock.uptimeMillis();
        long nextCheck = Long.MAX_VALUE;
        long waitingForNetwork = 0;

        for (PendingJob job : pendingJobs) {
            if (now >= job.uptimeMillis) {
                boolean canRun = true;
                if (job.info.getRequiredNetworkType() == JobInfo.NetworkType.CONNECTED
                        && networkProvider.getCurrentNetworkType() == NetworkProvider.TYPE_NONE) {
                    canRun = false;
                    waitingForNetwork++;
                }

                if (canRun) {
                    pendingJobs.remove(job);
                    executor.execute(new JobRunnable(job.info, creator, this, threadPriorityHelper));
                }
            } else {
                nextCheck = Math.min(nextCheck, job.uptimeMillis);
            }
        }

        if (nextCheck != Long.MAX_VALUE && nextCheck != this.nextCheck) {
            handler.removeCallbacks(pendingRunnable);
            handler.postAtTime(pendingRunnable, TAG, nextCheck);
        }
        this.nextCheck = nextCheck;
        if (waitingForNetwork > 0) {
            networkProvider.addListener(networkListener);
        } else {
            networkProvider.removeListener(networkListener);
        }
    }

    private final NetworkProvider.NetworkListener networkListener = new NetworkProvider.NetworkListener() {
        @Override
        public void onChanged(int type) {
            executePendingJobs();
        }
    };
}



