package com.vungle.warren;

import android.util.Log;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.log.LogManager;

public class VungleLogger {

    private static final String TAG = VungleLogger.class.getSimpleName();

    private static final VungleLogger _instance = new VungleLogger();

    public static final int LOGGER_MAX_ENTRIES = 100;

    /**
     * Property that specifies the active logging level for the logger object.
     * eg: if equal to `WARNING`, only messages with loggingLevel >= `WARNING` will be logged.
     */
    private LoggerLevel loggingLevel = LoggerLevel.DEBUG;

    private LogManager logManager;

    private VungleLogger() {
    }

    /**
     * Properly sets up the logger.
     *
     * @param logManager Logger manager
     * @param level      corresponds to the desired logging level for this instance
     * @param maxEntries is the maximum number of entries allowed in the log
     */
    static void setupLoggerWithLogLevel(@NonNull LogManager logManager, @NonNull LoggerLevel level, int maxEntries) {
        _instance.loggingLevel = level;
        _instance.logManager = logManager;
        _instance.logManager.setMaxEntries(maxEntries);
    }

    public static void verbose(@NonNull String context, @NonNull String message) {
        log(LoggerLevel.VERBOSE, context, message);
    }

    public static void verbose(boolean withConsoleLog, @Nullable String tag, @NonNull String context, @NonNull String message) {
        if (withConsoleLog) {
            Log.v(tag, "[" + context + "] " + message);
        }

        verbose(context, message);
    }

    public static void info(@NonNull String context, @NonNull String message) {
        log(LoggerLevel.INFO, context, message);
    }

    public static void info(boolean withConsoleLog, @Nullable String tag, @NonNull String context, @NonNull String message) {
        if (withConsoleLog) {
            Log.i(tag, "[" + context + "] " + message);
        }

        info(context, message);
    }

    public static void debug(@NonNull String context, @NonNull String message) {
        log(LoggerLevel.DEBUG, context, message);
    }

    public static void debug(boolean withConsoleLog, @Nullable String tag, @NonNull String context, @NonNull String message) {
        if (withConsoleLog) {
            Log.d(tag, "[" + context + "] " + message);
        }

        debug(context, message);
    }

    public static void warn(@NonNull String context, @NonNull String message) {
        log(LoggerLevel.WARNING, context, message);
    }

    public static void warn(boolean withConsoleLog, @Nullable String tag, @NonNull String context, @NonNull String message) {
        if (withConsoleLog) {
            Log.w(tag, "[" + context + "] " + message);
        }

        warn(context, message);
    }

    public static void error(@NonNull String context, @NonNull String message) {
        log(LoggerLevel.ERROR, context, message);
    }

    public static void error(boolean withConsoleLog, @Nullable String tag, @NonNull String context, @NonNull String message) {
        if (withConsoleLog) {
            Log.e(tag, "[" + context + "] " + message);
        }

        error(context, message);
    }

    public static void critical(boolean withConsoleLog, @Nullable String tag, @NonNull String context, @NonNull String message) {
        if (withConsoleLog) {
            Log.e(tag, "[" + context + "] " + message);
        }
        if (BuildConfig.DEBUG) {
            throw new RuntimeException("[" + context + "] " + message);
        }

        error(context, message);
    }

    public static void addCustomData(@NonNull String key, @NonNull String value) {
        if (_instance.logManager == null) {
            Log.d(TAG, "Please setup Logger first.");
            return;
        }

        _instance.logManager.addCustomData(key, value);
    }

    public static void removeCustomData(@NonNull String key) {
        if (_instance.logManager == null) {
            Log.d(TAG, "Please setup Logger first.");
            return;
        }

        _instance.logManager.removeCustomData(key);
    }

    private static boolean isLoggable(@NonNull LoggerLevel level) {
        return level.level >= _instance.loggingLevel.level;
    }

    /**
     * Enqueue a message to be added to the log.
     *
     * @param message a string representing the message that will be added to the log.
     * @param level   the logging level of the message being logged.
     * @param context optional string
     */
    private static void log(@NonNull LoggerLevel level, @NonNull String context, @NonNull String message) {
        if (_instance.logManager == null) {
            Log.d(TAG, "Please setup Logger first.");
            return;
        }

        if (!_instance.logManager.isLoggingEnabled()) {
            return;
        }

        if (!isLoggable(level)) {
            return;
        }

        _instance.logManager.saveLog(level, context, message, null, null);
    }

    @Keep
    public enum LoggerLevel {
        VERBOSE(0, "verbose"),
        DEBUG(1, "debug"),
        INFO(2, "info"),
        WARNING(3, "warn"),
        ERROR(4, "error"),
        CRASH(5, "crash");

        private int level;
        private String levelString;

        LoggerLevel(int level, @NonNull String levelString) {
            this.level = level;
            this.levelString = levelString;
        }

        @NonNull
        @Override
        public String toString() {
            return levelString;
        }
    }

}
