package com.vungle.warren.analytics;

import androidx.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;

import com.google.gson.JsonObject;
import com.vungle.warren.VungleApiClient;
import com.vungle.warren.model.AnalyticUrl;
import com.vungle.warren.network.Call;
import com.vungle.warren.network.Callback;
import com.vungle.warren.network.Response;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.VungleUrlUtility;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import static com.vungle.warren.VungleApiClient.ClearTextTrafficException;

public class VungleAnalytics implements AdAnalytics {

    private static final String TAG = VungleAnalytics.class.getSimpleName();

    private final VungleApiClient client;
    private final Repository repository;

    public VungleAnalytics(VungleApiClient client, Repository repository) {
        this.client = client;
        this.repository = repository;
    }

    @Override
    public String[] ping(@NonNull String[] urls) {
        if (urls.length == 0) return urls;

        ArrayList<String> failed = new ArrayList<>();
        for (String url : urls) {
            if (!TextUtils.isEmpty(url)) {
                try {
                    if (!client.pingTPAT(url)) {
                        failed.add(url);
                    } else {
                        repository.delete(new AnalyticUrl(url));
                    }
                } catch (ClearTextTrafficException e) {
                    Log.e(TAG, "Cleartext Network Traffic is Blocked : " + url);
                } catch (MalformedURLException e) {
                    try {
                        repository.delete(new AnalyticUrl(url));
                    } catch (DatabaseHelper.DBException dbException) {
                        Log.e(TAG, "DBException deleting : " + url);
                    }
                    Log.e(TAG, "Invalid Url : " + url);
                } catch (DatabaseHelper.DBException e) {
                    Log.e(TAG, "Can't delete sent ping URL : " + url);
                }
            }
        }
        return failed.toArray(new String[0]);
    }

    @Override
    public void ri(JsonObject object) {
        if (object == null) return;

        client.ri(object).enqueue(new Callback<JsonObject>() {

            @Override
            public void onResponse(@NonNull Call<JsonObject> call, Response<JsonObject> response) {
                Log.d(TAG, "send RI success");
            }

            @Override
            public void onFailure(Call<JsonObject> call, Throwable t) {
                Log.d(TAG, "send RI Failure");
            }
        });
    }

    @Override
    public String[] retryUnsent() {
        // Send any pending TPATs
        List<AnalyticUrl> analyticUrls = repository.loadAll(AnalyticUrl.class).get();
        if (analyticUrls != null && analyticUrls.size() > 0) {
            String[] urls = new String[analyticUrls.size()];
            for (int i = 0; i < urls.length ; i++) {
                urls[i] = analyticUrls.get(i).url;
            }
            return ping(urls);
        }
        return new String[0];
    }

    @Override
    public void saveVungleUrls(String[] urls) {
        for (String url : urls) {
            if (VungleUrlUtility.isVungleUrl(url)) {
                try {
                    repository.save(new AnalyticUrl(url));
                } catch (DatabaseHelper.DBException e) {
                    Log.e(TAG, "Can't save failed to ping URL : " + url);
                }
            }
        }
    }



}
