package com.vungle.warren.downloader;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.os.Build;
import android.text.TextUtils;
import android.util.Log;

import com.vungle.warren.AdLoader;
import com.vungle.warren.SessionTracker;
import com.vungle.warren.VungleLogger;
import com.vungle.warren.downloader.AssetDownloadListener.DownloadError.ErrorReason;
import com.vungle.warren.downloader.DownloadRequestMediator.Status;
import com.vungle.warren.error.VungleException;
import com.vungle.warren.model.SessionData;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.utility.FileUtility;
import com.vungle.warren.utility.NetworkProvider;
import com.vungle.warren.utility.VungleThreadPoolExecutor;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import javax.net.ssl.SSLException;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.core.util.Pair;
import okhttp3.Call;
import okhttp3.Headers;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.ResponseBody;
import okhttp3.internal.http.HttpHeaders;
import okhttp3.internal.http.RealResponseBody;
import okio.BufferedSink;
import okio.BufferedSource;
import okio.GzipSource;
import okio.Okio;

import static android.net.ConnectivityManager.TYPE_BLUETOOTH;
import static android.net.ConnectivityManager.TYPE_ETHERNET;
import static android.net.ConnectivityManager.TYPE_MOBILE;
import static android.net.ConnectivityManager.TYPE_MOBILE_DUN;
import static android.net.ConnectivityManager.TYPE_VPN;
import static android.net.ConnectivityManager.TYPE_WIFI;
import static android.net.ConnectivityManager.TYPE_WIMAX;
import static com.vungle.warren.downloader.AssetDownloadListener.DownloadError;
import static com.vungle.warren.downloader.AssetDownloadListener.Progress;
import static com.vungle.warren.downloader.AssetDownloadListener.Progress.ProgressStatus;
import static com.vungle.warren.downloader.DownloadRequest.Priority.CRITICAL;
import static com.vungle.warren.downloader.DownloadRequest.Priority.HIGHEST;
import static java.net.HttpURLConnection.HTTP_NOT_MODIFIED;
import static java.net.HttpURLConnection.HTTP_OK;
import static java.net.HttpURLConnection.HTTP_PARTIAL;

@SuppressLint("LogNotTimber")
public class AssetDownloader implements Downloader {

    public static final long VERIFICATION_WINDOW = TimeUnit.HOURS.toMillis(24);


    static final String DOWNLOAD_COMPLETE = "DOWNLOAD_COMPLETE";
    static final String LAST_MODIFIED = "Last-Modified";
    static final String ETAG = "ETag";
    static final String LAST_CACHE_VERIFICATION = "Last-Cache-Verification";
    static final String LAST_DOWNLOAD = "Last-Download";
    static final String DOWNLOAD_URL = "Download_URL";

    private static final String BYTES = "bytes";
    private static final String RANGE = "Range";

    private static final String ACCEPT_RANGES = "Accept-Ranges";
    private static final String CONTENT_ENCODING = "Content-Encoding";
    private static final String CONTENT_RANGE = "Content-Range";
    private static final String CONTENT_TYPE = "Content-Type";
    private static final String ACCEPT_ENCODING = "Accept-Encoding";
    private static final String IF_NONE_MATCH = "If-None-Match";
    private static final String IF_MODIFIED_SINCE = "If-Modified-Since";
    private static final String IF_RANGE = "If-Range";
    private static final String IDENTITY = "identity";
    private static final String GZIP = "gzip";
    private static final String META_POSTFIX_EXT = ".vng_meta";

    private static final String LOAD_CONTEXT = "AssetDownloader#load; loadAd sequence";
    private static final String KEY_TEMPLATE = "template";

    private static final int TIMEOUT = 30;
    private static final int PROGRESS_STEP = 5;
    private static final String TAG = AssetDownloader.class.getSimpleName();
    private static final int RETRY_COUNT_ON_CONNECTION_LOST = 5;
    private static final int CONNECTION_RETRY_TIMEOUT = 300;
    private static final int MAX_RECONNECT_ATTEMPTS = 10;
    private static final int RANGE_NOT_SATISFIABLE = 416;
    private static final long MAX_PERCENT = 100;

    @Nullable
    private final DownloaderCache cache;
    private final long timeWindow;

    //For testing
    int retryCountOnConnectionLost = RETRY_COUNT_ON_CONNECTION_LOST;
    int maxReconnectAttempts = MAX_RECONNECT_ATTEMPTS;
    int reconnectTimeout = CONNECTION_RETRY_TIMEOUT;

    private final NetworkProvider networkProvider;
    private final VungleThreadPoolExecutor downloadExecutor;
    private final OkHttpClient okHttpClient;
    private final ExecutorService uiExecutor;

    private static final int DOWNLOAD_CHUNK_SIZE = 2048; //Same as Okio Segment.SIZE
    private Map<String, DownloadRequestMediator> mediators = new ConcurrentHashMap<>();
    private List<DownloadRequest> transitioning = new ArrayList<>();
    private final Object addLock = new Object();

    private volatile int progressStep = PROGRESS_STEP;
    private boolean isCacheEnabled = true;

    public AssetDownloader(@NonNull NetworkProvider networkProvider,
                           @NonNull VungleThreadPoolExecutor downloadExecutor,
                           @NonNull ExecutorService uiExecutor) {
        this(null, 0, networkProvider, downloadExecutor, uiExecutor);
    }

    public AssetDownloader(@Nullable DownloaderCache cache,
                           long timeWindow,
                           @NonNull NetworkProvider networkProvider,
                           @NonNull VungleThreadPoolExecutor downloadExecutor,
                           @NonNull ExecutorService uiExecutor) {
        this.cache = cache;
        this.timeWindow = timeWindow;
        this.downloadExecutor = downloadExecutor;
        this.networkProvider = networkProvider;
        this.uiExecutor = uiExecutor;

        OkHttpClient.Builder builder = new OkHttpClient.Builder()
                .readTimeout(TIMEOUT, TimeUnit.SECONDS)
                .connectTimeout(TIMEOUT, TimeUnit.SECONDS)
                .cache(null)
                .followRedirects(true)
                .followSslRedirects(true);

        okHttpClient = builder.build();
    }

    @Override
    public synchronized void download(final DownloadRequest downloadRequest,
                                      final AssetDownloadListener downloadListener) {

        if (downloadRequest == null) {
            VungleLogger.error("AssetDownloader#download; loadAd sequence", "downloadRequest is null");
            if (downloadListener != null) {
                deliverError(null,
                        downloadListener,
                        new DownloadError(
                                -1,
                                new IllegalArgumentException("DownloadRequest is null"),
                                ErrorReason.REQUEST_ERROR));
            }
            return;
        }

        VungleLogger.verbose(true, TAG, AdLoader.TT_DOWNLOAD_CONTEXT,
                String.format("Waiting for download asset %1$s, at: %2$d", downloadRequest, System.currentTimeMillis()));

        transitioning.add(downloadRequest);

        //needs to be executed before any downloading runnable
        downloadExecutor.execute(new AssetDownloader.DownloadPriorityRunnable(new AssetPriority(CRITICAL, HIGHEST)) {
            @Override
            public void run() {
                VungleLogger.verbose(true, TAG, AdLoader.TT_DOWNLOAD_CONTEXT,
                        String.format("Start to download asset %1$s, at: %2$d", downloadRequest, System.currentTimeMillis()));
                try {
                    launchRequest(downloadRequest, downloadListener);
                } catch (IOException e) {
                    VungleLogger.error("AssetDownloader#download; loadAd sequence", "cannot launch " +
                            "request due to " + e);
                    Log.e(TAG, "Error on launching request", e);
                    deliverError(downloadRequest, downloadListener,
                            new DownloadError(-1, e, ErrorReason.REQUEST_ERROR));
                }
            }
        }, new Runnable() {
            @Override
            public void run() {
                deliverError(downloadRequest, downloadListener,
                        new DownloadError(-1, new VungleException(VungleException.OUT_OF_MEMORY), ErrorReason.REQUEST_ERROR));
            }
        });
    }

    private void launchRequest(DownloadRequest downloadRequest,
                               AssetDownloadListener downloadListener) throws IOException {
        synchronized (addLock) {
            DownloadRequestMediator mediator;

            synchronized (AssetDownloader.this) {

                if (downloadRequest.isCancelled()) {
                    transitioning.remove(downloadRequest);
                    Log.d(TAG, "Request " + downloadRequest.url + " is cancelled before starting");
                    Progress progress = new Progress();
                    progress.status = ProgressStatus.CANCELLED;

                    deliverError(downloadRequest, downloadListener, new AssetDownloadListener.DownloadError(
                            -1,
                            new IOException("Cancelled"),
                            ErrorReason.REQUEST_ERROR));

                    return;
                }

                mediator = mediators.get(mediatorKeyFromRequest(downloadRequest));

                if (mediator == null) {
                    transitioning.remove(downloadRequest);
                    mediator = makeNewMediator(downloadRequest, downloadListener);
                    mediators.put(mediator.key, mediator);
                    load(mediator);
                    return;
                }
            }

            try {
                mediator.lock();
                synchronized (AssetDownloader.this) {
                    transitioning.remove(downloadRequest);

                    if (mediator.is(Status.PUBLISHED) ||
                            (mediator.is(Status.CANCELLED) && !downloadRequest.isCancelled())) {
                        DownloadRequestMediator mediatorNew =
                                makeNewMediator(downloadRequest, downloadListener);
                        mediators.put(mediator.key, mediatorNew);
                        load(mediatorNew);
                    } else if (mediator.isCacheable) {
                        mediator.add(downloadRequest, downloadListener);
                        //Recheck with pause
                        if (mediator.is(Status.PAUSED)) {
                            load(mediator);
                        }
                    } else {
                        VungleLogger.warn("AssetDownloader#launchRequest; loadAd sequence", "request "
                                + downloadRequest + " is already running");
                        deliverError(
                                downloadRequest,
                                downloadListener,
                                new DownloadError(
                                        -1,
                                        new IllegalArgumentException("DownloadRequest is already running"),
                                        ErrorReason.REQUEST_ERROR)
                        );
                    }

                }
            } finally {
                mediator.unlock();
            }
        }
    }


    private DownloadRequestMediator makeNewMediator(DownloadRequest downloadRequest,
                                                    AssetDownloadListener downloadListener) throws IOException {

        File fileToSave;
        File metaFile;
        boolean cacheable = false;
        String key;
        if (!isCacheEnabled()) {
            fileToSave = new File(downloadRequest.path);
            metaFile = new File(fileToSave.getPath() + META_POSTFIX_EXT);
            key = downloadRequest.url + " " + downloadRequest.path;
        } else {
            fileToSave = cache.getFile(downloadRequest.url);
            metaFile = cache.getMetaFile(fileToSave);
            cacheable = true;
            key = downloadRequest.url;
        }

        Log.d(TAG, "Destination file " + fileToSave.getPath());

        return new DownloadRequestMediator(
                downloadRequest,
                downloadListener,
                fileToSave.getPath(),
                metaFile.getPath(),
                cacheable,
                key
        );
    }


    @Override
    public synchronized List<DownloadRequest> getAllRequests() {
        List<DownloadRequest> requests = new ArrayList<>();
        List<DownloadRequestMediator> mediatorList = new ArrayList<>(mediators.values());

        for (DownloadRequestMediator mediator : mediatorList) {
            requests.addAll(mediator.requests());
        }

        requests.addAll(transitioning);

        return requests;
    }

    private synchronized void load(final DownloadRequestMediator mediator) {
        addNetworkListener();
        mediator.set(DownloadRequestMediator.Status.IN_PROGRESS);

        downloadExecutor.execute(new DownloadPriorityRunnable(mediator) {
            @SuppressWarnings("ResultOfMethodCallIgnored")
            @Override
            public void run() {
                mediator.setRunnable(null);
                int reconnectRetries = 0;
                boolean done = false;
                Progress progress = new Progress();
                progress.timestampDownloadStart = System.currentTimeMillis();
                DownloadError downloadError = null;
                boolean isPaused = false;

                String url = mediator.url;
                String path = mediator.filePath;
                String metPath = mediator.metaPath;

                final File file = new File(path);
                final File metaFile = new File(metPath);

                if (cache != null && mediator.isCacheable) {
                    cache.startTracking(file);
                }

                while (!done) {
                    done = true;

                    Log.d(TAG, "Start load: url: " + url);

                    BufferedSink sink = null;
                    BufferedSource source = null;
                    long totalRead = 0;
                    Call call = null;
                    long downloaded;
                    int code = -1;
                    Response response = null;
                    try {

                        if (!mediator.is(Status.IN_PROGRESS)) {
                            Log.w(TAG, "Abort download, wrong state "
                                    + debugString(mediator));
                            return;
                        }

                        if (!isAnyConnected(mediator)) {
                            Log.d(TAG, "Request is not connected to required network");
                            VungleLogger.warn(LOAD_CONTEXT,
                                    String.format("No connected to required network requests in %s",
                                            debugString(mediator)));
                            throw new IOException("Not connected to correct network");
                        } else {
                            mediator.setConnected(true);
                        }

                        if (file.getParentFile() != null && !file.getParentFile().exists()) {
                            file.getParentFile().mkdirs();
                        }

                        downloaded = file.exists() ? file.length() : 0;

                        Log.d(TAG, "already downloaded : "
                                + downloaded
                                + ", file exists = "
                                + file.exists()
                                + debugString(mediator));

                        HashMap<String, String> metaMap = extractMeta(metaFile);

                        if (useCacheWithoutVerification(mediator, file, metaMap)) {
                            if (FileUtility.isVideoFile(mediator.key)) {
                               SessionTracker.getInstance().trackEvent(
                                       new SessionData.Builder().setEvent(SessionEvent.ADS_CACHED)
                                       .addData(SessionAttribute.URL, mediator.key)
                                       .addData(SessionAttribute.VIDEO_CACHED, "cached")
                                       .build()
                               );
                            }
                            mediator.set(Status.DONE);
                            Log.d(TAG, "Using cache without verification, " +
                                    "dispatch existing file");
                            return;
                        }

                        VungleLogger.verbose(true, TAG, AdLoader.TT_DOWNLOAD_CONTEXT,
                                String.format("Send network request: %1$s, at: %2$d", url, System.currentTimeMillis()));

                        Request.Builder requestBuilder = new Request.Builder().url(url);

                        appendHeaders(downloaded, file, metaMap, requestBuilder);

                        call = okHttpClient.newCall(requestBuilder.build());
                        response = call.execute();

                        long contentLength = getContentLength(response);

                        Log.d(TAG, "Response code: " + response.code() + " " + mediator);

                        code = response.code();

                        if (notModified(file, response, mediator, metaMap) ||
                                useCacheOnFail(mediator, file, metaMap, code)) {

                            if (code == HTTP_NOT_MODIFIED) {
                                metaMap.put(LAST_CACHE_VERIFICATION,
                                        String.valueOf(System.currentTimeMillis()));
                                saveMeta(metaFile, metaMap);
                                Log.d(TAG, "Verification success, " +
                                        "dispatch existing file");
                            } else {
                                Log.d(TAG, "Using local cache file despite response code = "
                                        + response.code());
                            }


                            mediator.set(Status.DONE);
                            return;
                        }

                        if (partialMalformed(downloaded, code, response, mediator)) {
                            //retry to get ad from the scratch
                            if (reconnectRetries++ < maxReconnectAttempts) {
                                deleteFileAndMeta(file, metaFile, false);
                                done = false;
                                continue;
                            }

                            throw new RequestException("Code: " + code);
                        }

                        if (!response.isSuccessful()) {
                            throw new RequestException("Code: " + code);
                        }

                        if (code != HttpURLConnection.HTTP_PARTIAL) {
                            downloaded = 0;
                            deleteFileAndMeta(file, metaFile, false);
                        }

                        FileUtility.deleteAndLogIfFailed(metaFile);

                        Headers headers = response.headers();

                        checkEncoding(file, metaFile, headers);
                        metaMap = makeMeta(metaFile, headers, url);

                        if (!HttpHeaders.hasBody(response)) {
                            VungleLogger.error(LOAD_CONTEXT,
                                    String.format("response has no body %s", response));
                            throw new IOException("Response body is null");
                        }

                        if (cache != null) {
                            cache.setCacheLastUpdateTimestamp(file, System.currentTimeMillis());
                        }

                        ResponseBody body = decodeGzipIfNeeded(response);

                        source = body.source();

                        Log.d(TAG, "Start download from bytes: " + downloaded
                                + debugString(mediator));

                        long offset = downloaded;
                        contentLength += offset;

                        Log.d(TAG, "final offset = " + offset);

                        sink = Okio.buffer(offset == 0 ? Okio.sink(file) : Okio.appendingSink(file));

                        long read;
                        progress.status = ProgressStatus.STARTED;
                        progress.sizeBytes = body.contentLength();
                        progress.startBytes = offset;

                        onProgressMediator(mediator, progress);
                        int current = 0;

                        while (mediator.is(Status.IN_PROGRESS)
                                && (read = (source.read(sink.buffer(), DOWNLOAD_CHUNK_SIZE))) != -1) {

                            if (!file.exists()) {
                                VungleLogger.error(LOAD_CONTEXT,
                                        String.format("file %s does not exist", file));
                                throw new RequestException("File is not existing");
                            }

                            sink.emit();

                            totalRead += read;

                            downloaded = offset + totalRead;

                            if (contentLength > 0) {
                                current = (int) ((downloaded * MAX_PERCENT) / contentLength);
                            }

                            if (!mediator.isConnected()) {
                                VungleLogger.error(LOAD_CONTEXT,
                                        String.format("mediator %s is not connected",
                                                debugString(mediator)));
                                throw new IOException("Request is not connected");
                            }

                            while (progress.progressPercent + progressStep <= current
                                    && progress.progressPercent + progressStep <= MAX_PERCENT) {

                                progress.status = ProgressStatus.IN_PROGRESS;
                                progress.progressPercent += progressStep;
                                onProgressMediator(mediator, progress);
                            }
                        }

                        sink.flush();

                        if (mediator.is(Status.IN_PROGRESS)) {

                            long now = System.currentTimeMillis();
                            metaMap.put(DOWNLOAD_COMPLETE, Boolean.TRUE.toString());
                            metaMap.put(LAST_CACHE_VERIFICATION, String.valueOf(now));
                            metaMap.put(LAST_DOWNLOAD, String.valueOf(now));
                            saveMeta(metaFile, metaMap);

                            if (FileUtility.isVideoFile(mediator.key)) {
                                SessionTracker.getInstance().trackEvent(
                                        new SessionData.Builder().setEvent(SessionEvent.ADS_CACHED)
                                                .addData(SessionAttribute.URL, mediator.key)
                                                .addData(SessionAttribute.VIDEO_CACHED, "cdn")
                                                .build()
                                );
                            }
                            //
                            mediator.set(Status.DONE);
                        } else {
                            progress.status = ProgressStatus.STATE_CHANGED;
                            onProgressMediator(mediator, progress);
                            Log.d(TAG, "State has changed, cancelling download " + debugString(mediator));
                        }

                    } catch (Throwable throwable) {
                        VungleLogger.error(LOAD_CONTEXT,
                                String.format("exception, cannot load due to %1$s, state is %2$s",
                                        throwable, debugString(mediator)));
                        Log.e(TAG, "Exception on download", throwable);
                        if (!mediator.is(Status.CANCELLED)) {
                            mediator.set(Status.ERROR);
                        }

                        if (throwable instanceof IOException) {
                            boolean connected = isAnyConnected(mediator);

                            // Only if we have connection issues and no response
                            if (!connected && response == null
                                    && useCacheOnFail(mediator, file, extractMeta(metaFile), -1)) {

                                if (!mediator.is(Status.CANCELLED)) {
                                    mediator.set(Status.DONE);
                                }

                                return;
                            }


                            mediator.setConnected(connected);

                            downloadError = new DownloadError(code, throwable,
                                    mapExceptionToReason(throwable, connected));

                            if (!connected) {
                                progress.status = ProgressStatus.LOST_CONNECTION;
                                onProgressMediator(mediator, progress);

                                if (!mediator.is(Status.CANCELLED)
                                        && reconnectRetries++ < maxReconnectAttempts) {

                                    for (int i = 0; i < retryCountOnConnectionLost; i++) {

                                        sleep(reconnectTimeout);

                                        if (mediator.is(Status.CANCELLED)) {
                                            break;
                                        }

                                        Log.d(TAG, "Trying to reconnect");

                                        if (isAnyConnected(mediator)) {
                                            Log.d(TAG, "Reconnected, starting download again");
                                            done = false;
                                            mediator.setConnected(true);
                                            mediator.set(Status.IN_PROGRESS);
                                            break;
                                        }

                                        mediator.setConnected(false);
                                    }
                                }
                            }

                            synchronized (AssetDownloader.this) {

                                if (done && !mediator.isConnected() && mediator.isPausable()) {
                                    if (isAnyConnected(mediator)) {
                                        Log.d(TAG, "Reconnected, starting download again");
                                        done = false;
                                        mediator.setConnected(true);
                                        mediator.set(Status.IN_PROGRESS);
                                    } else {
                                        isPaused = pause(mediator, progress, downloadError);
                                    }
                                }
                            }

                        } else if (throwable instanceof RequestException) {
                            deleteFileAndMeta(file, metaFile, true);
                            downloadError = new DownloadError(code, throwable,
                                    ErrorReason.REQUEST_ERROR);
                        } else {
                            deleteFileAndMeta(file, metaFile, true);
                            downloadError = new DownloadError(code, throwable,
                                    ErrorReason.INTERNAL_ERROR);
                        }

                    } finally {

                        if (response != null && response.body() != null) {
                            response.body().close();
                        }

                        if (call != null) {
                            call.cancel();
                        }

                        Log.d(TAG, "request is done " + debugString(mediator));


                        if (done) {
                            switch (mediator.getStatus()) {
                                case Status.PAUSED:
                                    //no-op
                                    break;
                                case Status.DONE:
                                    onSuccessMediator(file, mediator);
                                    break;
                                case Status.ERROR:
                                    onErrorMediator(downloadError, mediator);
                                    break;
                                case Status.CANCELLED:
                                    onCancelledMediator(mediator);
                                    break;
                                default:

                                    if (!isPaused) {
                                        removeMediator(mediator);
                                    }
                                    break;
                            }

                            Log.d(TAG, "Done with request in state " + mediator.getStatus() +
                                    " " + debugString(mediator));
                        } else {
                            Log.d(TAG, "Not removing connections and listener "
                                    + debugString(mediator));
                        }


                        synchronized (AssetDownloader.this) {
                            removeNetworkListener();
                        }

                        FileUtility.closeQuietly(sink);
                        FileUtility.closeQuietly(source);

                        if (done && cache != null && mediator.isCacheable) {
                            cache.stopTracking(file);
                            if (!isCacheEnabled()) {
                                cache.clear();
                            } else {
                                cache.purge();
                            }
                        }
                    }
                }
            }
        }, new Runnable() {
            @Override
            public void run() {
                onErrorMediator(new DownloadError(-1, new VungleException(VungleException.OUT_OF_MEMORY), ErrorReason.REQUEST_ERROR), mediator);
            }
        });
    }

    private boolean useCacheOnFail(DownloadRequestMediator mediator, File file,
                                   Map<String, String> meta, int code) {
        return cache != null
                && mediator.isCacheable
                && code != HTTP_OK // Vungle requirements, conflicts with normal HTTP caching flow
                && code != RANGE_NOT_SATISFIABLE
                && code != HTTP_PARTIAL
                && Boolean.parseBoolean(meta.get(DOWNLOAD_COMPLETE))
                && file.exists()
                && file.length() > 0;
    }

    private synchronized void removeMediator(DownloadRequestMediator mediator) {
        mediators.remove(mediator.key);
    }

    private void addNetworkListener() {
        Log.d(TAG, "Adding network listner");
        networkProvider.addListener(networkListener);
    }

    private boolean pause(DownloadRequestMediator mediator,
                          Progress progress,
                          DownloadError error) {
        if (mediator.is(Status.CANCELLED) || isAnyConnected(mediator))
            return false;

        boolean anyPaused = false;
        progress.status = ProgressStatus.PAUSED;
        Progress copy = Progress.copy(progress);

        for (Pair<DownloadRequest, AssetDownloadListener> pair : mediator.values()) {

            DownloadRequest request = pair.first;

            if (request == null)
                continue;

            if (!request.pauseOnConnectionLost) {
                mediator.remove(request);
                deliverError(request, pair.second, error);
                continue;
            }

            mediator.set(Status.PAUSED);
            anyPaused = true;

            Log.d(TAG, "Pausing download " + debugString(request));

            deliverProgress(copy, pair.first, pair.second);
        }

        if (!anyPaused)
            mediator.set(Status.ERROR);

        Log.d(TAG, "Attempted to pause - " + (mediator.getStatus() == Status.PAUSED));

        return anyPaused;
    }

    private void sleep(long time) {
        try {
            Thread.sleep(Math.max(0, time));
        } catch (InterruptedException ie) {
            Log.e(TAG, "InterruptedException ", ie);
            Thread.currentThread().interrupt();
        }
    }


    private HashMap<String, String> makeMeta(File metaFile, Headers headers, String URL) {
        HashMap<String, String> metaMap;
        metaMap = new HashMap<>();
        metaMap.put(DOWNLOAD_URL, URL);
        metaMap.put(ETAG, headers.get(ETAG));
        metaMap.put(LAST_MODIFIED, headers.get(LAST_MODIFIED));
        metaMap.put(ACCEPT_RANGES, headers.get(ACCEPT_RANGES));
        metaMap.put(CONTENT_ENCODING, headers.get(CONTENT_ENCODING));
        saveMeta(metaFile, metaMap);
        return metaMap;
    }

    private void checkEncoding(File file, File metaFile, Headers headers) throws IOException {
        String contentEncoding = headers.get(CONTENT_ENCODING);
        if (contentEncoding != null &&
                !GZIP.equalsIgnoreCase(contentEncoding) &&
                !IDENTITY.equalsIgnoreCase(contentEncoding)) {
            deleteFileAndMeta(file, metaFile, false);
            VungleLogger.error("AssetDownloader#checkEncoding; loadAd sequence",
                    String.format("unknown %1$s %2$s ", CONTENT_ENCODING, contentEncoding));
            throw new IOException("Unknown " + CONTENT_ENCODING);
        }
    }

    private boolean useCacheWithoutVerification(DownloadRequestMediator mediator, File file, Map<String, String> meta) {
        if (meta == null || cache == null || !mediator.isCacheable)
            return false;

        String lastCheck = meta.get(LAST_CACHE_VERIFICATION);
        if (lastCheck == null || !file.exists()
                || !Boolean.parseBoolean(meta.get(DOWNLOAD_COMPLETE))) {
            return false;
        }

        long timeStamp;
        try {
            timeStamp = Long.parseLong(lastCheck);
        } catch (NumberFormatException ex) {
            return false;
        }

        return (timeWindow >= Long.MAX_VALUE - timeStamp) ||
                (timeStamp + timeWindow >= System.currentTimeMillis());
    }

    private boolean partialMalformed(long downloaded, int code,
                                     Response response,
                                     DownloadRequestMediator mediator) {
        return (code == HttpURLConnection.HTTP_PARTIAL
                && !satisfiesPartialDownload(response, downloaded, mediator))
                || code == RANGE_NOT_SATISFIABLE;
    }

    private void appendHeaders(long downloaded,
                               @NonNull File file,
                               @NonNull HashMap<String, String> metaMap,
                               @NonNull Request.Builder requestBuilder) {

        requestBuilder.addHeader(ACCEPT_ENCODING, IDENTITY);

        if (!file.exists() || metaMap.isEmpty())
            return;

        String eTag = metaMap.get(ETAG);
        String lastModified = metaMap.get(LAST_MODIFIED);

        if (Boolean.parseBoolean(metaMap.get(DOWNLOAD_COMPLETE))) {
            if (!TextUtils.isEmpty(eTag)) {
                requestBuilder.addHeader(IF_NONE_MATCH, eTag);
            }

            if (!TextUtils.isEmpty(lastModified)) {
                requestBuilder.addHeader(IF_MODIFIED_SINCE, lastModified);
            }

            return;
        }

        if (!BYTES.equalsIgnoreCase(metaMap.get(ACCEPT_RANGES)))
            return;

        // We want to make sure that server accepts bytes and
        // also we don't yet deal with gzip
        if (metaMap.get(CONTENT_ENCODING) != null
                && !IDENTITY.equalsIgnoreCase(metaMap.get(CONTENT_ENCODING)))
            return;

        requestBuilder.addHeader(RANGE, BYTES + "=" + downloaded + "-");

        if (!TextUtils.isEmpty(eTag)) {
            requestBuilder.addHeader(IF_RANGE, eTag);
        } else if (!TextUtils.isEmpty(lastModified)) {
            requestBuilder.addHeader(IF_RANGE, lastModified);
        }
    }

    private ResponseBody decodeGzipIfNeeded(Response networkResponse) {
        if (GZIP.equalsIgnoreCase(networkResponse.header(CONTENT_ENCODING))
                && HttpHeaders.hasBody(networkResponse)
                && networkResponse.body() != null) {
            GzipSource responseBody = new GzipSource(networkResponse.body().source());
            String contentType = networkResponse.header(CONTENT_TYPE);
            return new RealResponseBody(contentType, -1L, Okio.buffer(responseBody));
        }

        return networkResponse.body();
    }


    private synchronized void onCancelledMediator(@NonNull DownloadRequestMediator mediator) {
        Collection<DownloadRequest> all = mediator.requests();
        for (DownloadRequest request : all) {
            onCancelled(request);
        }
    }

    private void onCancelled(@NonNull final DownloadRequest request) {
        if (request.isCancelled())
            return;

        request.cancel();

        DownloadRequestMediator mediator = findMediatorForCancellation(request);

        if (mediator != null && mediator.getStatus() != Status.CANCELLED) {

            final Pair<DownloadRequest, AssetDownloadListener> pair = mediator.remove(request);

            final DownloadRequest childRequest = pair == null ? null : pair.first;
            final AssetDownloadListener listener = pair == null ? null : pair.second;

            if (mediator.values().isEmpty()) {
                mediator.set(Status.CANCELLED);
            }

            if (childRequest == null)
                return;

            final Progress progressEnd = new Progress();
            progressEnd.status = ProgressStatus.CANCELLED;
            deliverProgress(progressEnd, childRequest, listener);
        }

        removeNetworkListener();
    }

    private synchronized DownloadRequestMediator findMediatorForCancellation(DownloadRequest request) {

        List<DownloadRequestMediator> mediatorList = new ArrayList<>(2);
        mediatorList.add(mediators.get(getCacheableKey(request)));
        mediatorList.add(mediators.get(getNonCacheableKey(request)));

        for (DownloadRequestMediator candidate : mediatorList) {
            if (candidate == null)
                continue;

            for (DownloadRequest downloadRequest : candidate.requests()) {
                if (downloadRequest.equals(request))
                    return candidate;
            }
        }

        return null;
    }

    private void removeNetworkListener() {
        if (mediators.isEmpty()) {
            Log.d(TAG, "Removing listener");
            networkProvider.removeListener(networkListener);
        }
    }

    private int mapExceptionToReason(Throwable e, boolean connected) {
        @ErrorReason int reason;
        if (e instanceof RuntimeException) {
            reason = ErrorReason.INTERNAL_ERROR;
        } else if (!connected
                || e instanceof SocketException
                || e instanceof SocketTimeoutException) {
            reason = ErrorReason.CONNECTION_ERROR;
        } else if (e instanceof UnknownHostException
                || e instanceof SSLException) {
            reason = ErrorReason.REQUEST_ERROR;
        } else {
            reason = ErrorReason.DISK_ERROR;
        }
        return reason;
    }

    private long getContentLength(Response response) {
        if (response == null)
            return -1;

        String header = response.headers().get("Content-Length");
        if (TextUtils.isEmpty(header))
            return -1;

        try {
            return Long.parseLong(header);
        } catch (Throwable t) {
            return -1;
        }
    }

    private boolean notModified(@NonNull File file,
                                @Nullable Response response,
                                @NonNull DownloadRequestMediator mediator,
                                @NonNull HashMap<String, String> metaMap) {

        if (response == null || !file.exists() || file.length() <= 0 || !mediator.isCacheable) {
            return false;
        }

        int code = response.code();
        boolean downloadComplete = Boolean.parseBoolean(metaMap.get(DOWNLOAD_COMPLETE));

        if (downloadComplete && code == HTTP_NOT_MODIFIED) {
            Log.d(TAG, "304 code, data size matches file size " + debugString(mediator));
            return true;
        }

        return false;
    }

    private boolean satisfiesPartialDownload(Response response,
                                             long bytesLoaded,
                                             DownloadRequestMediator downloadRequest) {
        RangeResponse rangeResponse = new RangeResponse(response.headers().get(CONTENT_RANGE));
        boolean result = response.code() == HTTP_PARTIAL
                && BYTES.equalsIgnoreCase(rangeResponse.dimension)
                && rangeResponse.rangeStart >= 0
                && bytesLoaded == rangeResponse.rangeStart;
        Log.d(TAG, "satisfies partial download: " + result + " " + debugString(downloadRequest));
        return result;
    }

    private String debugString(DownloadRequestMediator mediator) {
        return ", mediator url - " + mediator.url
                + ", path - " + mediator.filePath
                + ", th - " + Thread.currentThread().getName()
                + "id " + mediator;
    }

    private String debugString(DownloadRequest request) {
        return ", single request url - " + request.url
                + ", path - " + request.path
                + ", th - " + Thread.currentThread().getName()
                + "id " + request.id;
    }

    private boolean isAnyConnected(DownloadRequestMediator mediator) {
        for (DownloadRequest request : mediator.requests()) {

            if (request == null) {
                Log.d(TAG, "Request is null");
                continue;
            }

            if (isConnected(request))
                return true;
        }

        return false;
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private boolean isConnected(@NonNull DownloadRequest request) {
        int conType = networkProvider.getCurrentNetworkType();

        if (conType >= 0 && request.networkType == NetworkType.ANY)
            return true;

        int mapped;
        switch (conType) {
            case TYPE_ETHERNET:
            case TYPE_WIFI:
            case TYPE_WIMAX:
                mapped = NetworkType.WIFI;
                break;
            case TYPE_BLUETOOTH:
            case TYPE_VPN:
            case TYPE_MOBILE_DUN:
            case TYPE_MOBILE:
                mapped = NetworkType.CELLULAR;
                break;
            default:
                mapped = -1;
        }

        boolean connected = mapped > 0 && (request.networkType & mapped) == mapped;

        Log.d(TAG, "checking pause for type: " + conType
                + " connected " + connected
                + debugString(request));

        return connected;
    }

    @Override
    public boolean cancelAndAwait(@Nullable DownloadRequest request, long timeout) {
        if (request == null)
            return false;

        cancel(request);

        long until = System.currentTimeMillis() + Math.max(0, timeout);

        while (System.currentTimeMillis() < until) {
            DownloadRequestMediator mediator = findMediatorForCancellation(request);
            synchronized (this) {
                if (!transitioning.contains(request) && (mediator == null
                        || !mediator.requests().contains(request))) {
                    return true;
                }
            }

            sleep(10);
        }

        return false;
    }

    @Override
    public synchronized void cancel(@Nullable DownloadRequest request) {
        if (request == null)
            return;

        onCancelled(request);
    }

    @Override
    public synchronized void cancelAll() {
        Log.d(TAG, "Cancelling all");
        for (DownloadRequest downloadRequest : transitioning) {
            Log.d(TAG, "Cancel in transtiotion " + downloadRequest.url);
            cancel(downloadRequest);
        }

        Log.d(TAG, "Cancel in mediator " + mediators.values().size());
        for (DownloadRequestMediator request : mediators.values()) {
            Log.d(TAG, "Cancel in mediator " + request.key);
            onCancelledMediator(request);
        }
    }

    @Override
    public void setProgressStep(int value) {
        if (value != 0) {
            progressStep = value;
        }
    }

    @Override
    public synchronized void init() {
        if (cache != null) {
            cache.init();
        }
    }

    @Override
    public synchronized void clearCache() {
        if (cache != null) {
            cache.clear();
        }
    }

    private synchronized void onNetworkChanged(int connectionType) {
        Log.d(TAG, "Num of connections: " + mediators.values().size());
        for (DownloadRequestMediator mediator : mediators.values()) {

            if (mediator.is(Status.CANCELLED)) {
                Log.d(TAG, "Result cancelled");
                continue;
            }

            boolean connected = isAnyConnected(mediator);

            Log.d(TAG, "Connected = " + connected + " for " + connectionType);

            mediator.setConnected(connected);

            if (mediator.isPausable() && connected && mediator.is(Status.PAUSED)) {
                load(mediator);
                Log.d(TAG, "resumed " + mediator.key + " " + mediator);
            }

        }
    }


    private boolean responseVersionMatches(Response response, HashMap<String, String> metaMap) {

        Headers headers = response.headers();

        String etag = headers.get(ETAG);
        String lastModified = headers.get(LAST_MODIFIED);
        Log.d(TAG, "server etag: " + etag);
        Log.d(TAG, "server lastModified: " + lastModified);

        if (etag != null && !etag.equals(metaMap.get(ETAG))) {
            Log.d(TAG, "etags miss match current: " + metaMap.get(ETAG));
            return false;
        }

        if (lastModified != null && !lastModified.equals(metaMap.get(LAST_MODIFIED))) {
            Log.d(TAG, "lastModified miss match current: " + metaMap.get(LAST_MODIFIED));
            return false;
        }

        return true;
    }

    @IntDef(flag = true, value = {NetworkType.WIFI, NetworkType.CELLULAR, NetworkType.ANY})
    public @interface NetworkType {
        int CELLULAR = 1;
        int WIFI = 1 << 1;
        int ANY = CELLULAR | WIFI;
    }

    synchronized void shutdown() {
        cancel(null);
        transitioning.clear();
        mediators.clear();
        uiExecutor.shutdownNow();
        downloadExecutor.shutdownNow();
        try {
            downloadExecutor.awaitTermination(2, TimeUnit.SECONDS);
            uiExecutor.awaitTermination(2, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            Log.e(TAG, "InterruptedException ", e);
            Thread.currentThread().interrupt();
        }
    }

    private final NetworkProvider.NetworkListener networkListener = new NetworkProvider.NetworkListener() {
        @Override
        public void onChanged(int type) {
            Log.d(TAG, "Network changed: " + type);
            onNetworkChanged(type);
        }
    };

    public abstract static class DownloadPriorityRunnable implements Comparable, Runnable {
        private final static AtomicInteger seq = new AtomicInteger();
        private final int order = seq.incrementAndGet();

        private final DownloadRequestMediator mediator;
        private final AssetPriority priority;

        DownloadPriorityRunnable(DownloadRequestMediator mediator) {
            this.mediator = mediator;
            this.priority = mediator.priority;
            mediator.setRunnable(this);
        }

        DownloadPriorityRunnable(AssetPriority priority) {
            this.priority = priority;
            this.mediator = null;
        }

        AssetPriority getPriority() {
            return mediator != null ? mediator.getPriority() : priority;
        }

        @Override
        public int compareTo(@NonNull Object o) {
            if (o instanceof DownloadPriorityRunnable) {
                DownloadPriorityRunnable other = (DownloadPriorityRunnable) o;
                int result = getPriority().compareTo(other.getPriority());
                if (result == 0) {
                    result = ((Integer) order).compareTo(other.order);
                }
                return result;
            }
            return -1;
        }
    }

    private HashMap<String, String> extractMeta(File file) {
        return FileUtility.readMap(file.getPath());
    }

    private void saveMeta(@NonNull File metaFile, @NonNull HashMap<String, String> meta) {
        FileUtility.writeMap(metaFile.getPath(), meta);
    }

    /**
     * Do not use except testing
     */
    @VisibleForTesting
    synchronized void setDownloadedForTests(boolean downloaded, String url, String path) {
        List<File> files = new ArrayList<>(2);
        if (cache != null) {
            try {
                files.add(cache.getMetaFile(cache.getFile(url)));
            } catch (IOException e) {
                Log.e(TAG, "Cannot add or get meta file", e);
                throw new RuntimeException("Failed to get file for request");
            }
        }

        files.add(new File(path + META_POSTFIX_EXT));

        for (File file : files) {
            HashMap<String, String> meta = extractMeta(file);
            meta.put(DOWNLOAD_COMPLETE, Boolean.valueOf(downloaded).toString());
            FileUtility.writeSerializable(file, meta);
        }
    }


    private void copyToDestination(File from, File to,
                                   Pair<DownloadRequest, AssetDownloadListener> pair) {

        if (to.exists())
            FileUtility.deleteAndLogIfFailed(to);

        FileInputStream inStream = null;
        FileOutputStream outStream = null;

        if (to.getParentFile() != null && !to.getParentFile().exists()) {
            to.getParentFile().mkdirs();
        }

        try {
            inStream = new FileInputStream(from);
            outStream = new FileOutputStream(to);
            FileChannel inChannel = inStream.getChannel();
            FileChannel outChannel = outStream.getChannel();
            inChannel.transferTo(0, inChannel.size(), outChannel);
            Log.d(TAG, "Copying: finished " + pair.first.url + " copying to " + to.getPath());
        } catch (IOException e) {
            VungleLogger.error("AssetDownloader#copyToDestination; loadAd sequence",
                    String.format("cannot copy from %1$s(%2$s) to %3$s due to %4$s",
                            from.getPath(), pair.first.url, to.getPath(), e));
            deliverError(pair.first, pair.second,
                    new AssetDownloadListener.DownloadError(-1, e,
                            AssetDownloadListener.DownloadError.ErrorReason.DISK_ERROR));
            Log.d(TAG, "Copying: error" + pair.first.url + " copying to " + to.getPath());
        } finally {
            FileUtility.closeQuietly(inStream);
            FileUtility.closeQuietly(outStream);
        }

    }

    private void deliverSuccess(Pair<DownloadRequest, AssetDownloadListener> pair, File dest) {
        if (pair.second != null) {
            pair.second.onSuccess(dest, pair.first);
        }
    }

    private void onSuccessMediator(@NonNull final File file,
                                   @NonNull final DownloadRequestMediator mediator) {

        Log.d(TAG, "OnComplete - Removing connections and listener " + mediator);

        try {
            mediator.lock();

            List<Pair<DownloadRequest, AssetDownloadListener>> children =
                    mediator.values();

            if (!file.exists()) {
                VungleLogger.error("AssetDownloader#onSuccessMediator; loadAd sequence",
                        String.format("File %1$s does not exist; mediator %2$s ", file.getPath(),
                                debugString(mediator)));
                onErrorMediator(
                        new DownloadError(
                                -1,
                                new IOException("File is deleted"),
                                ErrorReason.DISK_ERROR
                        ),
                        mediator
                );
                return;
            }

            if (cache != null && mediator.isCacheable) {
                cache.onCacheHit(file, children.size());
                cache.setCacheLastUpdateTimestamp(file, System.currentTimeMillis());
            }

            for (Pair<DownloadRequest, AssetDownloadListener> pair : children) {
                File destFile = new File(pair.first.path);


                if (!destFile.equals(file)) {
                    copyToDestination(file, destFile, pair);
                } else {
                    destFile = file;
                }

                Log.d(TAG, "Deliver success:" + pair.first.url
                        + " dest file: " + destFile.getPath());

                deliverSuccess(pair, destFile);
            }

            removeMediator(mediator);
            mediator.set(Status.PUBLISHED);
            Log.d(TAG, "Finished " + debugString(mediator));
        } finally {
            mediator.unlock();
        }
    }

    private String mediatorKeyFromRequest(@NonNull DownloadRequest request) {
        return isCacheEnabled()
                ? getCacheableKey(request)
                : getNonCacheableKey(request);
    }

    private String getNonCacheableKey(DownloadRequest request) {
        return request.url + " " + request.path;
    }

    private String getCacheableKey(DownloadRequest request) {
        return request.url;
    }

    private void onErrorMediator(@Nullable DownloadError downloadError,
                                 @NonNull DownloadRequestMediator mediator) {

        VungleLogger.error("AssetDownloader#onErrorMediator; loadAd sequence",
                String.format("Error %1$s occured; mediator %2$s", downloadError,
                        debugString(mediator)));
        if (downloadError == null) {
            downloadError = new DownloadError(
                    -1,
                    new RuntimeException(),
                    ErrorReason.INTERNAL_ERROR);
        }

        try {
            mediator.lock();

            for (Pair<DownloadRequest, AssetDownloadListener> pair : mediator.values()) {
                deliverError(pair.first, pair.second, downloadError);
            }

            removeMediator(mediator);
            mediator.set(Status.PUBLISHED);
        } finally {
            mediator.unlock();
        }
    }

    private void deliverError(@Nullable final DownloadRequest downloadRequest,
                              @Nullable final AssetDownloadListener downloadListener,
                              @NonNull final DownloadError error) {
        VungleLogger.error("AssetDownloader#deliverError; loadAd sequence",
                String.format("Delivering error %1$s; request %2$s", error,
                        downloadRequest != null ? debugString(downloadRequest) : "null"));
        if (downloadListener != null) {
            uiExecutor.execute(new Runnable() {
                @Override
                public void run() {
                    downloadListener.onError(error, downloadRequest);
                }
            });
        }
    }


    private void onProgressMediator(
            final DownloadRequestMediator mediator,
            Progress progress) {

        if (mediator == null)
            return;

        final Progress copy = Progress.copy(progress);
        Log.d(TAG, "Progress " + progress.progressPercent +
                " status " + progress.status + " " + mediator +
                " " + mediator.filePath);

        for (Pair<DownloadRequest, AssetDownloadListener> pair : mediator.values()) {
            deliverProgress(copy, pair.first, pair.second);
        }

    }

    private void deliverProgress(final Progress copy,
                                 final DownloadRequest downloadRequest,
                                 final AssetDownloadListener listener) {
        if (listener != null) {
            uiExecutor.execute(new Runnable() {
                @Override
                public void run() {
                    Log.d(TAG, "On progress " + downloadRequest);
                    listener.onProgress(copy, downloadRequest);
                }
            });
        }
    }

    private void deleteFileAndMeta(File file, File metaFile, boolean removeFromCache) {
        if (file == null)
            return;

        FileUtility.deleteAndLogIfFailed(file);

        if (metaFile != null) {
            FileUtility.deleteAndLogIfFailed(metaFile);
        }

        if (cache != null && isCacheEnabled()) {
            if (removeFromCache)
                cache.deleteAndRemove(file);
            else
                cache.deleteContents(file);
        }

    }

    @Override
    public boolean dropCache(@Nullable String serverPath) {
        if (cache != null && serverPath != null) {
            try {
                File file = cache.getFile(serverPath);
                Log.d(TAG, "Deleting " + file.getPath());
                return cache.deleteAndRemove(file);
            } catch (IOException e) {
                VungleLogger.error("AssetDownloader#dropCache; loadAd sequence",
                        String.format("Error %1$s occured", e));
                Log.e(TAG, "There was an error to get file", e);
            }
        }

        return false;
    }

    @Override
    public synchronized boolean isCacheEnabled() {
        return cache != null && isCacheEnabled;
    }

    @Override
    public synchronized void setCacheEnabled(boolean isEnabled) {
        isCacheEnabled = isEnabled;
    }

    @Override
    public void updatePriority(DownloadRequest request) {
        final DownloadRequestMediator mediator = findMediatorForCancellation(request);
        if (mediator != null) {
            Runnable runnable = mediator.getRunnable();
            if (runnable != null && downloadExecutor.remove(runnable)) {
                Log.d(TAG, "prio: updated to " + mediator.getPriority());
                downloadExecutor.execute(runnable, new Runnable() {
                    @Override
                    public void run() {
                        onErrorMediator(new DownloadError(-1, new VungleException(VungleException.OUT_OF_MEMORY), ErrorReason.REQUEST_ERROR), mediator);
                    }
                });
            }
        }
    }
}
