package com.vungle.warren.downloader;

import android.util.Base64;
import android.util.Log;

import com.vungle.warren.SizeProvider;
import com.vungle.warren.VungleLogger;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.utility.FileUtility;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

public class CleverCache implements DownloaderCache {

    private static final String TAG = CleverCache.class.getSimpleName();

    public static final String CC_DIR = "clever_cache";
    static final String ASSETS_DIR = "assets";
    static final String CACHE_META = "meta";
    private static final String META_POSTFIX_EXT = ".vng_meta";
    public static final String CACHE_TOUCH_JOURNAL = "cache_touch_timestamp";
    public static final String FAILED_TO_DELETE = "cache_failed_to_delete";

    private final HashMap<File, Long> cacheTouchTime = new HashMap<>();
    private final CacheManager cacheManager;
    private final CachePolicy<File> policy;
    private final long expirationAge;
    private final SizeProvider sizeProvider;

    private final Map<File, Integer> trackedFiles = new ConcurrentHashMap<>();
    private final HashSet<File> failedToDelete = new HashSet<>();

    public CleverCache(@NonNull CacheManager cacheManager,
                       @NonNull CachePolicy<File> cachePolicy,
                       @NonNull SizeProvider sizeProvider,
                       long expirationAge) {
        this.cacheManager = cacheManager;
        this.policy = cachePolicy;
        this.sizeProvider = sizeProvider;
        this.expirationAge = Math.max(0, expirationAge);
    }

    @Override
    public synchronized void init() {
        policy.load();
        loadTouchTimestamps();
        expirationCleanup();
        loadFailedToDelete();
        failedToDeleteCleanUp();
    }

    @Override
    public synchronized void startTracking(@NonNull File file) {
        Integer count = trackedFiles.get(file);

        //Make sure it's tracked by policy
        policy.put(file, 0);
        policy.save();

        if (count == null || count <= 0) {
            count = 1;
        } else {
            count++;
        }
        trackedFiles.put(file, count);

        Log.d(TAG, "Start tracking file: " + file + " ref count " + count);
    }

    @Override
    public synchronized void stopTracking(@NonNull File file) {
        Integer count = trackedFiles.get(file);
        if (count == null) {
            trackedFiles.remove(file);
            return;
        }

        count--;

        if (count <= 0)
            trackedFiles.remove(file);

        Log.d(TAG, "Stop tracking file: " + file + " ref count " + count);
    }

    @Override
    public synchronized void onCacheHit(@NonNull File file, long score) {
        policy.put(file, score);
        policy.save();

        Log.d(TAG, "Cache hit " + file + " cache touch updated");
        purge();
    }


    @VisibleForTesting
    @NonNull
    public synchronized File getAssetsDir() {
        File assetsDir = new File(getCacheDir(), ASSETS_DIR);

        if (!assetsDir.isDirectory() && assetsDir.exists()) {
            FileUtility.deleteAndLogIfFailed(assetsDir);
        }

        if (!assetsDir.exists())
            assetsDir.mkdirs();

        return assetsDir;
    }

    @VisibleForTesting
    public synchronized File getMetaDir() {
        File file = new File(getAssetsDir(), CACHE_META);

        if (!file.isDirectory()) {
            FileUtility.deleteAndLogIfFailed(file);
        }

        if (!file.exists())
            file.mkdirs();

        return file;
    }

    @NonNull
    @Override
    public synchronized List<File> purge() {
        failedToDeleteCleanUp();

        long target = sizeProvider.getTargetSize();
        long totalSize = FileUtility.size(getAssetsDir());

        Log.d(TAG, "Purge check current cache total: " + totalSize + " target: " + target);

        if (totalSize < target)
            return Collections.emptyList();

        Log.d(TAG, "Purge start");

        List<File> deleted = new ArrayList<>();
        List<File> files = policy.getOrderedCacheItems();

        integrityCleanup(files);

        totalSize = FileUtility.size(getAssetsDir());

        if (totalSize < target) {
            Log.d(TAG, "Cleaned up not tracked files, size is ok");
            return Collections.emptyList();
        }

        long fileSize;
        for (File candidate : files) {

            if (candidate == null)
                continue;

            if (isProtected(candidate)) continue;

            fileSize = candidate.length();

            if (deleteContents(candidate)) {
                totalSize -= fileSize;
                deleted.add(candidate);
                Log.d(TAG, "Deleted file: " + candidate.getName()
                        + " size: " + fileSize
                        + " total: " + totalSize
                        + " target: " + target);
                policy.remove(candidate);
                cacheTouchTime.remove(candidate);

                if (totalSize < target) {
                    target = sizeProvider.getTargetSize();
                    if (totalSize < target) {
                        Log.d(TAG, "Cleaned enough total: " + totalSize
                                + " target: " + target);
                        break;
                    }
                }
            }
        }

        if (deleted.size() > 0) {
            policy.save();
            saveTouchTimestamps();
        }

        Log.d(TAG, "Purge complete");
        return deleted;
    }

    private boolean isProtected(@NonNull File candidate) {
        Integer trackedCount = trackedFiles.get(candidate);

        if ((trackedCount != null && trackedCount > 0)) {
            Log.d(TAG, "File is tracked and protected : " + candidate);
            return true;
        }

        return false;
    }

    private void integrityCleanup(List<File> existingFiles) {
        //Defensive cleanup
        File metaDir = getMetaDir();
        File[] dirFiles = getAssetsDir().listFiles();

        if (dirFiles != null) {

            List<File> nonTrackedByPolicy = new ArrayList<>(Arrays.asList(dirFiles));
            nonTrackedByPolicy.removeAll(existingFiles);
            nonTrackedByPolicy.remove(metaDir);

            for (File file : nonTrackedByPolicy) {
                deleteContents(file);
                Log.d(TAG, "Deleted non tracked file " + file);
            }
        }
    }

    private synchronized void expirationCleanup() {
        //Additional cleanup to deleteAndRemove files even if cache is ok
        final long before = System.currentTimeMillis() - expirationAge;
        File[] files = getAssetsDir().listFiles();

        HashSet<File> current = new HashSet<>(cacheTouchTime.keySet());

        if (files != null && files.length > 0) {
            for (File file : files) {

                long savedTime = getCacheUpdateTimestamp(file);

                current.remove(file);

                if (isProtected(file))
                    continue;

                if (savedTime != 0 && savedTime > before)
                    continue;

                if (deleteContents(file)) {
                    cacheTouchTime.remove(file);
                    policy.remove(file);
                }

                Log.d(TAG, "Deleted expired file " + file);
            }

            for (File file : current) {
                cacheTouchTime.remove(file);
            }

            policy.save();
            saveTouchTimestamps();
        }
    }

    @Override
    public synchronized long getCacheUpdateTimestamp(@NonNull File file) {
        Long lastKnown = cacheTouchTime.get(file);
        return lastKnown == null ? file.lastModified() : lastKnown;
    }

    @Override
    public synchronized void setCacheLastUpdateTimestamp(@NonNull File file, long timestamp) {
        cacheTouchTime.put(file, timestamp);
        saveTouchTimestamps();
    }

    @Override
    public synchronized boolean deleteAndRemove(@NonNull File file) {
        if (deleteContents(file)) {
            cacheTouchTime.remove(file);
            policy.remove(file);

            policy.save();
            saveTouchTimestamps();
            failedToDelete.remove(file);
            saveFailedToDelete();
            return true;
        }
        failedToDelete.add(file);
        saveFailedToDelete();
        return false;
    }

    @Override
    public synchronized boolean deleteContents(@NonNull File file) {
        boolean fileDeleted = false;
        try {
            FileUtility.delete(file);
            fileDeleted = true;
            File meta = getMetaFile(file);
            FileUtility.delete(meta);
            return true;
        } catch (IOException ex) {
            VungleLogger.error("CleverCache#deleteContents; loadAd sequence",
                    String.format("Cannot delete %1$s for file %2$s; Error %3$s occured",
                            fileDeleted ? "meta" : "file", file.getPath(), ex));
            return false;
        }
    }

    @SuppressWarnings("squid:S4790")
    @Override
    @NonNull
    public synchronized File getFile(@NonNull String url) throws IOException {
        String algorithm = "SHA-256";
        String charset = "UTF-8";
        try {
            MessageDigest md = MessageDigest.getInstance(algorithm);
            md.update(url.getBytes(charset));
            byte[] digest = md.digest();

            String name = Base64.encodeToString(digest,
                    Base64.URL_SAFE | Base64.NO_WRAP);

            File file = new File(getAssetsDir(), name);
            policy.put(file, 0);
            return file;
        } catch (UnsupportedEncodingException e) {
            VungleLogger.error("CleverCache#getFile; loadAd sequence", "cannot encode url with " +
                    "charset = " + charset);
            throw new IOException(e);
        } catch (NoSuchAlgorithmException e) {
            VungleLogger.error("CleverCache#getFile; loadAd sequence", "cannot get instance of " +
                    "MessageDigest with algorithm " + algorithm);
            throw new IOException(e);
        }
    }

    @Override
    @NonNull
    public synchronized File getMetaFile(@NonNull File cacheFile) {
        return new File(getMetaDir(), cacheFile.getName() + META_POSTFIX_EXT);
    }

    @Override
    public synchronized void clear() {
        List<File> files = policy.getOrderedCacheItems();
        int deleteCount = 0;

        integrityCleanup(files);
        for (File file : files) {

            if (file == null || isProtected(file))
                continue;

            if (deleteContents(file)) {
                deleteCount++;
                policy.remove(file);
                cacheTouchTime.remove(file);
            }

        }

        if (deleteCount > 0) {
            policy.save();
            saveTouchTimestamps();
        }
    }


    private void loadTouchTimestamps() {
        Serializable ser = FileUtility.readSerializable(getTouchTimestampsFile());
        if (!(ser instanceof HashMap))
            return;

        try {
            HashMap<File, Long> oldMap = (HashMap<File, Long>) ser;
            cacheTouchTime.putAll(oldMap);
        } catch (ClassCastException ex) {
            VungleLogger.error("CleverCache#loadTouchTimestamps; loadAd sequence",
                    String.format("Error %1$s occurred; old map is not File -> Long", ex));

            FileUtility.deleteAndLogIfFailed(getTouchTimestampsFile());
        }
    }

    private void loadFailedToDelete() {
        Serializable ser = FileUtility.readSerializable(getFailedToDeleteFile());
        if (!(ser instanceof HashSet))
            return;

        try {
            HashSet<File> old = (HashSet<File>) ser;
            failedToDelete.addAll(old);
        } catch (ClassCastException ex) {
            VungleLogger.error("CleverCache#loadFailedToDelete;",
                    String.format("Error %1$s occurred; old set is not set of File", ex));

            FileUtility.deleteAndLogIfFailed(getFailedToDeleteFile());
        }
    }

    private void saveTouchTimestamps() {
        HashMap<File, Long> copy = new HashMap<>(cacheTouchTime);
        FileUtility.writeSerializable(getTouchTimestampsFile(), copy);
    }

    private void saveFailedToDelete() {
        File file = getFailedToDeleteFile();
        if (!failedToDelete.isEmpty()) {
            HashSet<File> copy = new HashSet<>(failedToDelete);
            FileUtility.writeSerializable(file, copy);
        } else if (file.exists()) {
            FileUtility.deleteAndLogIfFailed(file);
        }
    }

    private File getTouchTimestampsFile() {
        return new File(getCacheDir(), CACHE_TOUCH_JOURNAL);
    }

    private File getCacheDir(){
       File file = new File(cacheManager.getCache(), CC_DIR);
       if (!file.isDirectory())
            FileUtility.deleteAndLogIfFailed(file);

       if (!file.exists())
           file.mkdirs();

       return file;
    }

    private File getFailedToDeleteFile() {
        return new File(getCacheDir(), FAILED_TO_DELETE);
    }
    
    private void failedToDeleteCleanUp() {
        HashSet<File> candidates = new HashSet<>(failedToDelete);
        for (File candidate : candidates) {
            if (isProtected(candidate)) {
                continue;
            }
            deleteAndRemove(candidate);
        }
    }
}
