package com.vungle.warren.downloader;

import android.text.TextUtils;

import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;

public class DownloadRequest {

    public final int networkType;
    public final String url;
    public final String path;
    public final boolean pauseOnConnectionLost;
    private AtomicReference<AssetPriority> priority = new AtomicReference<>();
    final String id;
    public final String cookieString;
    private AtomicBoolean cancelled = new AtomicBoolean(false);
    private String advertisementId;

    @Override
    public String toString() {
        return "DownloadRequest{" +
                "networkType=" + networkType +
                ", priority=" + priority +
                ", url='" + url + '\'' +
                ", path='" + path + '\'' +
                ", pauseOnConnectionLost=" + pauseOnConnectionLost +
                ", id='" + id + '\'' +
                ", cookieString='" + cookieString + '\'' +
                ", cancelled=" + cancelled +
                ", advertisementId=" + advertisementId +
                '}';
    }

    @IntDef(value = {Priority.CRITICAL, Priority.HIGHEST, Priority.HIGH, Priority.LOWEST})
    public @interface Priority {
        int CRITICAL = -Integer.MAX_VALUE;//for internal usage in AssetDownloader only
        int HIGHEST = 0;
        int HIGH = 1;
        int LOWEST = Integer.MAX_VALUE;
    }

    public DownloadRequest(@NonNull String url, String path) {
        this(Downloader.NetworkType.ANY, Priority.HIGHEST, url, path, false, null);
    }

    public DownloadRequest(@NonNull String url, String path, String cookieString) {
        this(Downloader.NetworkType.ANY, Priority.HIGHEST, url, path, false, cookieString);
    }

    public DownloadRequest(@Downloader.NetworkType int networkType,
                           @Priority int priority,
                           @NonNull String url,
                           @NonNull String path,
                           boolean pauseOnConnectionLost,
                           String cookieString) {
        this(networkType, new AssetPriority(priority, Priority.HIGHEST), url, path, pauseOnConnectionLost, cookieString);
    }

    public DownloadRequest(@Downloader.NetworkType int networkType,
                           AssetPriority priority,
                           @NonNull String url,
                           @NonNull String path,
                           boolean pauseOnConnectionLost,
                           String cookieString) {
        this(networkType, priority, url, path, pauseOnConnectionLost, cookieString, null);
    }

    public DownloadRequest(@Downloader.NetworkType int networkType,
                           AssetPriority priority,
                           @NonNull String url,
                           @NonNull String path,
                           boolean pauseOnConnectionLost,
                           String cookieString,
                           String advertisementId) {
        if (TextUtils.isEmpty(path) || TextUtils.isEmpty(url)) {
            throw new IllegalArgumentException("Url or path is empty");
        }

        this.networkType = networkType;
        this.priority.set(priority);
        this.url = url;
        this.path = path;
        this.id = UUID.nameUUIDFromBytes((path + "_" + url).getBytes()).toString();
        this.pauseOnConnectionLost = pauseOnConnectionLost;
        this.cookieString = cookieString;
        this.advertisementId = advertisementId;
    }

    void cancel() {
        cancelled.set(true);
    }

    boolean isCancelled() {
        return cancelled.get();
    }

    public void setPriority(AssetPriority priority) {
        this.priority.set(priority);
    }

    public AssetPriority getPriority() {
        return priority.get();
    }

    public String getAdvertisementId() {
        return advertisementId;
    }
}
