package com.vungle.warren.downloader;

import androidx.annotation.NonNull;

import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.utility.FileUtility;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;

public class LRUCachePolicy implements CachePolicy<File> {

    static final String CACHE_POLICY_JOURNAL = "cache_policy_journal";

    private final CacheManager cacheManager;
    private final String dirName;
    private LinkedHashSet<File> files = new LinkedHashSet<>();

    public LRUCachePolicy(@NonNull CacheManager cacheManager, @NonNull String dirName) {
        this.cacheManager = cacheManager;
        this.dirName = dirName;
    }

    @Override
    public void put(@NonNull File obj, long score) {
        if (score > 0) {
            files.remove(obj);
        }

        files.add(obj);
    }

    @Override
    public List<File> getOrderedCacheItems() {
        return new ArrayList<>(files);
    }

    @Override
    public void remove(@NonNull File obj) {
        files.remove(obj);
    }

    @Override
    public void clean() {
        files.clear();
    }

    @Override
    public void save() {
        FileUtility.writeSerializable(getPolicyFile(), files);
    }

    private File getPolicyFile() {
        File homeDir = new File(cacheManager.getCache(), dirName);
        if (homeDir.exists() && !homeDir.isDirectory())
            FileUtility.deleteAndLogIfFailed(homeDir);

        if (!homeDir.exists()) {
            homeDir.mkdirs();
        }

        return new File(homeDir, CACHE_POLICY_JOURNAL);
    }

    @SuppressWarnings("unchecked")
    @Override
    public void load() {
        File policyFile = getPolicyFile();
        Serializable serializable = FileUtility.readSerializable(policyFile);

        if (serializable == null)
            return;

        if (serializable instanceof Collection) {
            files.addAll((Collection) serializable);
        } else {
            FileUtility.deleteAndLogIfFailed(policyFile);
        }
    }

}
