package com.vungle.warren.error;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;

/**
 * Vungle Exception class that we can use both internally and through our public API to message errors
 * and inconsistencies in business logic. This is meeant to be an external-facing error class.
 */
public class VungleException extends Exception {

    @IntDef({NO_SERVE, UNKNOWN_ERROR, CONFIGURATION_ERROR,
            AD_EXPIRED, MISSING_REQUIRED_ARGUMENTS_FOR_INIT,
            APPLICATION_CONTEXT_REQUIRED, OPERATION_ONGOING, VUNGLE_NOT_INTIALIZED,
            AD_UNABLE_TO_PLAY, PLACEMENT_NOT_FOUND, AD_FAILED_TO_DOWNLOAD, NO_AUTO_CACHED_PLACEMENT,
            SERVER_RETRY_ERROR, ALREADY_PLAYING_ANOTHER_AD, NO_SPACE_TO_INIT, NO_SPACE_TO_LOAD_AD,
            NO_SPACE_TO_LOAD_AD_AUTO_CACHED, NO_SPACE_TO_DOWNLOAD_ASSETS,
            NETWORK_ERROR, SERVER_ERROR, SERVER_TEMPORARY_UNAVAILABLE,
            ASSET_DOWNLOAD_RECOVERABLE, ASSET_DOWNLOAD_ERROR,
            OPERATION_CANCELED, DB_ERROR, RENDER_ERROR, INVALID_SIZE, INCORRECT_DEFAULT_API_USAGE,
            INCORRECT_BANNER_API_USAGE, WEB_CRASH, WEBVIEW_RENDER_UNRESPONSIVE, NETWORK_UNREACHABLE,
            NETWORK_PERMISSIONS_NOT_GRANTED, UNSUPPORTED_CONFIGURATION, SDK_VERSION_BELOW_REQUIRED_VERSION,
            MISSING_HBP_EVENT_ID, AD_PAST_EXPIRATION, AD_RENDER_NETWORK_ERROR, OUT_OF_MEMORY, CREATIVE_ERROR,
            INCORRECT_DEFAULT_API_USAGE_NATIVE
    })

    public @interface ExceptionCode {
    }

    public static final int NO_SERVE = 1;
    public static final int UNKNOWN_ERROR = 2;
    public static final int CONFIGURATION_ERROR = 3;
    public static final int AD_EXPIRED = 4;
    public static final int UNSUPPORTED_CONFIGURATION = 5;
    public static final int MISSING_REQUIRED_ARGUMENTS_FOR_INIT = 6;
    public static final int APPLICATION_CONTEXT_REQUIRED = 7;
    public static final int OPERATION_ONGOING = 8;
    public static final int VUNGLE_NOT_INTIALIZED = 9;
    public static final int AD_UNABLE_TO_PLAY = 10;
    public static final int AD_FAILED_TO_DOWNLOAD = 11;
    public static final int NO_AUTO_CACHED_PLACEMENT = 12;
    public static final int PLACEMENT_NOT_FOUND = 13;
    public static final int SERVER_RETRY_ERROR = 14;
    public static final int ALREADY_PLAYING_ANOTHER_AD = 15;
    public static final int NO_SPACE_TO_INIT = 16;
    public static final int NO_SPACE_TO_LOAD_AD = 17;
    public static final int NO_SPACE_TO_LOAD_AD_AUTO_CACHED = 18;
    public static final int NO_SPACE_TO_DOWNLOAD_ASSETS = 19;
    public static final int NETWORK_ERROR = 20;
    public static final int SERVER_ERROR = 21;
    public static final int SERVER_TEMPORARY_UNAVAILABLE = 22;
    public static final int ASSET_DOWNLOAD_RECOVERABLE = 23;
    public static final int ASSET_DOWNLOAD_ERROR = 24;
    public static final int OPERATION_CANCELED = 25;
    public static final int DB_ERROR = 26;
    public static final int RENDER_ERROR = 27;
    public static final int INVALID_SIZE = 28;
    public static final int INCORRECT_DEFAULT_API_USAGE = 29;
    public static final int INCORRECT_BANNER_API_USAGE = 30;
    public static final int WEB_CRASH = 31;
    public static final int WEBVIEW_RENDER_UNRESPONSIVE = 32;
    public static final int NETWORK_UNREACHABLE = 33;
    public static final int NETWORK_PERMISSIONS_NOT_GRANTED = 34;
    public static final int SDK_VERSION_BELOW_REQUIRED_VERSION = 35;
    public static final int MISSING_HBP_EVENT_ID = 36;
    public static final int AD_PAST_EXPIRATION = 37;
    public static final int AD_RENDER_NETWORK_ERROR = 38;
    public static final int OUT_OF_MEMORY = 39;
    public static final int CREATIVE_ERROR = 40;
    public static final int INCORRECT_DEFAULT_API_USAGE_NATIVE = 41;

    @ExceptionCode
    private final int exceptionCode;

    private String exceptionBody;

    public VungleException(@ExceptionCode int cause) {
        exceptionCode = cause;
        exceptionBody = "";
    }

    public VungleException(@ExceptionCode int cause, String body) {
        exceptionCode = cause;
        exceptionBody = body;
    }

    @VungleException.ExceptionCode
    public int getExceptionCode() {
        return exceptionCode;
    }

    @Override
    @NonNull
    public String getLocalizedMessage() {
        switch (exceptionCode) {
            case CONFIGURATION_ERROR:
                return "Configuration Error Occurred. Please check your appID and placementIDs, and try again when network connectivity is available.";
            case NO_SERVE:
                return "No advertisements are available for your current bid. Please try again later.";
            case UNKNOWN_ERROR:
                return "Unknown Error Occurred.";
            case AD_EXPIRED:
                return "The advertisement in the cache has expired and can no longer be played. Please load another ad";
            case MISSING_REQUIRED_ARGUMENTS_FOR_INIT:
                return "Please ensure all parameter for init marked as NonNull are provided, as they are essential for functioning of our SDK";
            case APPLICATION_CONTEXT_REQUIRED:
                return "Please provide Application context so our SDK can continue to support our API beyond Activity lifecycle";
            case OPERATION_ONGOING:
                return "There is already an ongoing operation for the action you requested. Please wait until the operation finished before starting another.";
            case VUNGLE_NOT_INTIALIZED:
                return "Vungle is not initialized/no longer initialized. Please call Vungle.init() to reinitialize.";
            case AD_UNABLE_TO_PLAY:
                return "Unable to play advertisement";
            case AD_FAILED_TO_DOWNLOAD:
                return "Advertisement failed to download";
            case NO_AUTO_CACHED_PLACEMENT:
                return "No auto-cached Placement on config";
            case PLACEMENT_NOT_FOUND:
                return "Placement is not valid";
            case SERVER_RETRY_ERROR:
                return "Remote Server responded with http Retry-After, SDK will retry this request.";
            case ALREADY_PLAYING_ANOTHER_AD:
                return "Vungle is already playing different ad.";
            case NO_SPACE_TO_INIT:
                return "There is not enough file system size on a device to initialize VungleSDK";
            case NO_SPACE_TO_LOAD_AD:
                return "There is not enough file system size on a device to request an ad.";
            case NO_SPACE_TO_LOAD_AD_AUTO_CACHED:
                return "There is not enough file system size on a device to request an ad for auto cache.";
            case NO_SPACE_TO_DOWNLOAD_ASSETS:
                return "There is not enough file system size on a device to download assets for an ad.";
            case NETWORK_ERROR:
                return "Network error. Try again later";
            case SERVER_ERROR:
                return "Server error";
            case SERVER_TEMPORARY_UNAVAILABLE:
                return "Server temporary unavailable. Try again later";
            case ASSET_DOWNLOAD_RECOVERABLE:
                return "Assets download failed. Try again later";
            case ASSET_DOWNLOAD_ERROR:
                return "Assets download failed.";
            case OPERATION_CANCELED:
                return "Operation was canceled";
            case INVALID_SIZE:
                return "Ad size is invalid";
            case DB_ERROR:
                return "Database error";
            case RENDER_ERROR:
                return "Render error";
            case INCORRECT_DEFAULT_API_USAGE:
                return "Cannot request or play Banner Ads from Vungle API, please use Banners.loadBanner() or Banners.getBanner()";
            case INCORRECT_BANNER_API_USAGE:
                return "Cannot request or play MREC or FullScreen Ads from Banner API, please use Vungle.loadAd(), " +
                        "Vungle.playAd()";
            case WEB_CRASH:
                return "Android web view has crashed";
            case WEBVIEW_RENDER_UNRESPONSIVE:
                return "Android web view render became unresponsive, please clean-up your Webview process if any";
            case NETWORK_UNREACHABLE:
                return "Network error. Please check if network is available and permission for " +
                        "network access is granted.";
            case NETWORK_PERMISSIONS_NOT_GRANTED:
                return "Network permissions not granted. Please check manifest for " +
                        "android.permission.INTERNET and android.permission.ACCESS_NETWORK_STATE";
            case UNSUPPORTED_CONFIGURATION:
                return "Placement not configured properly, check Vungle's dashboard or contact support for help with configuration";
            case SDK_VERSION_BELOW_REQUIRED_VERSION:
                return "The SDK minimum version should not be overridden. Will not work as expected.";
            case MISSING_HBP_EVENT_ID:
                return "No Event id passed for HBP";
            case AD_PAST_EXPIRATION:
                return "The cached Ad is no longer available due to expired timestamp";
            case AD_RENDER_NETWORK_ERROR:
                return "Ad rendering failed due to network connectivity issue";
            case OUT_OF_MEMORY:
                return "Out of memory";
            case CREATIVE_ERROR:
                return "Creative error: " + exceptionBody;
            case INCORRECT_DEFAULT_API_USAGE_NATIVE:
                return "Cannot request or play Native Ads from Vungle API, please use NativeAd API instead.";
            default:
                return "Unknown Exception Code";
        }
    }

    /**
     * Returns exception code for Throwable or -1
     * if throwable is not {@link VungleException}
     * or is null
     *
     * @param throwable {@link Throwable} to analyze for getting corresponding exceptionCode
     * @return exception code {@link ExceptionCode} or -1
     */
    public static int getExceptionCode(Throwable throwable) {
        if (throwable instanceof VungleException) {
            return ((VungleException) throwable).getExceptionCode();
        }

        return -1;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        VungleException that = (VungleException) o;

        return exceptionCode == that.exceptionCode;
    }

    @Override
    public int hashCode() {
        return exceptionCode;
    }
}
