package com.vungle.warren.log;

import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.utility.FileUtility;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.LineNumberReader;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Locale;

class BaseFilePersistor {

    private static final String TAG = BaseFilePersistor.class.getSimpleName();

    @Nullable
    protected File fileDir;

    protected String fileNamePrefix;

    protected String fileNameSuffix;

    protected String rootFileDirectory;

    /**
     * @param cacheDir
     * @param rootFileDirectory
     * @param fileNamePrefix
     * @param fileNameSuffix
     */
    public BaseFilePersistor(@Nullable File cacheDir, String rootFileDirectory,
                             String fileNamePrefix, String fileNameSuffix) {
        this.fileNamePrefix = fileNamePrefix;
        this.fileNameSuffix = fileNameSuffix;
        this.rootFileDirectory = rootFileDirectory;
        if (cacheDir != null) {
            fileDir = getOrCreateDirectory(cacheDir, rootFileDirectory);
        }
    }

    @Nullable
    protected File getOrCreateDirectory(@NonNull File baseDir, String newDirectory) {
        File dataDir = createFileOrDirectory(baseDir, newDirectory, true);
        if (dataDir == null || !dataDir.exists()) {
            return null;
        }
        return dataDir;
    }

    protected static String getDateText(long time) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS", Locale.getDefault());
        return sdf.format(new Date(time));
    }

    protected boolean renameFile(@NonNull File srcFile, @NonNull String destFileName) {
        boolean ret;
        if (!srcFile.exists() || !srcFile.isFile()) {
            return false;
        }

        File destFile = new File(fileDir, destFileName);
        if (destFile.exists()) {
            return false;
        }

        ret = srcFile.renameTo(destFile);

        return ret;
    }

    @Nullable
    protected File createFileOrDirectory(@NonNull File parentDir, @NonNull String fileName,
                                         boolean isDir) {
        boolean ret = false;
        File file = new File(parentDir, fileName);
        if (!file.exists()) {
            if (!isDir) {
                try {
                    ret = file.createNewFile();
                } catch (IOException e) {
                    Log.e(TAG, "Can't create new file " + file.getName(), e);
                }
            } else {
                ret = file.mkdir();
            }
        } else {
            ret = true;
        }

        return ret ? file : null;
    }

    @Nullable
    protected File[] getFilesWithSuffix(final String suffix) {
        if (isRootDirInvalid()) {
            return null;
        }

        return fileDir.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(suffix);
            }
        });
    }

    protected boolean appendLineToFile(File file, String value, @Nullable FileSaveCallback saveCallback) {
        FileWriter fileWriter = null;
        int lineCount = 0;
        try {
            fileWriter = new FileWriter(file, true);
            lineCount = getLineCount(file);
            if (lineCount > 0) {
                fileWriter.append("\n");
            }
            fileWriter.append(value);
            fileWriter.flush();
        } catch (IOException e) {
            if (saveCallback != null) {
                saveCallback.onFailure();
            }
            return false;
        } finally {
            FileUtility.closeQuietly(fileWriter);
        }
        if (saveCallback != null) {
            saveCallback.onSuccess(file, lineCount + 1);
        }
        return true;
    }

    @Nullable
    File getNewFile(String name) {
        return createFileOrDirectory(fileDir, name, false);
    }

    public void sortFilesByNewest(File[] files) {
        //sorts files by newest based on standard naming convention
        Arrays.sort(files, new Comparator<File>() {
            @Override
            public int compare(File o1, File o2) {
                return o2.getName().compareTo(o1.getName());
            }
        });
    }

    /**
     * Retrieves Linecount from Cache or Manually counts;
     *
     * @param file
     * @return
     */
    protected int getLineCount(File file) {
        if (file == null) {
            return 0;
        }

        int lineCount;
        LineNumberReader reader = null;
        try {
            reader = new LineNumberReader(new FileReader(file.getAbsolutePath()));
            while (reader.readLine() != null) ;
            lineCount = reader.getLineNumber();
        } catch (Exception ex) {
            lineCount = -1;
        } finally {
            FileUtility.closeQuietly(reader);
        }

        return lineCount;
    }

    public boolean isRootDirInvalid() {
        return fileDir == null || !fileDir.exists();
    }

    public void deleteFile(File file) {
        try {
            FileUtility.delete(file);
        } catch (IOException e) {
            Log.e(TAG, "File failed to delete");
        }
    }

    interface FileSaveCallback {
        void onSuccess(File file, int numLines);

        void onFailure();
    }

}
