package com.vungle.warren.log;

import android.util.Log;

import com.vungle.warren.VungleLogger;

import androidx.annotation.NonNull;

class JVMCrashCollector implements Thread.UncaughtExceptionHandler {

    private final Thread.UncaughtExceptionHandler defaultExceptionHandler;
    private LogManager.SdkLoggingEventListener sdkLoggingEventListener;
    private String collectFilter = LogManager.sDefaultCollectFilter;

    JVMCrashCollector(@NonNull LogManager.SdkLoggingEventListener sdkLoggingEventListener) {
        this.sdkLoggingEventListener = sdkLoggingEventListener;
        defaultExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
        Thread.setDefaultUncaughtExceptionHandler(this);
    }

    @Override
    public void uncaughtException(Thread thread, Throwable ex) {
        if (sdkLoggingEventListener.isCrashReportEnabled() && thread != null && ex != null) {
            boolean crashIsOurConcerned = false;
            Throwable cause = ex;
            while (cause != null) {
                final StackTraceElement[] stackTraceElements = cause.getStackTrace();
                for (final StackTraceElement stackTraceElement : stackTraceElements) {
                    if (stackTraceElement.getClassName().startsWith(collectFilter)) {
                        crashIsOurConcerned = true;
                        break;
                    }
                }
                cause = cause.getCause();
            }

            if (crashIsOurConcerned) {
                String message = Log.getStackTraceString(ex);
                String exClass = ex.getClass().toString();
                String threadId = String.valueOf(thread.getId());
                sdkLoggingEventListener.saveLog(VungleLogger.LoggerLevel.CRASH, "crash", message, exClass, threadId);
            }
        }

        if (defaultExceptionHandler != null) {
            defaultExceptionHandler.uncaughtException(thread, ex);
        }

    }

    void updateConfig(@NonNull String filter) {
        this.collectFilter = filter;
    }

}
