package com.vungle.warren.log;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import com.google.gson.Gson;
import com.vungle.warren.Vungle;
import com.vungle.warren.VungleApiClient;
import com.vungle.warren.VungleLogger;
import com.vungle.warren.persistence.CacheManager;
import com.vungle.warren.persistence.FilePreferences;

import java.io.File;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

public class LogManager {

    private static final String TAG = LogManager.class.getSimpleName();

    static final String LOGGER_PREFS_FILENAME = "com.vungle.sdk";
    @VisibleForTesting
    static final String PREFS_LOGGING_ENABLED_KEY = "logging_enabled";
    static final String PREFS_CRASH_REPORT_ENABLED_KEY = "crash_report_enabled";
    static final String PREFS_CRASH_COLLECT_FILTER_KEY = "crash_collect_filter";
    static final String PREFS_CRASH_BATCH_MAX_KEY = "crash_batch_max";

    public static final boolean DEFAULT_LOGGING_ENABLED = false;
    public static final boolean DEFAULT_CRASH_COLLECT_ENABLED = false;
    public static final int DEFAULT_CRASH_SEND_BATCH_MAX = 5; // default report batch is 5 send in one period
    public static String sDefaultCollectFilter = "com.vungle";

    private final LogPersister logPersister;
    private final LogSender logSender;
    private final Executor ioExecutor;
    private final FilePreferences prefs;
    private JVMCrashCollector jvmCrashCollector;

    private final AtomicBoolean loggingEnabled = new AtomicBoolean(DEFAULT_LOGGING_ENABLED);
    private final AtomicBoolean crashReportEnabled = new AtomicBoolean(DEFAULT_CRASH_COLLECT_ENABLED);
    private String crashCollectFilter = sDefaultCollectFilter;
    private AtomicInteger crashBatchMax = new AtomicInteger(DEFAULT_CRASH_SEND_BATCH_MAX);

    private boolean isCrashReportInit = false;

    private final String bundleID;
    private final Map<String, String> customDataMap = new ConcurrentHashMap<>();

    private Gson gson = new Gson();

    public LogManager(@NonNull Context context, @NonNull CacheManager cacheManager,
                      @NonNull VungleApiClient vungleApiClient, @NonNull Executor executor,
                      @NonNull FilePreferences prefs) {
        this(context, new LogPersister(cacheManager.getCache()), new LogSender(vungleApiClient, prefs), executor, prefs);
    }

    @VisibleForTesting
    LogManager(@NonNull Context context, @NonNull LogPersister logPersister,
               @NonNull LogSender logSender, @NonNull Executor executor,
               @NonNull FilePreferences prefs) {
        this.bundleID = context.getPackageName();
        this.logSender = logSender;
        this.logPersister = logPersister;
        this.ioExecutor = executor;
        this.prefs = prefs;

        this.logPersister.setSdkLoggingEventListener(sdkLoggingEventListener);
        Package vungleSdkPackage = Vungle.class.getPackage();
        if (vungleSdkPackage != null) {
            sDefaultCollectFilter = vungleSdkPackage.getName();
        }
        loggingEnabled.set(prefs.getBoolean(PREFS_LOGGING_ENABLED_KEY, DEFAULT_LOGGING_ENABLED));
        crashReportEnabled.set(prefs.getBoolean(PREFS_CRASH_REPORT_ENABLED_KEY, DEFAULT_CRASH_COLLECT_ENABLED));
        crashCollectFilter = prefs.getString(PREFS_CRASH_COLLECT_FILTER_KEY, sDefaultCollectFilter);
        crashBatchMax.set(prefs.getInt(PREFS_CRASH_BATCH_MAX_KEY, DEFAULT_CRASH_SEND_BATCH_MAX));

        initJvmCollector();
    }

    public void setMaxEntries(int maxEntries) {
        logPersister.setMaximumEntries(
                maxEntries <= 0 ? VungleLogger.LOGGER_MAX_ENTRIES : maxEntries);
    }

    public void setLoggingEnabled(boolean isEnabled) {
        if (loggingEnabled.compareAndSet(!isEnabled, isEnabled)) {
            prefs.put(PREFS_LOGGING_ENABLED_KEY, isEnabled);
            prefs.apply();
        }
    }

    public boolean isLoggingEnabled() {
        return loggingEnabled.get();
    }

    public boolean isCrashReportEnabled() {
        return crashReportEnabled.get();
    }

    /**
     * Add custom data with key-value pair to log entry
     *
     * @param key   for custom data
     * @param value for custom data
     */
    public void addCustomData(@NonNull String key, @NonNull String value) {
        customDataMap.put(key, value);
    }

    public void removeCustomData(@NonNull String key) {
        customDataMap.remove(key);
    }

    /**
     * save log document to disk file.
     */
    public void saveLog(@NonNull final VungleLogger.LoggerLevel level, @NonNull final String context, @NonNull final String message,
                        @Nullable final String exClass, @Nullable final String threadId) {
        final String eventID = "";
        final String userAgent = VungleApiClient.getHeaderUa();
        if (level == VungleLogger.LoggerLevel.CRASH && isCrashReportEnabled()) {
            synchronized (LogManager.this) {
                logPersister.saveCrashLogData(message, level.toString(), context, eventID, userAgent, bundleID, getCustomData(), exClass, threadId);
            }
        } else {
            ioExecutor.execute(new Runnable() {
                @Override
                public void run() {
                    if (isLoggingEnabled()) {
                        logPersister.saveLogData(message, level.toString(), context, eventID, userAgent, bundleID, getCustomData(), exClass, threadId);
                    }
                }
            });
        }
    }

    /**
     * Send any pending or crash logs to server.
     */
    public void sendSdkLogs() {
        sendCrashLogs();
        sendPendingLogs();
    }

    private String getCustomData() {
        return customDataMap.isEmpty() ? null : gson.toJson(customDataMap);
    }

    /**
     * If we happened to hit maximum entries, send logs to server.
     */
    private void sendPendingLogs() {
        if (!isLoggingEnabled()) {
            Log.d(TAG, "Logging disabled, no need to send log files.");
            return;
        }

        File[] pendingFiles = logPersister.getPendingFiles();
        if (pendingFiles == null || pendingFiles.length == 0) {
            Log.d(TAG, "No need to send empty files.");
            return;
        }

        logSender.sendLogs(pendingFiles);
    }

    public synchronized void updateCrashReportConfig(boolean enabled, @Nullable String collectFilter, int batchMax) {
        boolean enableChanged = (crashReportEnabled.get() != enabled);
        boolean filterChanged = !TextUtils.isEmpty(collectFilter) && !collectFilter.equals(crashCollectFilter);
        batchMax = Math.max(batchMax, 0);
        boolean batchMaxChanged = (crashBatchMax.get() != batchMax);
        if (enableChanged || filterChanged || batchMaxChanged) {
            if (enableChanged) {
                crashReportEnabled.set(enabled);
                prefs.put(PREFS_CRASH_REPORT_ENABLED_KEY, enabled);
            }
            if (filterChanged) {
                if ("*".equals(collectFilter)) {
                    // "*" means we collect all exceptions
                    crashCollectFilter = "";
                } else {
                    crashCollectFilter = collectFilter;
                }
                prefs.put(PREFS_CRASH_COLLECT_FILTER_KEY, crashCollectFilter);
            }
            if (batchMaxChanged) {
                crashBatchMax.set(batchMax);
                prefs.put(PREFS_CRASH_BATCH_MAX_KEY, batchMax);
            }
            prefs.apply();

            if (jvmCrashCollector != null) {
                jvmCrashCollector.updateConfig(crashCollectFilter);
            }

            if (enabled) {
                initJvmCollector();
            }
        }
    }

    /**
     * Send crash logs to server.
     */
    private void sendCrashLogs() {
        if (!isCrashReportEnabled()) {
            Log.d(TAG, "Crash report disabled, no need to send crash log files.");
            return;
        }

        File[] crashReportFiles = logPersister.getCrashReportFiles(crashBatchMax.get());
        if (crashReportFiles == null || crashReportFiles.length == 0) {
            Log.d(TAG, "No need to send empty crash log files.");
            return;
        }

        logSender.sendLogs(crashReportFiles);
    }

    synchronized void initJvmCollector() {
        if (!isCrashReportInit) {
            if (!isCrashReportEnabled()) {
                Log.d(TAG, "crash report is disabled.");
                return;
            }

            if (jvmCrashCollector == null) {
                jvmCrashCollector = new JVMCrashCollector(sdkLoggingEventListener);
            }

            jvmCrashCollector.updateConfig(crashCollectFilter);

            isCrashReportInit = true;
        }
    }

    interface SdkLoggingEventListener {
        void sendPendingLogs();

        boolean isCrashReportEnabled();

        void saveLog(@NonNull VungleLogger.LoggerLevel level, @NonNull String context, @NonNull String message,
                     @Nullable String exClass, @Nullable String threadId);
    }

    private SdkLoggingEventListener sdkLoggingEventListener = new SdkLoggingEventListener() {
        @Override
        public void sendPendingLogs() {
            LogManager.this.sendPendingLogs();
        }

        @Override
        public boolean isCrashReportEnabled() {
            return LogManager.this.isCrashReportEnabled();
        }

        @Override
        public void saveLog(@NonNull VungleLogger.LoggerLevel level, @NonNull String context, @NonNull String message,
                            @Nullable String exClass, @Nullable String threadId) {
            LogManager.this.saveLog(level, context, message, exClass, threadId);
        }
    };
}
