package com.vungle.warren.log;

import android.util.Log;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Arrays;
import java.util.TimeZone;
import java.util.UUID;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.vungle.warren.VungleLogger;

class LogPersister extends BaseFilePersistor {

    private static final String TAG = LogPersister.class.getSimpleName();

    private static final String SDK_LOGS_DIR = "sdk_logs";

    /**
     * sdk log file name pattern.
     * eg: log_{timestamps}
     */
    private static final String SDK_LOG_FILE_PATTERN = "log_";

    /**
     * sdk crash log file name pattern.
     * eg: crash_{timestamps}
     */
    private static final String SDK_CRASH_LOG_FILE_PATTERN = "crash_";

    /**
     * pending files pattern. Log files end with this string means ready to send to server.
     */
    private static final String SDK_LOG_FILE_PENDING = "_pending";

    /**
     * crash files pattern. Crash files end with this string means ready to send to server.
     */
    private static final String SDK_LOG_FILE_CRASH = "_crash";


    private LogManager.SdkLoggingEventListener listener;

    /**
     * Current working log file.
     */
    private File currentFile;

    /**
     * Maximum number of entries that can go into the log disk file. Must larger than zero.
     */
    private int maximumEntries = VungleLogger.LOGGER_MAX_ENTRIES;

    LogPersister(@Nullable File cacheDir) {
        super(cacheDir, SDK_LOGS_DIR, SDK_LOG_FILE_PATTERN, SDK_LOG_FILE_PENDING);

        if (fileDir != null) {
            currentFile = getOrCreateLastFile();
        }
    }

    void setSdkLoggingEventListener(@NonNull LogManager.SdkLoggingEventListener listener) {
        this.listener = listener;
    }

    void setMaximumEntries(int maximumEntries) {
        this.maximumEntries = maximumEntries;
    }

    void saveLogData(@NonNull String message, @NonNull String logLevel, @NonNull String context,
                     @Nullable String eventID, @Nullable String userAgent, @Nullable String bundleID,
                     @Nullable String customData, @Nullable String exClass, @Nullable String threadId) {


        String timeZone = TimeZone.getDefault().getID();
        String creationDate = getDateText(System.currentTimeMillis());
        LogEntry logEntry = new LogEntry(message, logLevel, context, eventID, userAgent, bundleID,
                timeZone, creationDate, customData, exClass, threadId);

        saveEntryToFile(currentFile, logEntry.toJsonString(), new FileSaveCallback() {
            @Override
            public void onSuccess(File file, int lineCount) {
                if (lineCount >= maximumEntries) {
                    boolean pendingSucceed = renameFile(currentFile, file.getName() + SDK_LOG_FILE_PENDING);
                    if (pendingSucceed) {
                        currentFile = getOrCreateLastFile();
                        if (listener != null) {
                            listener.sendPendingLogs();
                        }
                    }
                }
            }

            @Override
            public void onFailure() {
                Log.e(TAG, "Failed to write sdk logs.");
            }
        });

    }

    /**
     * Log files need to send.
     */
    @Nullable
    File[] getPendingFiles() {
        return getFilesWithSuffix(SDK_LOG_FILE_PENDING);
    }

    @Nullable
    File[] getCrashReportFiles(int crashBatchMax) {
        File[] crashFiles = getFilesWithSuffix(SDK_LOG_FILE_CRASH);

        if (crashFiles == null || crashFiles.length == 0) {
            return null;
        }
        sortFilesByNewest(crashFiles);

        return Arrays.copyOfRange(crashFiles, 0, Math.min(crashFiles.length, crashBatchMax));
    }

    protected boolean saveEntryToFile(File file, String value, @Nullable FileSaveCallback fileSaveCallback) {
        if (file == null || !file.exists()) {
            Log.d(TAG, "current log file maybe deleted, create new one.");
            currentFile = getOrCreateLastFile();
            file = currentFile;
            if (currentFile == null || !currentFile.exists()) {
                Log.w(TAG, "Can't create log file, maybe no space left.");
                return false;
            }
        }
        return appendLineToFile(file, value, fileSaveCallback);
    }

    @Nullable
    File getOrCreateLastFile() {
        if (fileDir == null || !fileDir.exists()) {
            Log.w(TAG, "No log cache dir found.");
            return null;
        }

        // get non pending files
        File[] dataFiles = fileDir.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return !name.endsWith(fileNameSuffix);
            }
        });

        //if there are no files, then return a new file
        if (dataFiles == null || dataFiles.length == 0) {
            return getNewFile(fileNamePrefix + System.currentTimeMillis() + UUID.randomUUID().toString());
        }

        //sort files alphabetically and numerically
        sortFilesByNewest(dataFiles);

        //get newest file
        File loggingFile = dataFiles[0];
        int lineCount = getLineCount(loggingFile);

        //if available lines then return file
        if (lineCount <= 0 || lineCount < maximumEntries) {
            return loggingFile;
        } else {
            File newLoggingFile = null;
            try {
                //rename existing file that was previously fetched
                boolean pendingSucceed = renameFile(loggingFile, loggingFile.getName() + fileNameSuffix);
                if (pendingSucceed) {
                    //create new file
                    newLoggingFile = getOrCreateLastFile();
                }
            } catch (Exception ignored) {
            }

            if (newLoggingFile == null) {
                newLoggingFile = loggingFile;
            }

            return newLoggingFile;
        }
    }

    public void saveCrashLogData(@NonNull String message, @NonNull String logLevel, @NonNull String context,
                                 @Nullable String eventID, @Nullable String userAgent, @Nullable String bundleID,
                                 @Nullable final String customData, @Nullable String exClass, @Nullable String threadId) {
        if (fileDir == null) {
            Log.w(TAG, "No log cache dir found.");
            return;
        }

        String timeZone = TimeZone.getDefault().getID();
        String creationDate = getDateText(System.currentTimeMillis());
        LogEntry logEntry = new LogEntry(message, logLevel, context, eventID, userAgent, bundleID,
                timeZone, creationDate, customData, exClass, threadId);

        String crashFileName = SDK_CRASH_LOG_FILE_PATTERN + System.currentTimeMillis();
        final File crashFile = createFileOrDirectory(fileDir, crashFileName, false);
        if (crashFile != null) {
            appendLineToFile(crashFile, logEntry.toJsonString(), new FileSaveCallback() {
                @Override
                public void onSuccess(File file, int lineCount) {
                    renameFile(crashFile, crashFile.getName() + SDK_LOG_FILE_CRASH);
                }

                @Override
                public void onFailure() {
                    Log.e(TAG, "Failed to write crash log.");
                }
            });
        }
    }
}