package com.vungle.warren.log;

import android.text.TextUtils;
import android.util.Log;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.vungle.warren.VungleApiClient;
import com.vungle.warren.network.Response;
import com.vungle.warren.persistence.FilePreferences;
import com.vungle.warren.utility.FileUtility;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

class LogSender {
    private static final String TAG = LogSender.class.getSimpleName();

    private static final String PREFS_DEVICE_ID_KEY = "device_id";
    private static final String PREFS_BATCH_ID_KEY = "batch_id";

    //payload json schema
    private static final String HEADER_LOG_BATCH_ID = "batch_id";
    private static final String HEADER_LOG_DEVICE_ID = "device_guid";
    private static final String HEADER_LOG_PAYLOAD = "payload";

    private final VungleApiClient vungleApiClient;
    private final FilePreferences prefs;

    private final String deviceId; //unique ID for each device
    private int batchId;

    LogSender(@NonNull VungleApiClient apiClient, @NonNull FilePreferences prefs) {
        this.vungleApiClient = apiClient;
        this.prefs = prefs;
        deviceId = getDeviceId();
        batchId = getBatchId();
    }

    void sendLogs(@NonNull File[] pendingFiles) {
        for (File loggingFile : pendingFiles) {
            JsonObject requestBody = new JsonObject();
            requestBody.addProperty(HEADER_LOG_BATCH_ID, batchId);
            requestBody.addProperty(HEADER_LOG_DEVICE_ID, deviceId);
            try {
                JsonArray payload = getPayload(loggingFile);
                if (payload == null) {
                    FileUtility.delete(loggingFile);
                    continue;
                }
                requestBody.add(HEADER_LOG_PAYLOAD, payload);

                Response<JsonObject> response = vungleApiClient.sendLog(requestBody).execute();
                if (response.isSuccessful()) {
                    FileUtility.delete(loggingFile);
                }
            } catch (IOException e) {
                Log.e(TAG, "Failed to generate request payload.");
            }

            if (batchId >= Integer.MAX_VALUE) {
                batchId = -1;
            }

            // increase batch ID
            batchId++;
        }

        saveBatchId();
    }

    @Nullable
    private JsonArray getPayload(@NonNull File logFile) {
        JsonArray logArray = new JsonArray();
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new FileReader(logFile));
            String logItem;
            while ((logItem = reader.readLine()) != null) {
                JsonObject logEntry = JsonParser.parseString(logItem).getAsJsonObject();
                logArray.add(logEntry);
            }
        } catch (Exception e) {
            Log.e(TAG, "Invalidate log document file.");
            return null;
        } finally {
            FileUtility.closeQuietly(reader);
        }

        return logArray;
    }

    @NonNull
    private String getDeviceId() {
        String uuid = prefs.getString(PREFS_DEVICE_ID_KEY, "");
        if (TextUtils.isEmpty(uuid)) {
            uuid = java.util.UUID.randomUUID().toString();
            prefs.put(PREFS_DEVICE_ID_KEY, uuid);
            prefs.apply();
        }
        return uuid;
    }

    private int getBatchId() {
        return prefs.getInt(PREFS_BATCH_ID_KEY, 0);
    }

    private void saveBatchId() {
        prefs.put(PREFS_BATCH_ID_KEY, batchId);
        prefs.apply();
    }

}
