package com.vungle.warren.model;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.UUID;

/**
 * A Placement is an object that represents a single ad configuration. Publishers use placements to
 * test ads in different locations within their app, or to try different kinds of campaigns for
 * A/B testing. The identifier is generated by the Vungle dashboard, and also contains a collection of
 * assets.
 */
public class AdAsset {

    /**
     * Unique identifier for the Asset.
     */
    public final String identifier;

    /**
     * Unique identifier for the Ad (Foreign key).
     */
    public final String adIdentifier;

    /**
     * Parent id, like ZIP for ZIP_ASSET
     */
    public String parentId;

    /**
     * Server path of asset
     */
    public final String serverPath;

    /**
     * Local path of asset
     */
    public final String localPath;

    /**
     * Download Status
     */
    @IntDef({Status.NEW, Status.DOWNLOAD_RUNNING, Status.DOWNLOAD_FAILED, Status.DOWNLOAD_SUCCESS, Status.PROCESSED})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Status {
        int NEW = 0;
        int DOWNLOAD_RUNNING = 1;
        int DOWNLOAD_FAILED = 2;
        int DOWNLOAD_SUCCESS = 3;
        int PROCESSED = 4;
    }

    @Status
    public int status;

    /**
     * File type
     */
    @IntDef({FileType.ZIP, FileType.ZIP_ASSET, FileType.ASSET})
    @Retention(RetentionPolicy.SOURCE)
    public @interface FileType {
        int ZIP = 0;
        int ZIP_ASSET = 1;
        int ASSET = 2;
    }

    @FileType
    public int fileType;

    /**
     * File size to check the integrity of file
     */
    public long fileSize;

    /**
     * Retry Counter for each asset. //may be required in future
     */
    public int retryCount;

    /**
     * Download Error Type
     */
    @IntDef({ErrorType.CAN_RETRY_ERROR, ErrorType.CANNOT_RETRY_ERROR, ErrorType.NO_ERROR})
    @Retention(RetentionPolicy.SOURCE)
    public @interface ErrorType {
        int NO_ERROR = 0;
        int CAN_RETRY_ERROR = 1;
        int CANNOT_RETRY_ERROR = 2;
    }

    @ErrorType
    int retryTypeError;

    /**
     * Basic constructor, accepts an identifier for the placement.
     *
     * @param adIdentifier The foreign reference id of the placement.
     * @param serverPath   The server url.
     * @param localPath    The local file path.
     */
    public AdAsset(@NonNull String adIdentifier, @Nullable String serverPath, @NonNull String localPath) {
        this(adIdentifier, serverPath, localPath, UUID.randomUUID().toString());
    }

    AdAsset(@NonNull String adIdentifier, @Nullable String serverPath, @NonNull String localPath, String id) {
        this.identifier = id;
        this.adIdentifier = adIdentifier;

        this.serverPath = serverPath;
        this.localPath = localPath;
        this.fileSize = -1;
        this.retryCount = 0;
        this.retryTypeError = ErrorType.NO_ERROR;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        AdAsset adAsset = (AdAsset) o;

        if (status != adAsset.status) return false;
        if (fileType != adAsset.fileType) return false;
        if (fileSize != adAsset.fileSize) return false;
        if (retryCount != adAsset.retryCount) return false;
        if (retryTypeError != adAsset.retryTypeError) return false;
        if (identifier != null ? !identifier.equals(adAsset.identifier) : adAsset.identifier != null) return false;
        if (adIdentifier != null ? !adIdentifier.equals(adAsset.adIdentifier) : adAsset.adIdentifier != null)
            return false;
        if (parentId != null ? !parentId.equals(adAsset.parentId) : adAsset.parentId != null) return false;
        if (serverPath != null ? !serverPath.equals(adAsset.serverPath) : adAsset.serverPath != null) return false;
        return localPath != null ? localPath.equals(adAsset.localPath) : adAsset.localPath == null;
    }

    @Override
    public int hashCode() {
        int result = identifier != null ? identifier.hashCode() : 0;
        result = 31 * result + (adIdentifier != null ? adIdentifier.hashCode() : 0);
        result = 31 * result + (parentId != null ? parentId.hashCode() : 0);
        result = 31 * result + (serverPath != null ? serverPath.hashCode() : 0);
        result = 31 * result + (localPath != null ? localPath.hashCode() : 0);
        result = 31 * result + status;
        result = 31 * result + fileType;
        result = 31 * result + (int) (fileSize ^ (fileSize >>> 32));
        result = 31 * result + retryCount;
        result = 31 * result + retryTypeError;
        return result;
    }

    @Override
    public String toString() {
        return "AdAsset{" +
                "identifier='" + identifier + '\'' +
                ", adIdentifier='" + adIdentifier + '\'' +
                ", serverPath='" + serverPath + '\'' +
                ", localPath='" + localPath + '\'' +
                ", status=" + status +
                ", fileType=" + fileType +
                ", fileSize=" + fileSize +
                ", retryCount=" + retryCount +
                ", retryTypeError=" + retryTypeError +
                '}';
    }
}
