package com.vungle.warren.model;

import android.os.Build;

import androidx.annotation.IntDef;
import androidx.annotation.RequiresApi;
import androidx.annotation.VisibleForTesting;

import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.annotations.SerializedName;
import com.vungle.warren.persistence.Repository;

import java.util.Arrays;
import java.util.Objects;

/**
 * The CacheBust class contains all the data about a request to bust cache. It has reference to
 * campaign/creative(event) that needs to be busted, info about at which time window event was invalid,
 * and when request itself was created on server.
 */
public class CacheBust {

    private static final String TAG = CacheBust.class.getSimpleName();

    /**
     * Enumeration of event types. Basically event is either campaign or creative.
     */
    @IntDef({EVENT_TYPE_CAMPAIGN, EVENT_TYPE_CREATIVE})
    public @interface EventType {

    }

    public static final int EVENT_TYPE_CAMPAIGN = 1;
    public static final int EVENT_TYPE_CREATIVE = 2;

    /* fields from JSON */
    /**
     * Id of event that needs to be busted
     */
    @SerializedName("id")
    String id;

    /**
     * End of time window that defines - if event was delivered(loaded, cached) during that window -
     * then it needs to be busted. UTC
     */
    @SerializedName("timestamp_bust_end")
    long timeWindowEnd;

    /* internal fields */
    /**
     * Type of event - either {@link #EVENT_TYPE_CAMPAIGN} or {@link #EVENT_TYPE_CREATIVE}
     */
    @EventType int idType;
    /**
     * Id(s) of event(s) that was(were) busted. 1-length array in case of {@link #EVENT_TYPE_CREATIVE},
     * any-length array in case of {@link #EVENT_TYPE_CAMPAIGN}
     */
    String[] eventIds;

    /**
     * Timestamp of when bust was executed. UTC
     */
    @SerializedName("timestamp_processed")
    long timestampProcessed;


    /**
     * Method to calculate id based on actually important fields. Any request pointing to same eventId
     * at same time window is considered to be the same or update to one registered earlier as same
     * work needs to be done in order for cache bust to count as successful.
     *
     * This id is unique and used for proper {@link Repository} interactions.
     * */
    @VisibleForTesting
    public String calculateId() {
        return id + ":" + timeWindowEnd;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public long getTimeWindowEnd() {
        return timeWindowEnd;
    }

    public void setTimeWindowEnd(long timeWindowEnd) {
        this.timeWindowEnd = timeWindowEnd;
    }

    public int getIdType() {
        return idType;
    }

    public void setIdType(int idType) {
        this.idType = idType;
    }

    public String[] getEventIds() {
        return eventIds;
    }

    public void setEventIds(String[] eventIds) {
        this.eventIds = eventIds;
    }

    public long getTimestampProcessed() {
        return timestampProcessed;
    }

    public void setTimestampProcessed(long timestampProcessed) {
        this.timestampProcessed = timestampProcessed;
    }

    public static CacheBust fromJson(JsonObject jsonObject) {
        return new GsonBuilder().create().fromJson(jsonObject, CacheBust.class);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        CacheBust cacheBust = (CacheBust) o;
        return idType == cacheBust.idType &&
                timestampProcessed == cacheBust.timestampProcessed &&
                id.equals(cacheBust.id) &&
                timeWindowEnd == cacheBust.timeWindowEnd &&
                Arrays.equals(eventIds, cacheBust.eventIds);
    }

    @RequiresApi(api = Build.VERSION_CODES.KITKAT)
    @Override
    public int hashCode() {
        int result = Objects.hash(id, timeWindowEnd, idType, timestampProcessed);
        result = 31 * result + Arrays.hashCode(eventIds);
        return result;
    }

    @Override
    public String toString() {
        return "CacheBust{" +
                "id='" + id + '\'' +
                ", timeWindowEnd=" + timeWindowEnd +
                ", idType=" + idType +
                ", eventIds=" + Arrays.toString(eventIds) +
                ", timestampProcessed=" + timestampProcessed +
                '}';
    }
}
