package com.vungle.warren.model;

import androidx.annotation.NonNull;

import com.vungle.warren.VungleLogger;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Cookies allow our sdk to store state on disk without requiring a global state manager singleton.
 * They are used to store arbitrary data in key-value form for any part of the SDK that requires
 * persisting information.
 */
public class Cookie {

    /**
     * Special cookie identifier for the cookie that stores publisher appID for Moat
     */
    public static final String APP_ID = "appId";

    /**
     * Special cookie identifier for the cookie that stores GDPR consent state.
     */
    public static final String CONSENT_COOKIE = "consentIsImportantToVungle";

    /**
     * Special cookie identifier for the cookie that stores CCPA state.
     */
    public static final String CCPA_COOKIE = "ccpaIsImportantToVungle";

    public static final String CCPA_CONSENT_STATUS = "ccpa_status";

    public static final String COPPA_COOKIE = "coppa_cookie";

    /**
     * Mirroring values in json payload
     */
    public static final String COPPA_KEY = "coppa";
    public static final String COPPA_STATUS_KEY = "is_coppa";
    public static final String COPPA_DISABLE_AD_ID = "disable_ad_id";

    public static final String CONSENT_STATUS_OPTED_IN = "opted_in";

    public static final String CONSENT_STATUS_OPTED_OUT = "opted_out";

    /**
     * Special cookie identifier for the cookie that stores incentivized back button text by pubs.
     */
    public static final String INCENTIVIZED_TEXT_COOKIE = "incentivizedTextSetByPub";

    public static final String CONFIG_COOKIE = "configSettings";

    /**
     * Cookie to store cache bust last updated timestamp
     */
    public static final String CACHE_BUST_COOKIE = "cacheBustSettings";

    public static final String LAST_CACHE_BUST = "last_cache_bust";

    /**
     * Cookie identifier to store User Agent in cookie.
     */
    public static final String USER_AGENT_ID_COOKIE = "userAgent";

    /**
     * Cookie identifier to store User Agent in cookie.
     */
    public static final String IS_PLAY_SERVICE_AVAILABLE = "isPlaySvcAvailable";

    public static final String CONFIG_EXTENSION = "config_extension";

    /**
     * Persisting the value retrieved from GPS API for faster retrieval
     */
    public static final String APP_SET_ID_COOKIE = "appSetIdCookie";
    public static final String APP_SET_ID = "appSetId";

    /**
     * All of the strings stored within this cookie
     */
    Map<String, String> strings = new ConcurrentHashMap<>();

    /**
     * All of the booleans stored within this cookie
     */
    Map<String, Boolean> booleans = new ConcurrentHashMap<>();

    /**
     * All of the integers stored within this cookie
     */
    Map<String, Integer> integers = new ConcurrentHashMap<>();

    /**
     * All of the longs stored within this cookie
     */
    Map<String, Long> longs = new ConcurrentHashMap<>();

    /**
     * The identifier for this cookie. This is set by the creator of the cookie and is used to retrieve
     * it.
     */
    String identifier;

    public Cookie(String identifier) {
        this.identifier = identifier;
    }


    public <T> void putValue(String key, T value) {
        if (value == null) {
            VungleLogger.critical(
                    true,
                    Cookie.class.getSimpleName(),
                    "putValue",
                    String.format("Value for key \"%s\" should not be null", key));
            return;
        }
        if (value instanceof String) {
            strings.put(key, (String) value);
        } else if (value instanceof Boolean) {
            booleans.put(key, (Boolean) value);
        } else if (value instanceof Integer) {
            integers.put(key, (Integer) value);
        } else if (value instanceof Long) {
            longs.put(key, (Long) value);
        } else {
            VungleLogger.critical(true,
                    Cookie.class.getSimpleName(),
                    "putValue",
                    "Value type is not supported!");
        }
    }

    public Integer getInt(String key) {
        return integers.get(key);
    }

    public String getString(String key) {
        return strings.get(key);
    }

    public Boolean getBoolean(String key) {
        return booleans.get(key) != null && booleans.get(key);
    }

    public Boolean getBooleanOrNull(String key) {
        return booleans.get(key);
    }

    public Long getLong(String key) {
        return longs.get(key) != null ? longs.get(key) : 0L;
    }

    @NonNull
    public String getId() {
        return identifier;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Cookie cookie = (Cookie) o;

        if (strings != null ? !strings.equals(cookie.strings) : cookie.strings != null)
            return false;
        if (booleans != null ? !booleans.equals(cookie.booleans) : cookie.booleans != null)
            return false;
        if (integers != null ? !integers.equals(cookie.integers) : cookie.integers != null)
            return false;
        if (longs != null ? !longs.equals(cookie.longs) : cookie.longs != null) return false;
        return identifier != null ? identifier.equals(cookie.identifier) : cookie.identifier == null;
    }

    @Override
    public int hashCode() {
        int result = strings != null ? strings.hashCode() : 0;
        result = 31 * result + (booleans != null ? booleans.hashCode() : 0);
        result = 31 * result + (integers != null ? integers.hashCode() : 0);
        result = 31 * result + (longs != null ? longs.hashCode() : 0);
        result = 31 * result + (identifier != null ? identifier.hashCode() : 0);
        return result;
    }
}
