package com.vungle.warren.model;

import android.text.TextUtils;

import androidx.annotation.NonNull;

import com.google.gson.JsonObject;
import com.vungle.warren.model.token.Gdpr;
import com.vungle.warren.persistence.DatabaseHelper;
import com.vungle.warren.persistence.Repository;
import com.vungle.warren.utility.TimeoutProvider;

import java.util.concurrent.TimeUnit;

public class GdprCookie {

    public static String CONSENT_STATUS = "consent_status";
    public static String CONSENT_SOURCE = "consent_source";
    public static String NO_INTERACTION = "no_interaction";
    public static String TIMESTAMP = "timestamp";
    public static String CONSENT_MESSAGE_VERSION = "consent_message_version";
    public static String UNKNOWN = "unknown";


    private final Cookie cookie;
    private final Repository repository;

    public GdprCookie(@NonNull Repository repository, TimeoutProvider timeoutProvider) {
        this.repository = repository;
        Cookie cookie = repository
                .load(Cookie.CONSENT_COOKIE, Cookie.class)
                .get(timeoutProvider.getTimeout(), TimeUnit.MILLISECONDS);
        if (cookie == null) {
            cookie = createDefaultCookie();
        }
        this.cookie = cookie;
    }

    /// If we have no consent status, default to unknown values. We use lack of data to infer
    /// this status.
    public String getConsentStatus() {
        return cookie != null ? cookie.getString(CONSENT_STATUS) : Gdpr.UNKNOWN;
    }

    public String getSource() {
        return cookie != null ? cookie.getString(CONSENT_SOURCE) : NO_INTERACTION;
    }

    public Long getTimeStamp() {
        return cookie != null ? cookie.getLong(TIMESTAMP) : 0L;
    }

    // return empty string for non updated consent status/vungle default
    // returning a number will break jaeger
    public String getMessageVersion() {
        return cookie != null ? cookie.getString(CONSENT_MESSAGE_VERSION) : "";
    }

    private Cookie createDefaultCookie() {
        Cookie cookie = new Cookie(Cookie.CONSENT_COOKIE);
        cookie.putValue(CONSENT_MESSAGE_VERSION, "");
        cookie.putValue(CONSENT_STATUS, UNKNOWN);
        cookie.putValue(CONSENT_SOURCE, NO_INTERACTION);
        cookie.putValue(TIMESTAMP, 0L);
        return cookie;
    }

    public void save(JsonObject gdprJsonObject) throws DatabaseHelper.DBException {
        boolean isCountryDataProtected = JsonUtil.hasNonNull(gdprJsonObject, "is_country_data_protected") && gdprJsonObject.get("is_country_data_protected").getAsBoolean();
        String consentTitle = JsonUtil.hasNonNull(gdprJsonObject, "consent_title") ? gdprJsonObject.get("consent_title").getAsString() : "";
        String consentMessage = JsonUtil.hasNonNull(gdprJsonObject, "consent_message") ? gdprJsonObject.get("consent_message").getAsString() : "";
        String consentMessageVersion = JsonUtil.hasNonNull(gdprJsonObject, "consent_message_version") ? gdprJsonObject.get("consent_message_version").getAsString() : "";
        String acceptText = JsonUtil.hasNonNull(gdprJsonObject, "button_accept") ? gdprJsonObject.get("button_accept").getAsString() : "";
        String denyText = JsonUtil.hasNonNull(gdprJsonObject, "button_deny") ? gdprJsonObject.get("button_deny").getAsString() : "";

        cookie.putValue("is_country_data_protected", isCountryDataProtected);
        cookie.putValue("consent_title", TextUtils.isEmpty(consentTitle) ? "Targeted Ads" : consentTitle);
        cookie.putValue("consent_message", TextUtils.isEmpty(consentMessage) ?
                "To receive more relevant ad content based on your interactions with our ads, click \"I Consent\" below. Either way, you will see the same amount of ads." : consentMessage);

        // what ever SDK get messageVersion from server, SDK saves and pass
        String strGDPRCurrentSource = cookie.getString(CONSENT_SOURCE);
        if (!"publisher".equalsIgnoreCase(strGDPRCurrentSource)) {      //Condition for not to override the publisher version with server version
            cookie.putValue(CONSENT_MESSAGE_VERSION, TextUtils.isEmpty(consentMessageVersion) ? "" : consentMessageVersion);
        }
        cookie.putValue("button_accept", TextUtils.isEmpty(acceptText) ? "I Consent" : acceptText);
        cookie.putValue("button_deny", TextUtils.isEmpty(denyText) ? "I Do Not Consent" : denyText);

        repository.save(cookie);
    }
}
