package com.vungle.warren.model;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import android.text.TextUtils;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.annotations.SerializedName;
import com.vungle.warren.AdConfig;
import com.vungle.warren.SessionTracker;
import com.vungle.warren.utility.HashUtility;

import java.util.ArrayList;
import java.util.List;

/**
 * A container for an advertisement report, this will be populated as the advertisement plays and
 * once the advertisement is over, will save itself to a report directory, where it can be picked
 * up by the reporting mechanism to send to the server.
 */
public class Report {

    private static final String DOWNLOAD_ACTION = "download";

    @IntDef({NEW, READY, SENDING, FAILED})
    public @interface Status {}

    public static final int NEW = 0;
    public static final int READY = 1;
    public static final int SENDING = 2;
    public static final int FAILED = 3;

    /**
     * status = NEW for newly created reports.
     * status = READY for Ad done playing and Ready to be Send.
     * status = SENDING, is being send.
     * status = FAILED if received network error.
     *
     * Default = NEW
     */
    int status = NEW;

    /**
     * Unique identifier for the placement where the advertisement was played.
     */
    String placementId;

    /**
     * Ad Token of the advertisement that was played. The server will use this identifier to link
     * attribution back to us.
     */
    String adToken;

    /**
     * The identifier of the publisher application the ad was played in.
     */
    String appId;

    /**
     * Whether or not the advertisement was incentivized.
     */
    boolean incentivized;

    /**
     * Whether or not the placement was support header bidding.
     */
    boolean headerBidding;

    /**
     * Whether or not the ad is using remote url.
     */
    boolean playRemoteUrl;

    /**
     * The time at which the ad started playing, in seconds since epoch.
     */
    long adStartTime;

    /**
     * The url where the advertisement was played from.
     */
    String url;

    /**
     * The length of the video, in milliseconds. The advertisement metadata does not always
     * contain this information, sometimes we have to add it to the report based on the media file
     * information.
     */
    long videoLength;
    /**
     * Duration of complete ad experience in milliseconds.
     */
    long adDuration;

    /**
     * Legacy field for time to download the ad, in milliseconds.
     */
    long ttDownload;

    /**
     * The creative campaign where the advertisement was sourced from.
     */
    String campaign;

    /**
     * The identifier of the advertisement which this report is about.
     */
    String advertisementID;


    /**
     * The amount of the advertisement that was viewed, in milliseconds. This will always be less than
     * or equal to {@link #videoLength}.
     */
    int videoViewed;

    /**
     * The dynamically growing list of user actions, a new action is added to this collection when
     * the user interacts with the advertisement or continues watching it.
     */
    final List<UserAction> userActions = new ArrayList<>();

    /**
     *
     */
    final List<String> clickedThrough = new ArrayList<>();


    /**
     * The errors that occurred while playing the ad, if any.
     */
    final List<String> errors = new ArrayList<>();

    /**
     * The vungle ad type for the reported ad. e.g. "vungle_local" or "vungle_mraid"
     */
    String adType;

    /**
     * The identifier for the dynamic template used, if any.
     */
    String templateId;

    /**
     * There are some cases where we need to include a user identifier with our report, which allows
     * publishers to verify their own in-app reward systems with the user identifiers of those players
     * who watched the advertisement.
     */
    String userID;

    /**
     * Tracks the number of advertisements that have been played during this application's lifecycle.
     * It is stored along with the placement and extracted and deleted when creating the advertisement
     * report. This variable is set by the publisher because it is possible for them to have multiple
     * advertisement providers.
     * <p>
     * It has a sentinel value of -1, indicating it should not be included in the report.
     */
    int ordinal;

    String adSize;

    volatile boolean wasCTAClicked;

    /**
     * Tracks when the SDK was successfully initialized when this report ad was created
     */
    @VisibleForTesting
    public long initTimeStamp;

    /**
     * Records when assets began downloading
     */
    @VisibleForTesting
    public long assetDownloadDuration;

    Report() {

    }

    /**
     * Construct a new report for an ad that is being played for the given {@link Placement}, and
     * playing the given {@link Advertisement}. This should be created when the advertisement starts
     * playing and actions can be added to it by the presenter.
     *
     * @param advertisement The Advertisement unit that is being played.
     * @param placement     The placement where the advertisement is being played.
     * @param startTime     The time (in seconds since epoch) when the advertisement started playing.
     */
    public Report(@NonNull Advertisement advertisement, @NonNull Placement placement, long startTime) {
        this(advertisement, placement, startTime, null);
    }

    /**
     * Construct a new report for an ad that is being played for the given {@link Placement}, and
     * playing the given {@link Advertisement}. This should be created when the advertisement starts
     * playing and actions can be added to it by the presenter.
     *
     * @param advertisement The Advertisement unit that is being played.
     * @param placement     The placement where the advertisement is being played.
     * @param startTime     The time (in seconds since epoch) when the advertisement started playing.
     * @param userID        The identifier of the user watching the advertisement, provided by the publisher.
     */
    public Report(@NonNull Advertisement advertisement, @NonNull Placement placement, long startTime, @Nullable String userID) {
        placementId = placement.getId();
        adToken = advertisement.getAdToken();
        advertisementID = advertisement.getId();
        appId = advertisement.getAppID();
        incentivized = placement.isIncentivized();
        headerBidding = placement.isHeaderBidding();
        adStartTime = startTime;
        url = advertisement.getUrl();
        ttDownload = -1; /// TODO: store the time to download in the advertisement class?
        campaign = advertisement.getCampaign();
        initTimeStamp = SessionTracker.getInstance().getInitTimestamp();
        assetDownloadDuration = advertisement.getAssetDownloadDuration();
        switch (advertisement.getAdType()) {
            case Advertisement.TYPE_VUNGLE_LOCAL:
                adType = "vungle_local";
                break;
            case Advertisement.TYPE_VUNGLE_MRAID:
                adType = "vungle_mraid";
                break;
            default:
                throw new IllegalArgumentException("Unknown ad type, cannot process!");
        }
        templateId = advertisement.getTemplateId();
        if (userID == null) {
            this.userID = "";
        } else {
            this.userID = userID;
        }
        ordinal = advertisement.getAdConfig().getOrdinal();

        AdConfig.AdSize tempAdSize = advertisement.getAdConfig().getAdSize();
        if (AdConfig.AdSize.isNonMrecBannerAdSize(tempAdSize)) {
            adSize = tempAdSize.getName();
        }
    }

    /**
     * Record the given action in the current report.
     *
     * @param action    The action taken
     * @param value     The value, if any, of the action
     * @param timestamp The time at which the action occurred, in milliseconds since epoch.
     */
    public synchronized void recordAction(String action, String value, long timestamp) {
        userActions.add(new UserAction(action, value, timestamp));
        clickedThrough.add(action);
        if (action.equals(DOWNLOAD_ACTION)) {
            wasCTAClicked = true;
        }
    }

    /**
     * Record the error that occurred while rendering the advertisement.
     *
     * @param description The description of the error
     */
    public synchronized void recordError(String description) {
        errors.add(description);
    }


    /**
     * Records the updated progress of the advertisement. New values override the older values.
     *
     * @param viewed The amount of the advertisement that has been viewed, in milliseconds.
     */
    public void recordProgress(int viewed) {
        videoViewed = viewed;
    }

    /**
     * Provides the report with the duration of the advertisement.
     *
     * @param duration The duration of the advertisement, in milliseconds.
     */
    public void setVideoLength(long duration) {
        videoLength = duration;
    }

    /**
     * Provide the report with complete duration of ad experience. Video viewed + postroll.
     *
     * @param duration in milliseconds
     */
    public void setAdDuration(long duration) {
        adDuration = duration;
    }

    public long getAdDuration() {
        return adDuration;
    }

    public void setTtDownload(long duration) {
        ttDownload = duration;
    }

    public void setAllAssetDownloaded(boolean allAssetDownloaded) {
        this.playRemoteUrl = !allAssetDownloaded;
    }

    /**
     * Accessor for the placement identifier this report is for.
     *
     * @return The placement identifier
     */
    public String getPlacementId() {
        return placementId;
    }

    /**
     * Accessor for the advertisement identifier this report is about.
     *
     * @return The advertisement identifier.
     */
    public String getAdvertisementID() {
        return advertisementID;
    }

    /**
     * If CTA action was ever performed on the Ad, it is used to return on
     * {@link com.vungle.warren.PlayAdCallback#onAdEnd(String, boolean, boolean)}
     *
     * @return {@code true} if CTA action was ever performed on the Ad
     */
    public boolean isCTAClicked() {
        return wasCTAClicked;
    }

    /**
     * Get userId for reporting Incentivized ads
     *
     * @return userId as {@link String}
     */
    public String getUserID() {
        return userID;
    }

    /**
     * Generates the report body that needs to be sent to the server in order to count the ad as
     * viewed and to generate revenue. It has the following basic structure:
     * <code><pre>
     * "request": {
     * "placement_reference_id": "string",
     * "ad_token": "string",
     * "app_id": "string",
     * "incentivized": 0,
     * "header_bidding": true,
     * "adStartTime": 0,
     * "plays": [
     * {
     * "startTime": 0,
     * "userActions": [
     * {
     * "action": "string",
     * "value": "string",
     * "timestamp_millis": 0
     * }
     * ]
     * }
     * ],
     * "clickedThrough": [
     * "string"
     * ],
     * "url": "string",
     * "adDuration": 0,
     * "ttDownload": 0,
     * "campaign": "string",
     * "adType": "string",
     * "errors": [
     * "string"
     * ],
     * "templateId": "string"
     * },
     * </pre></code>
     *
     * @return {@link JsonObject} which represents report body needed to be sent to server in order
     * to count the ad as viewed and generate revenue
     */
    public synchronized JsonObject toReportBody() {
        JsonObject request = new JsonObject();

        request.addProperty("placement_reference_id", placementId);
        request.addProperty("ad_token", adToken);
        request.addProperty("app_id", appId);
        request.addProperty("incentivized", incentivized ? 1 : 0);
        request.addProperty("header_bidding", headerBidding);
        request.addProperty("play_remote_assets", playRemoteUrl);
        request.addProperty("adStartTime", adStartTime);

        //on v5 server it is better not to send url if url field does not exist
        //since it saves some processing time and memory since server creates pointers
        //on empty string
        if (!TextUtils.isEmpty(url)) {
            request.addProperty("url", url);
        }

        request.addProperty("adDuration", adDuration);
        request.addProperty("ttDownload", ttDownload);
        request.addProperty("campaign", campaign);
        request.addProperty("adType", adType);
        request.addProperty("templateId", templateId);

        request.addProperty("init_timestamp", initTimeStamp);
        request.addProperty("asset_download_duration", assetDownloadDuration);

        if (!TextUtils.isEmpty(adSize)) {
            request.addProperty("ad_size", adSize);
        }

        /// Plays
        JsonArray plays = new JsonArray();
        JsonObject playsObject = new JsonObject();
        playsObject.addProperty("startTime", adStartTime);
        if (videoViewed > 0) {
            playsObject.addProperty("videoViewed", videoViewed);
        }
        if (videoLength > 0) {
            playsObject.addProperty("videoLength", videoLength);
        }
        JsonArray userActionJson = new JsonArray();
        for (UserAction action : userActions) {
            userActionJson.add(action.toJson());
        }
        playsObject.add("userActions", userActionJson);
        plays.add(playsObject);
        request.add("plays", plays);

        /// Errors
        JsonArray errorsJson = new JsonArray();
        for (String s : errors) {
            errorsJson.add(s);
        }
        request.add("errors", errorsJson);

        /// Clicked Through
        JsonArray clicked = new JsonArray();
        for (String s : clickedThrough) {
            clicked.add(s);
        }
        request.add("clickedThrough", clicked);

        /// Incentivized User ID
        if (incentivized && !TextUtils.isEmpty(userID)) {
            request.addProperty("user", userID);
        }

        /// Ordinal data. Only include this if value is >0.
        if (ordinal > 0) {
            request.addProperty("ordinal_view", ordinal);
        }

        return request;
    }


    @NonNull
    public String getId() {
        return placementId + "_" + adStartTime;
    }

    public long getAdStartTime() {
        return adStartTime;
    }

    public static class UserAction {

        @SerializedName("action")
        private String action;
        @SerializedName("value")
        private String value;
        @SerializedName("timestamp")
        private long timestamp;

        public UserAction(String action, String value, long timestamp) {
            this.action = action;
            this.value = value;
            this.timestamp = timestamp;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) return true;
            if (obj == null || getClass() != obj.getClass()) return false;

            UserAction wrap = (UserAction) obj;

            if (!wrap.action.equals(action)) return false;
            if (!wrap.value.equals(value)) return false;
            if (wrap.timestamp != timestamp) return false;

            return true;
        }

        @Override
        public int hashCode() {
            int result = action.hashCode();
            result = 31 * result + value.hashCode();
            result = 31 * result + (int) (timestamp ^ (timestamp >>> 32));
            return result;
        }

        public JsonObject toJson() {
            JsonObject ret = new JsonObject();
            ret.addProperty("action", action);

            //allow us to send json objects with empty values
            if (value != null && !value.isEmpty()) {
                ret.addProperty("value", value);
            }

            ret.addProperty("timestamp_millis", timestamp);

            return ret;
        }
    }

    @Override
    public synchronized boolean equals(Object obj) {
        if (this == obj) return true;
        if (obj == null || getClass() != obj.getClass()) return false;

        Report wrap = (Report) obj;

        if (!wrap.placementId.equals(placementId)) return false;
        if (!wrap.adToken.equals(adToken)) return false;
        if (!wrap.appId.equals(appId)) return false;
        if (wrap.incentivized != incentivized) return false;
        if (wrap.headerBidding != headerBidding) return false;
        if (wrap.adStartTime != adStartTime) return false;
        if (!wrap.url.equals(url)) return false;
        if (wrap.videoLength != videoLength) return false;
        if (wrap.adDuration != adDuration) return false;
        if (wrap.ttDownload != ttDownload) return false;
        if (!wrap.campaign.equals(campaign)) return false;
        if (!wrap.adType.equals(adType)) return false;
        if (!wrap.templateId.equals(templateId)) return false;
        if (wrap.wasCTAClicked != wasCTAClicked) return false;
        if (!wrap.userID.equals(userID)) return false;
        if (wrap.initTimeStamp != initTimeStamp) return false;
        if (wrap.assetDownloadDuration != assetDownloadDuration) return false;

        if (wrap.clickedThrough.size() != clickedThrough.size()) return false;
        for (int x = 0; x < clickedThrough.size(); x++) {
            if (!wrap.clickedThrough.get(x).equals(clickedThrough.get(x))) {
                return false;
            }
        }

        if (wrap.errors.size() != errors.size()) return false;
        for (int x = 0; x < errors.size(); x++) {
            if (!wrap.errors.get(x).equals(errors.get(x))) {
                return false;
            }
        }

        if (wrap.userActions.size() != userActions.size()) return false;
        for (int x = 0; x < userActions.size(); x++) {
            if (!wrap.userActions.get(x).equals(userActions.get(x))) {
                return false;
            }
        }

        return true;
    }

    @Override
    public synchronized int hashCode() {
        int result = HashUtility.getHashCode(placementId);
        result = 31 * result + HashUtility.getHashCode(adToken);
        result = 31 * result + HashUtility.getHashCode(appId);
        result = 31 * result + (incentivized ? 1 : 0);
        result = 31 * result + (headerBidding ? 1 : 0);
        result = 31 * result + (int) (adStartTime ^ (adStartTime >>> 32));
        result = 31 * result + HashUtility.getHashCode(url);
        result = 31 * result + (int) (videoLength ^ (videoLength >>> 32));
        result = 31 * result + (int) (adDuration ^ (adDuration >>> 32));
        result = 31 * result + (int) (ttDownload ^ (ttDownload >>> 32));
        result = 31 * result + (int) (initTimeStamp ^ (initTimeStamp >>> 32));
        result = 31 * result + (int) (assetDownloadDuration ^ (assetDownloadDuration >>> 32));
        result = 31 * result + HashUtility.getHashCode(campaign);
        result = 31 * result + HashUtility.getHashCode(userActions);
        result = 31 * result + HashUtility.getHashCode(clickedThrough);
        result = 31 * result + HashUtility.getHashCode(errors);
        result = 31 * result + HashUtility.getHashCode(adType);
        result = 31 * result + HashUtility.getHashCode(templateId);
        result = 31 * result + HashUtility.getHashCode(userID);
        result = 31 * result + (wasCTAClicked ? 1 : 0);
        return result;
    }

    public @Status int getStatus() {
        return status;
    }

    public void setStatus(@Status int status) {
        this.status = status;
    }
}
