package com.vungle.warren.model;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.vungle.warren.session.SessionAttribute;
import com.vungle.warren.session.SessionEvent;
import com.vungle.warren.utility.HashUtility;

/**
 * A session data object cannot be modified once its created
 */
public class SessionData {

    /**
     * Name of event
     */
    public SessionEvent sessionEvent;

    private static final Gson GSON = new Gson();

    private int sendAttempts;

    /**
     * Store of events
     */
    private JsonObject sessionEventJsonObject;

    protected SessionData(String jsonString, int sendAttempts) {
        sessionEventJsonObject = GSON.fromJson(jsonString, JsonObject.class);
        this.sendAttempts = sendAttempts;
    }

    private SessionData(SessionEvent sessionEvent, JsonObject attributes) {
        this.sessionEvent = sessionEvent;
        this.sessionEventJsonObject = attributes;
        attributes.addProperty(SessionAttribute.TIMESTAMP.toString(),
                System.currentTimeMillis());
    }

    @Override
    public boolean equals(@Nullable Object obj) {
        if (obj == null || !(obj instanceof SessionData)) {
            return false;
        }
        SessionData sessionData = (SessionData) obj;
        if (!this.sessionEvent.equals(sessionData.sessionEvent)) {
            return false;
        }

        if (!this.sessionEventJsonObject.equals(sessionData.sessionEventJsonObject)) {
            return false;
        }

        return true;
    }

    public void removeEvent(SessionAttribute attribute) {
        sessionEventJsonObject.remove(attribute.toString());
    }

    public void addAttribute(SessionAttribute attribute, String data) {
        sessionEventJsonObject.addProperty(attribute.toString(), data);
    }

    public static class Builder {

        JsonObject jsonObject = new JsonObject();
        SessionEvent event;

        public SessionData build() {
            if (event == null) {
                throw new IllegalArgumentException("SessionData must have event");
            }
            return new SessionData(event, jsonObject);
        }

        public Builder setEvent(SessionEvent event) {
            this.event = event;
            jsonObject.addProperty("event", event.toString());
            return this;
        }

        public Builder addData(SessionAttribute attribute, String value) {
            jsonObject.addProperty(attribute.toString(), value);
            return this;
        }

        public Builder addData(SessionAttribute attribute, int value) {
            jsonObject.addProperty(attribute.toString(), value);
            return this;
        }

        public Builder addData(SessionAttribute attribute, double value) {
            jsonObject.addProperty(attribute.toString(), value);
            return this;
        }

        public Builder addData(SessionAttribute attribute, boolean value) {
            jsonObject.addProperty(attribute.toString(), value);
            return this;
        }
    }

    public String getAsJsonString() {
        String gsonString = GSON.toJson(sessionEventJsonObject);
        return gsonString;
    }

    public String getStringAttribute(SessionAttribute attribute) {
        JsonElement element = sessionEventJsonObject.get(attribute.toString());
        if (element != null) {
            return element.getAsString();
        }
        return null;
    }

    @NonNull
    public String getId() {
        String id = HashUtility.sha256(getAsJsonString());

        //if sha fails
        if (id == null) {
            id = String.valueOf(getAsJsonString().hashCode());
        }

        return id;
    }

    public int getSendAttempts() {
        return sendAttempts;
    }

    public int incrementSendAttempt() {
        return sendAttempts++;
    }
}