package com.vungle.warren.network;

import android.text.TextUtils;

import com.google.gson.JsonObject;
import com.vungle.warren.network.converters.Converter;
import com.vungle.warren.network.converters.EmptyResponseConverter;
import com.vungle.warren.network.converters.JsonConverter;

import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import okhttp3.HttpUrl;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;

public class VungleApiImpl implements VungleApi {

    private static final String CONFIG = "config";
    private static final Converter<ResponseBody, JsonObject> jsonConverter =
            new JsonConverter();
    private static final Converter<ResponseBody, Void> emptyResponseConverter =
            new EmptyResponseConverter();

    @VisibleForTesting
    HttpUrl baseUrl;

    @VisibleForTesting
    okhttp3.Call.Factory okHttpClient;

    @VisibleForTesting
    String appId;

    public VungleApiImpl(@NonNull HttpUrl baseUrl, @NonNull okhttp3.Call.Factory okHttpClient) {
        this.baseUrl = baseUrl;
        this.okHttpClient = okHttpClient;
    }

    public void setAppId(String appId) {
        this.appId = appId;
    }

    @Override
    public Call<JsonObject> config(String ua, JsonObject body) {
        return createNewPostCall(ua, baseUrl.toString() + CONFIG, body);
    }

    @Override
    public Call<JsonObject> reportNew(String ua, String path, Map<String, String> query) {
        return createNewGetCall(ua, path, query, jsonConverter);
    }

    @Override
    public Call<JsonObject> ads(String ua, String path, JsonObject body) {
        return createNewPostCall(ua, path, body);
    }

    @Override
    public Call<JsonObject> willPlayAd(String ua, String path, JsonObject body) {
        return createNewPostCall(ua, path, body);
    }

    @Override
    public Call<JsonObject> ri(String ua, String path, JsonObject body) {
        return createNewPostCall(ua, path, body);
    }

    @Override
    public Call<JsonObject> reportAd(String ua, String path, JsonObject body) {
        return createNewPostCall(ua, path, body);
    }

    @Override
    public Call<JsonObject> sendLog(String ua, String path, JsonObject body) {
        return createNewPostCall(ua, path, body);
    }

    @Override
    public Call<Void> pingTPAT(String ua, String url) {
        return createNewGetCall(ua, url, null, emptyResponseConverter);
    }

    @Override
    public Call<JsonObject> cacheBust(String ua, String path, JsonObject body) {
        return createNewPostCall(ua, path, body);
    }

    @Override
    public Call<JsonObject> sendBiAnalytics(String ua, String path, JsonObject body) {
        return createNewPostCall(ua, path, body);
    }

    private Call<JsonObject> createNewPostCall(String ua, @NonNull String path, JsonObject requestObj) {
        String requestBody = "";
        if (requestObj != null) {
            requestBody = requestObj.toString();
        }

        okhttp3.Request request = defaultBuilder(ua, path)
                .post(RequestBody.create(null, requestBody))
                .build();

        return new OkHttpCall<>(okHttpClient.newCall(request), jsonConverter);
    }

    private <T> Call<T> createNewGetCall(String ua, @NonNull String path, @Nullable Map<String, String> query, Converter<ResponseBody, T> converter) {
        HttpUrl.Builder urlBuilder = HttpUrl.get(path).newBuilder();
        if (query != null) {
            for (Map.Entry<String, String> q : query.entrySet()) {
                urlBuilder.addQueryParameter(q.getKey(), q.getValue());
            }
        }
        okhttp3.Request request = defaultBuilder(ua, urlBuilder.build().toString())
                .get()
                .build();
        return new OkHttpCall<>(okHttpClient.newCall(request), converter);
    }

    @NonNull
    private Request.Builder defaultBuilder(@NonNull String ua, @NonNull String path) {
        Request.Builder builder = new Request.Builder()
                .url(path)
                .addHeader("User-Agent", ua)
                .addHeader("Vungle-Version", "5.10.0")
                .addHeader("Content-Type", "application/json");
        if (!TextUtils.isEmpty(appId)) {
            builder.addHeader("X-Vungle-App-Id", appId);
        }
        return builder;
    }

}
