package com.vungle.warren.persistence;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Environment;
import android.os.FileObserver;
import android.os.StatFs;
import android.util.Log;

import com.vungle.warren.VungleLogger;
import com.vungle.warren.utility.CollectionsConcurrencyUtil;
import com.vungle.warren.utility.FileUtility;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Selects between {@link Context#getExternalFilesDir(String)} and {@link Context#getFilesDir()} with priority to the first.
 * On every {@link CacheManager#getCache()} checks the existence of current cache directory,
 * observers external files dir and calls {@link Listener#onCacheChanged()} on any change.
 */
public class CacheManager {

    private static final String TAG = CacheManager.class.getSimpleName();

    public interface Listener {
        void onCacheChanged();
    }

    private static final String PATH_ID = "cache_path";
    private static final String PATH_IDS = "cache_paths";
    private static final String VUNGLE_DIR = "vungle_cache";
    private final Context context;
    private final FilePreferences prefs;
    private final Set<Listener> listeners = new HashSet<>();
    private File current;
    private final List<File> old = new ArrayList<>();
    private boolean changed;
    private final List<FileObserver> observers = new ArrayList<>();
    static final long UNKNOWN_SIZE = -1L;

    public CacheManager(@NonNull Context context, @NonNull FilePreferences prefs) {
        this.context = context;
        this.prefs = prefs;
        this.prefs.addSharedPrefsKey(PATH_ID, PATH_IDS).apply();
    }

    @SuppressLint("NewApi")
    private synchronized void selectFileDest() {
        if (current == null) {
            String path = prefs.getString(PATH_ID, null);
            current = (path != null ? new File(path) : null);
        }

        final File external = context.getExternalFilesDir(null);
        final File internal = context.getFilesDir();
        boolean canUseExternal =
                Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)
                        && external != null;
        ArrayList<File> candidates = new ArrayList<>();

        if (canUseExternal) {
            File parent = external.getParentFile();
            if (parent != null) {
                candidates.add(new File(parent, "no_backup"));
            }
        }
        candidates.add(context.getNoBackupFilesDir());

        //failsafe - these will be backed up, should switch back to no_backup ASAP
        if (canUseExternal) {
            candidates.add(external);
        }
        candidates.add(internal);

        File result = null;
        boolean success, created = false;
        for (File parent : candidates) {
            File dir = new File(parent, VUNGLE_DIR);

            //If cache dir is not dir delete it and create new dir
            deleteIfFile(dir);

            if (dir.exists()) {
                success = dir.isDirectory() && dir.canWrite();
            } else {
                created = success = dir.mkdirs();
            }

            if (success) {
                result = dir;
                break;
            }
        }

        final File obsoleted = context.getCacheDir();
        HashSet<String> known = prefs.getStringSet(PATH_IDS, new HashSet<String>());

        if (result != null) {
            CollectionsConcurrencyUtil.addToSet(known,result.getPath());
        }
        CollectionsConcurrencyUtil.addToSet(known,obsoleted.getPath());
        prefs.put(PATH_IDS, known).apply();

        old.clear();
        for (String path : known) {
            if (result == null || !result.getPath().equals(path)) {
                old.add(new File(path));
            }
        }

        if (created || (result != null && !result.equals(current)) || (current != null && !current.equals(result))) {
            current = result;
            if (current != null) {
                prefs.put(PATH_ID, current.getPath()).apply();
            }

            for (Listener l : listeners) {
                l.onCacheChanged();
            }
            changed = true;

            //v5 folders are removed in VungleDatabaseCreator#dropOldFilesData
            for (File dir : old) {
                if (!dir.equals(obsoleted)) {
                    try {
                        FileUtility.delete(dir);
                    } catch (IOException e) {
                        VungleLogger.error(true, TAG, "CacheManager", "Can't remove old cache:" + dir.getPath());
                    }
                }
            }
        }

        observeDirectory(external);
    }

    private void check() {
        if (current == null || !current.exists() || !current.isDirectory() || !current.canWrite()) {
            selectFileDest();
        }
    }

    private synchronized void observeDirectory(File root) {
        if (root == null) //keep observing removed sdcard to handle inserting
            return;
        observers.clear();
        observers.add(new FileObserver(root.getPath(), FileObserver.DELETE_SELF) {
            @Override
            public void onEvent(int event, @Nullable String path) {
                stopWatching();
                selectFileDest();
            }
        });
        while (root.getParent() != null) {
            final String dirName = root.getName();
            observers.add(new FileObserver(root.getParent(), FileObserver.CREATE) {
                @Override
                public void onEvent(int event, @Nullable String path) {
                    stopWatching();
                    if (dirName.equals(path)) {
                        //force switching back to external
                        selectFileDest();
                    }
                }
            });
            root = root.getParentFile();
        }
        for (FileObserver observer : observers) {
            try {
                observer.startWatching();
            } catch (Exception e) {
                VungleLogger.warn(true, TAG, "ExceptionContext", Log.getStackTraceString(e));
            }
        }
    }

    @Nullable
    public synchronized File getCache() {
        check();
        return current;
    }

    public synchronized List<File> getOldCaches() {
        check();
        return old;
    }

    public synchronized void addListener(Listener listener) {
        check();
        listeners.add(listener);
        if (changed) {
            listener.onCacheChanged();
        }
    }

    public synchronized void removeListener(Listener listener) {
        listeners.remove(listener);
    }

    public long getBytesAvailable() {
        //try internal storage on fail which expected to be accessible
        return getBytesAvailable(1);
    }

    @SuppressLint("NewApi")
    private long getBytesAvailable(int retry) {
        File dir = getCache();
        long bytesAvailable = UNKNOWN_SIZE;
        if (dir == null) {
            return bytesAvailable;
        }

        StatFs stats = null;
        try {
            stats = new StatFs(dir.getPath());
        } catch (IllegalArgumentException e) {
            Log.w(TAG, "Failed to get available bytes", e);
            if (retry > 0) {
                return getBytesAvailable(retry - 1);
            }
        }

        if (stats != null) {
            bytesAvailable = stats.getBlockSizeLong() * stats.getAvailableBlocksLong();
        }
        return bytesAvailable;
    }

    private static void deleteIfFile(File dir) {
        if (dir.exists() && dir.isFile()) {
            FileUtility.deleteAndLogIfFailed(dir);
        }
    }
}
