package com.vungle.warren.persistence;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.content.ContextWrapper;
import android.database.Cursor;
import android.database.DatabaseErrorHandler;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import com.vungle.warren.VungleLogger;
import com.vungle.warren.utility.FileUtility;

import java.io.File;
import java.io.IOException;

import androidx.annotation.NonNull;

public class DatabaseHelper extends SQLiteOpenHelper {

    public static class DBException extends Exception {
        public DBException(String message) {
            super(message);
        }
    }

    public static final String TAG = DatabaseHelper.class.getSimpleName();
    public static final String DB_NAME = "vungle_db";
    private final DatabaseFactory databaseFactory;

    public DatabaseHelper(@NonNull Context context, int version, @NonNull DatabaseFactory databaseFactory) {
        // Make sure we use app context to avoid db not closed warnings. When app is terminated system should close connections
        super(new NoBackupDatabaseWrapperContext(context.getApplicationContext()), DB_NAME, null, version);
        this.databaseFactory = databaseFactory;
    }

    @Override
    public synchronized void onCreate(SQLiteDatabase db) {
        databaseFactory.create(db);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        databaseFactory.onUpgrade(db, oldVersion, newVersion);
        //todo
    }

    @Override
    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        databaseFactory.onDowngrade(db, oldVersion, newVersion);
    }

    public long insertWithConflict(String table, ContentValues values, int strategy) throws DBException {
        try {
            //insertWithOnConflict?
            return loadWritableDB().insertWithOnConflict(table, null, values, strategy);
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public long update(Query query, ContentValues values) throws DBException {
        try {
            return loadWritableDB().update(query.tableName, values, query.selection, query.args);
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public Cursor query(Query query) {
        return loadWritableDB().query(
                query.tableName,
                query.columns,
                query.selection,
                query.args,
                query.groupBy,
                query.having,
                query.orderBy,
                query.limit
        );
    }

    public Cursor queryRaw(String sql, String[] selectionArgs) {
        return loadWritableDB().rawQuery(sql, selectionArgs);
    }


    public void delete(Query query) throws DBException {
        try {
            loadWritableDB().delete(query.tableName, query.selection, query.args);
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public void execSQL(String s) throws DBException {
        try {
            loadWritableDB().execSQL(s);
        } catch (SQLException e) {
            throw new DBException(e.getMessage());
        }
    }

    public synchronized void dropDb() {
        databaseFactory.deleteData(loadWritableDB());
        close();
        onCreate(loadWritableDB());
    }

    public void init() {
        loadWritableDB();
    }

    private synchronized SQLiteDatabase loadWritableDB() {
        return getWritableDatabase();
    }

    public interface DatabaseFactory {

        void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion);

        void create(SQLiteDatabase database);

        void deleteData(SQLiteDatabase database);

        void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion);
    }

    private static class NoBackupDatabaseWrapperContext extends ContextWrapper {
        public NoBackupDatabaseWrapperContext(Context context) {
            super(context);
        }

        @SuppressLint("NewApi")
        @Override
        public File getDatabasePath(String name) {
            File oldDb = super.getDatabasePath(name);
            File newDb = new File(getApplicationContext().getNoBackupFilesDir(), name);
            try {
                FileUtility.delete(new File(oldDb.getPath()));
                FileUtility.delete(new File(oldDb.getPath() + "-journal"));
            } catch (IOException e) {
                VungleLogger.error(true, TAG, "DatabaseHelper", "Failed to delete old db/-journal");
            }
            return newDb;
        }

        @SuppressLint("NewApi")
        private int getFlags(int mode) {
            int flags = SQLiteDatabase.CREATE_IF_NECESSARY;
            flags |= ((mode & MODE_ENABLE_WRITE_AHEAD_LOGGING) != 0) ? SQLiteDatabase.ENABLE_WRITE_AHEAD_LOGGING : 0;//from API 16
            flags |= ((mode & MODE_NO_LOCALIZED_COLLATORS) != 0) ? SQLiteDatabase.NO_LOCALIZED_COLLATORS : 0;
            return flags;
        }

        @Override
        public SQLiteDatabase openOrCreateDatabase(String name, int mode, SQLiteDatabase.CursorFactory factory) {
            return SQLiteDatabase.openDatabase(getDatabasePath(name).getPath(), factory, getFlags(mode));
        }

        @Override
        public SQLiteDatabase openOrCreateDatabase(String name, int mode, SQLiteDatabase.CursorFactory factory, DatabaseErrorHandler errorHandler) {
            return SQLiteDatabase.openDatabase(getDatabasePath(name).getPath(), factory, getFlags(mode), errorHandler);
        }
    }
}
