package com.vungle.warren.persistence;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import com.vungle.warren.VungleLogger;
import com.vungle.warren.utility.CollectionsConcurrencyUtil;
import com.vungle.warren.utility.FileUtility;

import java.io.File;
import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;

public class FilePreferences {
    @VisibleForTesting
    static final String FILENAME = "vungle_settings";
    @VisibleForTesting
    static final String OLD_SHARED_PREFS = "com.vungle.sdk";
    private final File file;
    private final Executor ioExecutor;
    private final ConcurrentHashMap<String, Object> values = new ConcurrentHashMap<>();
    private final SharedPreferences preferences;
    private final HashSet<String> sharedPrefValues = new HashSet<>();

    @SuppressLint("NewApi")
    public FilePreferences(Context context, Executor ioExecutor) {
        this.ioExecutor = ioExecutor;
        File dir = context.getNoBackupFilesDir();
        file = new File(dir, FILENAME);
        File old = new File(context.getFilesDir(), FILENAME);
        if (old.exists() && !old.renameTo(file)) {
            VungleLogger.error("FilePreferences", "Can't move old FilePreferences");
        }
        Object saved = FileUtility.readSerializable(file);
        if (saved instanceof HashMap) {
            values.putAll(((HashMap) saved));
        }

        preferences = context.getSharedPreferences(OLD_SHARED_PREFS, Context.MODE_PRIVATE);
        migrateFromSharedPrefs();
    }

    public void apply() {
        final Serializable serializable = new HashMap<>(values);
        ioExecutor.execute(new Runnable() {
            @Override
            public void run() {
                FileUtility.writeSerializable(file, serializable);
            }
        });
    }

    public FilePreferences addSharedPrefsKey(@NonNull String... keys) {
        sharedPrefValues.addAll(Arrays.asList(keys));
        return this;
    }

    private void migrateFromSharedPrefs() {
        for (Map.Entry<String, ?> entry : preferences.getAll().entrySet()) {
            Object value = entry.getValue();
            if (value instanceof Boolean) {
                put(entry.getKey(), (Boolean) value);
            } else if (value instanceof String) {
                put(entry.getKey(), (String) value);
            } else if (value instanceof Integer) {
                put(entry.getKey(), (Integer) value);
            } else if (value instanceof HashSet) {
                put(entry.getKey(), (HashSet) value);
            }
        }
        preferences.edit().clear().apply();
        apply();
    }

    public FilePreferences put(@NonNull String key, boolean value) {
        values.put(key, value);
        if (sharedPrefValues.contains(key)) {
            preferences.edit().putBoolean(key, value).apply();
        }
        return this;
    }

    public boolean getBoolean(String key, boolean defaultValue) {
        Object value = values.get(key);
        return (value instanceof Boolean) ? (boolean) value : defaultValue;
    }

    public FilePreferences put(@NonNull String key, @NonNull String value) {
        values.put(key, value);
        if (sharedPrefValues.contains(key)) {
            preferences.edit().putString(key, value).apply();
        }
        return this;
    }

    public String getString(String key, String defaultValue) {
        Object value = values.get(key);
        return (value instanceof String) ? (String) value : defaultValue;
    }

    public FilePreferences put(@NonNull String key, int value) {
        values.put(key, value);
        if (sharedPrefValues.contains(key)) {
            preferences.edit().putInt(key, value).apply();
        }
        return this;
    }

    public int getInt(String key, int defaultValue) {
        Object value = values.get(key);
        return (value instanceof Integer) ? (int) value : defaultValue;
    }

    public FilePreferences put(@NonNull String key, @NonNull HashSet<String> value) {
        values.put(key, CollectionsConcurrencyUtil.getNewHashSet(value));
        if (sharedPrefValues.contains(key)) {
            preferences.edit().putStringSet(key, CollectionsConcurrencyUtil.getNewHashSet(value)).apply();
        }
        return this;
    }

    public HashSet<String> getStringSet(String key, HashSet<String> defaultValue) {
        Object value = values.get(key);
        return (value instanceof HashSet) ? CollectionsConcurrencyUtil.getNewHashSet((HashSet<String>) value) : defaultValue;
    }
}
