package com.vungle.warren.persistence;

import androidx.annotation.NonNull;
import android.util.Log;

import com.vungle.warren.utility.FileUtility;

import java.io.File;
import java.io.IOException;

/**
 * The name of this class will likely change. This class implements the {@link Designer} contract,
 * managing creative assets on disk for storage and retrieval, as well as regular maintenance such
 * as expiring old assets.
 */
public class GraphicDesigner implements Designer, CacheManager.Listener {

    private static final String FOLDER_NAME = "vungle";
    private static final String TAG = GraphicDesigner.class.getSimpleName();

    private CacheManager cacheManager;
    /**
     * Initialize the GraphicDesigner. It requires a reference to the application context in order
     * to access the cache directory.
     * @param cacheManager cache manager.
     */
    public GraphicDesigner(@NonNull CacheManager cacheManager){
        this.cacheManager = cacheManager;
        this.cacheManager.addListener(this);
        FileUtility.printDirectoryTree(getCacheDirectory());
    }

    @Override
    public File getAssetDirectory(String identifier) throws IllegalStateException {
        File assetDir = new File(getCacheDirectory().getPath() + File.separator + identifier);
        if(!assetDir.exists()){
            assetDir.mkdir();
        }
        return assetDir;
    }

    @Override
    public void deleteAssets(String identifier) throws IOException, IllegalStateException {
        /// Check if we have a folder for that placement in the cache directory
        File[] files = getCacheDirectory().listFiles();
        if (files == null) {
            return;
        }

        for(File f : files){
            if(f.isDirectory() && f.getName().equals(identifier)){
                /// If the folder is not empty, then we have some assets for this placement!
                FileUtility.delete(f);
            }
        }
    }

    @Override
    public File getCacheDirectory() throws IllegalStateException {
        if(cacheManager == null) {
            throw new IllegalStateException("Context has expired, cannot continue.");
        }

        File dir = new File(cacheManager.getCache() + File.separator + FOLDER_NAME);

        if(!dir.exists()){
            dir.mkdir();
        }
        return dir;
    }

    @Override
    public void clearCache() {
        if (cacheManager == null || cacheManager.getCache() == null)
            return;

        File dir = new File(cacheManager.getCache().getPath() + File.separator + FOLDER_NAME);
        if (dir.exists()) {
            try {
                FileUtility.delete(dir);
            } catch (IOException e) {
                Log.e(TAG, "Failed to delete cached files. Reason: " + e.getLocalizedMessage());
            }
        }

        if (!dir.exists()){
            dir.mkdir();
        }
    }

    @Override
    public void onCacheChanged() {
        //clear old caches
        if (cacheManager == null)
            return;

        for (File root : cacheManager.getOldCaches()) {
            try {
                FileUtility.delete(new File(root.getPath() + File.separator + FOLDER_NAME));
            } catch (IOException e) {
                Log.e(TAG, "Failed to delete cached files. Reason: " + e.getLocalizedMessage());
            }
        }
    }
}
